/**
 * Copyright (C) 2008-09 Tan Miaoqing
 * Contact: Tan Miaoqing <rabbitrun84@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <hildon/hildon-defines.h>
#include <hildon-uri.h>
#include <rtcom-eventlogger/eventlogger.h>

#include "contact-history-view.h"
#include "urllabel.h"
#include "url-parser.h"
#include "common.h"

#define FIRST_ROUND_LIMIT 3
#define EACH_ROUND_LIMIT 30 /* each round limit */

#define HISTORY_STATUS_WIDTH_SIZE 700
#define ROW_SPACE 10
#define ICON_STATUS_SPACE 10

#define GET_PRIVATE(o) ((ContactHistoryViewPrivate *)       \
                        ((ContactHistoryView *)(o))->priv)

typedef struct _ContactHistoryViewPrivate ContactHistoryViewPrivate;

/* Signals */
enum {
  SIG_ROW_ACTIVATED,
  SIGNALS
};
static gulong signals[SIGNALS] = {0};

struct _ContactHistoryViewPrivate
{
  RTComEl *eventlogger;
  RTComElQuery *query;

  GList *cached_statuses;
  GThread *cache_thread;

  GArray *status_list; /* a list of HistoryStatus instances added to view so far */

  guint offset;
  guint32 pressed_time;
};

G_DEFINE_TYPE (ContactHistoryView, contact_history_view, GTK_TYPE_VBOX);

static gboolean populate_history (gpointer data);
static GList *add_history_to_list (RTComElIter *iter);

/**************************************/
/* Private functions                  */
/**************************************/

/**
 * Callback for urllabel
 * Click a link in the status will launch the browser
 */
static void
url_clicked_cb (GtkWidget *url_label,
                gchar *url,
                gpointer userdata)
{
  hildon_uri_open (url, NULL, NULL);
}

static gboolean
contact_history_released_cb (GtkWidget *widget,
                             GdkEventButton *event,
                             gpointer data)
{
  HistoryStatus *status = (HistoryStatus *)data;
  ContactHistoryViewPrivate *priv = GET_PRIVATE (status->view);

  /* TODO Lassi: connect to button-release-event, or possible both to pressed
     and released and on released, check how far away the release happened.
     The event contains the coordinates:event->x, event->y
     alternatively you can use the event time or both

     Lassi: actually, the first thing you should do is checking
     whether the release event happens with the allocation of the box.
     you have event->x and event->y and widget->allocation.{width,height,x,y} */

  if (event->type == GDK_BUTTON_RELEASE &&
      event->time - priv->pressed_time >= 300) {
    g_signal_emit (status->view, signals[SIG_ROW_ACTIVATED], 0, status);
  }

  gtk_widget_set_state (widget, GTK_STATE_NORMAL);

  return FALSE;
}

static gboolean
contact_history_pressed_cb (GtkWidget *widget,
                            GdkEventButton *event,
                            gpointer data)
{
  HistoryStatus *status = (HistoryStatus *)data;
  ContactHistoryViewPrivate *priv = GET_PRIVATE (status->view);

  priv->pressed_time = event->time;

  gtk_widget_set_state (widget, GTK_STATE_ACTIVE);

  return FALSE;
}

static void
add_history_status (ContactHistoryView *view,
                    HistoryStatus *status)
{
  ContactHistoryViewPrivate *priv = GET_PRIVATE (view);
  GtkWidget *event_box;
  GtkWidget *hbox;
  GtkWidget *icon;
  GtkWidget *label;
  gchar *result;
  gchar *parsed_text = NULL;

  g_return_if_fail (status != NULL);

  url_parser_process_message (status->status, &parsed_text);

  result = g_strdup_printf (
      "<span>%s</span>\n"
      "<span foreground=\"%s\" size=\"x-small\"><sup>%s</sup></span>",
      parsed_text,
      get_secondary_text_color (), status->timestamp);

  event_box = gtk_event_box_new ();
  hbox = gtk_hbox_new (FALSE, 0);
  icon = gtk_image_new_from_icon_name (
      "internet-group-chat", HILDON_ICON_SIZE_FINGER);
  label = url_label_new ();
  gtk_widget_set_size_request(label, HISTORY_STATUS_WIDTH_SIZE, -1);
  url_label_set_text (URL_LABEL (label), result);
  gtk_label_set_line_wrap (GTK_LABEL(label), TRUE);

  gtk_box_pack_start (GTK_BOX (hbox), icon, FALSE, FALSE, ICON_STATUS_SPACE);
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER (event_box), hbox);
  gtk_box_pack_start (GTK_BOX (view), event_box, FALSE, FALSE, ROW_SPACE);
  gtk_box_pack_start (GTK_BOX (view), gtk_hseparator_new (), FALSE, FALSE, 0);

  g_signal_connect (label, "url_clicked",
      G_CALLBACK (url_clicked_cb), NULL);

  g_signal_connect (event_box, "button-press-event",
      G_CALLBACK (contact_history_pressed_cb), status);
  g_signal_connect (event_box, "button-release-event",
      G_CALLBACK (contact_history_released_cb), status);

  g_array_append_val (priv->status_list, status);

  g_free (result);
  g_free (parsed_text);
}

/*
 * The history retrieval is done in two rounds:
 * 1st round quickly retrieves three statuses and shows them on UI;
 * 2nd round will start a new thread to retrieve rest of history from
 * database, and add these statuses to UI in an idle loop
 *
 * threaded_populate_history is for 2nd round
 */
static gpointer
threaded_populate_history (gpointer data)
{
  ContactHistoryView *view = (ContactHistoryView *)data;
  ContactHistoryViewPrivate *priv = GET_PRIVATE (view);
  RTComElIter *iter;

  g_return_val_if_fail (priv->query != NULL, NULL);

  rtcom_el_query_set_limit (priv->query, EACH_ROUND_LIMIT);
  rtcom_el_query_set_offset (priv->query, priv->offset);

  iter = rtcom_el_get_events (priv->eventlogger, priv->query);

  priv->cached_statuses = add_history_to_list (iter);
  g_object_unref (iter);

  if (priv->cached_statuses) {
    g_idle_add ((GSourceFunc) populate_history, view);
    priv->offset += EACH_ROUND_LIMIT;
  }
  return NULL;
}

static gboolean
populate_history (gpointer data)
{
  ContactHistoryView *view = (ContactHistoryView *)data;
  ContactHistoryViewPrivate *priv = GET_PRIVATE (view);
  GList *status_iter;

  priv->cached_statuses = g_list_first (priv->cached_statuses);
  for (status_iter = priv->cached_statuses;
       status_iter;
       status_iter = status_iter->next) {
     HistoryStatus *status;
     GValueArray *values;
     gint id;
     const gchar *text;
     gint timestamp;

     values = (GValueArray *) status_iter->data;

     id = g_value_get_int (g_value_array_get_nth (values, 0));
     text = g_value_get_string (g_value_array_get_nth (values, 1));
     timestamp = g_value_get_int (g_value_array_get_nth (values, 2));

     status = g_slice_new0 (HistoryStatus);
     status->event_id = id;
     status->status = g_strdup (text);
     status->timestamp = get_timestamp (timestamp);
     status->view = view;

     add_history_status (view, status);

     g_value_array_free (values);
  }

  g_list_free (priv->cached_statuses);
  priv->cached_statuses = NULL;

  return FALSE;
}

static GList *
add_history_to_list (RTComElIter *iter)
{
  GList *list = NULL;

  if (iter && rtcom_el_iter_first (iter)) {
    do {
      GValueArray *values;

      values = rtcom_el_iter_get_valuearray (
          iter,
          "id",
          "content",
          "start-time",
          NULL);

      list = g_list_prepend (list, values);
    } while (rtcom_el_iter_next (iter));

    list = g_list_reverse (list);
  }

  return list;
}

static RTComElQuery *
build_history_retrieval_query (RTComEl *eventlogger,
                               const gchar *local_uid,
                               const gchar *remote_uid)
{
  RTComElQuery *query;
  gboolean res;

  query = rtcom_el_query_new (eventlogger);

  if (local_uid) {
    res = rtcom_el_query_prepare (
                query,
                "service", "RTCOM_EL_SERVICE_STATUS", RTCOM_EL_OP_EQUAL,
                "local-uid", local_uid, RTCOM_EL_OP_EQUAL,
                "remote-uid", remote_uid, RTCOM_EL_OP_EQUAL,
                NULL);
  }
  else {
    res = rtcom_el_query_prepare (
                query,
                "service", "RTCOM_EL_SERVICE_STATUS", RTCOM_EL_OP_EQUAL,
                "remote-uid", remote_uid, RTCOM_EL_OP_EQUAL,
                NULL);
  }

  if (!res) {
    g_object_unref (query);
    return NULL;
  }

  return query;
}

/**************************************/
/* GObject functions                  */
/**************************************/

static void
contact_history_view_init (ContactHistoryView *view)
{
  ContactHistoryViewPrivate *priv;

  priv = view->priv = G_TYPE_INSTANCE_GET_PRIVATE(view, CONTACT_TYPE_HISTORY_VIEW,
                                                  ContactHistoryViewPrivate);

  priv->status_list = g_array_new (FALSE, FALSE, sizeof (HistoryStatus *));
}

static void
contact_history_view_dispose (GObject *obj)
{
  ContactHistoryViewPrivate *priv = GET_PRIVATE (obj);
  guint i, length = priv->status_list->len;

  g_object_unref (priv->query);

  for (i = 0; i < length; i++) {
    HistoryStatus *status;

    status = g_array_index (priv->status_list, HistoryStatus *, i);
    g_free (status->status);
    g_free (status->timestamp);
    g_slice_free (HistoryStatus, status);
  }
  g_array_free (priv->status_list, TRUE);

  G_OBJECT_CLASS (contact_history_view_parent_class)->dispose (obj);
}

static void
contact_history_view_class_init (ContactHistoryViewClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  object_class->dispose = contact_history_view_dispose;

  /**
   * ContactHistoryView::row-activated:
   * @contact_history_view: The object that emitted the signal
   * @status: HistoryStatus instance of this history status message
   *
   * Signals that a history row is pressed.
   */
  signals[SIG_ROW_ACTIVATED] = g_signal_new (
      "row-activated",
      G_TYPE_FROM_CLASS (klass),
      G_SIGNAL_RUN_LAST,
      0, NULL, NULL,
      g_cclosure_marshal_VOID__POINTER,
      G_TYPE_NONE, 1,
      G_TYPE_POINTER);

  g_type_class_add_private (klass, sizeof (ContactHistoryViewPrivate));
}

/**************************************/
/* Public functions                  */
/**************************************/

GtkWidget *contact_history_view_new (void)
{
  return g_object_new (CONTACT_TYPE_HISTORY_VIEW, NULL);
}

void
contact_history_view_set_eventlogger (ContactHistoryView *view,
                                      RTComEl *eventlogger)
{
  ContactHistoryViewPrivate *priv = GET_PRIVATE (view);

  g_return_if_fail (eventlogger != NULL);

  priv->eventlogger = eventlogger; /* increase ref count? */
}

void
contact_history_view_update_contact (ContactHistoryView *view,
                                     const gchar *local_uid,
                                     const gchar *remote_uid)
{
  ContactHistoryViewPrivate *priv = GET_PRIVATE (view);
  RTComElIter *iter;

  priv->query = build_history_retrieval_query (
      priv->eventlogger, local_uid, remote_uid);

  if (!priv->query) {
    /* TODO do cleanup???? */
    return ;
  }

  rtcom_el_query_set_limit (priv->query, FIRST_ROUND_LIMIT);
  rtcom_el_query_set_offset (priv->query, 0);

  iter = rtcom_el_get_events (priv->eventlogger, priv->query);

  priv->cached_statuses = add_history_to_list (iter);
  g_object_unref (iter);

  if (priv->cached_statuses) {
    populate_history (view);
    priv->offset = FIRST_ROUND_LIMIT;

    /* TODO
     * 1.need this priv->cache_thread????????????
     * 2.joinable is FALSE?????????? */
    priv->cache_thread = g_thread_create (
        (GThreadFunc) threaded_populate_history,
        view, FALSE, NULL);
  }
}
