/**
 * Copyright (C) 2008-09 Tan Miaoqing
 * Contact: Tan Miaoqing <rabbitrun84@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <glib.h>
#include <gmodule.h>
#include <gtk/gtk.h>
#include <sqlite3.h>
#include <string.h>

#include <rtcom-eventlogger/eventlogger-plugin.h>
#include <rtcom-eventlogger/eventlogger.h>

#define PLUGIN_NAME  "STATUS"
#define PLUGIN_DESC  "Status plugin"
#define SERVICE_NAME "RTCOM_EL_SERVICE_STATUS"
#define SERVICE_DESC "Service for logging of IM status messages."

/* This is the max length of the name displayed in the event, because there
 * need to be enough space for the date and time. */
#define NAME_LENGTH "25"

/* Max length of text, in characters. */
#define TEXT_LENGTH 100

const gchar * g_module_check_init(
        GModule * module)
{
    g_message("Plugin registered: %s.", PLUGIN_NAME);
    return NULL; /* NULL means success */
}

const gchar * rtcom_el_plugin_name(void)
{
    return PLUGIN_NAME;
}

const gchar * rtcom_el_plugin_desc(void)
{
    return PLUGIN_DESC;
}

RTComElService * rtcom_el_plugin_service(void)
{
    RTComElService * service = rtcom_el_service_new(
            SERVICE_NAME,
            SERVICE_DESC);
    return service;
}

GList * rtcom_el_plugin_eventtypes(void)
{
    struct
    {
        gchar * name;
        gchar * desc;
    } types[] = {
        {"RTCOM_EL_EVENTTYPE_STATUS_FRIEND", "Friend status"},
        {"RTCOM_EL_EVENTTYPE_STATUS_MY", "My status"},
        {NULL, NULL}
    };

    GList * l = NULL;
    guint i;

    for(i = 0; types[i].name != NULL; ++i)
    {
        l = g_list_prepend(l, rtcom_el_eventtype_new(
                    types[i].name, types[i].desc));
    }

    return g_list_reverse(l);
}

GList * rtcom_el_plugin_flags(void)
{
    return NULL;
}

static gint
count_status_messages (RTComElIter *it,
                       const gchar *local_uid,
                       const gchar *remote_uid)
{
    RTComEl *el;
    sqlite3 *db;
    sqlite3_stmt *stmt;
    char *sql;
    int ret;
    gint count = 0;

    if ((local_uid == NULL) ||
        (local_uid[0] == '\0') ||
        (remote_uid == NULL) ||
        (remote_uid[0] == '\0'))
        return 0;

    g_object_get(it, "el", &el, NULL);
    g_object_get(el, "db", &db, NULL);

    sql = sqlite3_mprintf ("SELECT COUNT(*) FROM Events "
            "JOIN EventTypes ON event_type_id = EventTypes.id "
            "JOIN Services ON service_id = Services.id "
            "WHERE Services.name = 'RTCOM_EL_SERVICE_STATUS' "
            "AND EventTypes.name = 'RTCOM_EL_EVENTTYPE_STATUS_FRIEND' "
            "AND local_uid = %Q AND remote_uid = %Q;", local_uid, remote_uid);

    if (SQLITE_OK != sqlite3_prepare_v2(db, sql, -1, &stmt, NULL))
    {
        g_error("%s: can't compile SQL: %s", G_STRFUNC, sql);
        sqlite3_free(sql);
        return 0;
    }

    while (SQLITE_BUSY == (ret = sqlite3_step(stmt)));

    if (ret == SQLITE_ROW)
    {
        count = sqlite3_column_int(stmt, 0);
    }
    else
    {
        g_error("%s: error while executing SQL", G_STRFUNC);
    }

    sqlite3_finalize(stmt);
    sqlite3_free(sql);

    return count;
}

gboolean rtcom_el_plugin_get_value(
        RTComElIter * it,
        const gchar * item,
        GValue * value)
{
    GValue event_type = {0};
    const gchar * type;
    gboolean retval;

    g_return_val_if_fail(it, FALSE);
    g_return_val_if_fail(item, FALSE);
    g_return_val_if_fail(value, FALSE);

    rtcom_el_iter_get_raw(it, "event-type", &event_type);
    type = g_value_get_string(&event_type);

    if(!type)
    {
        g_debug("Corrupted db.");
        retval = FALSE;
    }
    else if(!strcmp(item, "additional-text"))
    {
        /* We don't care about this value, let's set it empty to
         * avoid glib warnings. */
        g_value_init(value, G_TYPE_STRING);
        g_value_set_string(value, "");
        retval = TRUE;
    }
    else if(!strcmp(item, "icon-name"))
    {
        g_debug(G_STRLOC ": somebody requested the icon-path.");

        g_value_init(value, G_TYPE_STRING);
        g_value_set_string(value, "internet-group-chat");
        retval = TRUE;
    }
    else if(!strcmp(item, "vcard-field"))
    {
        gchar * vcard_field = rtcom_el_iter_get_header_raw(it, item);

        g_value_init(value, G_TYPE_STRING);
        if(vcard_field)
        {
            g_value_take_string(value, vcard_field);
        }
        else
        {
            g_debug("Plugin %s couldn't find item %s", PLUGIN_NAME, item);
            g_value_set_static_string(value, NULL);
        }
        retval = TRUE;
    }
    else if(!strcmp(item, "event-count"))
    {
        gint num = 0; /* event in the group */

        RTComElQuery * query;
        RTComElQueryGroupBy group_by;

        g_object_get(it, "query", &query, NULL);
        g_object_get(query, "group-by", &group_by, NULL);

        /* Only Friend status view has group mode */
        if (strcmp(type, "RTCOM_EL_EVENTTYPE_STATUS_FRIEND") == 0 &&
            group_by == RTCOM_EL_QUERY_GROUP_BY_CONTACT) {
            GValue local_uid = {0};
            GValue remote_uid = {0};

            rtcom_el_iter_get_raw(it, "local-uid", &local_uid);
            rtcom_el_iter_get_raw(it, "remote-uid", &remote_uid);
            num = count_status_messages (it,
                g_value_get_string(&local_uid),
                g_value_get_string(&remote_uid));

            g_value_unset(&local_uid);
            g_value_unset(&remote_uid);
        }

        g_value_init(value, G_TYPE_INT);
        g_value_set_int(value, num);
        retval = TRUE;
    }
    else if(!strcmp(item, "content"))
    {
        const gchar *txt;
        GValue free_text = {0};

        g_value_init(value, G_TYPE_STRING);

        rtcom_el_iter_get_raw (it, "free-text", &free_text);
        txt = g_value_get_string (&free_text);

        if (txt == NULL)
            txt = "";

        g_value_set_string(value, txt);
        g_value_unset(&free_text);
        retval = TRUE;
    }
    else if (!strcmp(item, "group-title"))
    {
        /* Status don't have group-title */
        g_value_init(value, G_TYPE_STRING);
        g_value_set_static_string(value, NULL);
        retval = TRUE;
    }
    else
    {
        retval = FALSE;
    }

    g_value_unset(&event_type);
    return retval;
}

