/**
 * Source file from libuiw, which is under LGPL v2.1 license
 * http://repository.maemo.org/catalogue/certified/pool/diablo/user/libu/libuiw/
 *
 * @file url_label.c
 */

#include <stdio.h>
#include <stdlib.h>
#include <glib.h>
#include <string.h>
#include <gtk/gtk.h>

#include "urllabel.h"

G_DEFINE_TYPE(UrlLabel, url_label, GTK_TYPE_LABEL)

/* signal enum */
enum
{
  URL_CLICKED,
  LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = {0};

/* private structure */
typedef struct _UrlLabelPrivate UrlLabelPrivate;

typedef struct {
  gchar *url;
  gint start;
  gint end;
} URLLink;

struct _UrlLabelPrivate
{
  GSList *urls;
  URLLink *last_url;
  GdkWindow *window;
};

#define URL_LABEL_GET_PRIVATE(obj) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((obj), TYPE_URL_LABEL, UrlLabelPrivate))

/* Static forward UrlLabel declarations */
static gint url_label_button_press_event(GtkWidget *widget, GdkEventButton *event);
static gint url_label_button_release_event(GtkWidget *widget, GdkEventButton *event);
static void url_label_finalize(GObject *object);
static void url_label_size_allocate(GtkWidget *widget, GtkAllocation *allocation);
static void url_label_realize(GtkWidget *widget);
static void url_label_unrealize(GtkWidget *widget);
static void url_label_map(GtkWidget *widget);
static void url_label_unmap(GtkWidget *widget);

/* Static forward URLLink declarations */
static URLLink *new_url_link(gchar *s, gint len, gint start, gint end);
static void free_url_link(URLLink *link);
static GSList *free_url_link_slist(GSList *list);

static void
url_label_init (UrlLabel *self)
{
  UrlLabelPrivate *priv = URL_LABEL_GET_PRIVATE(self);

  priv->urls = NULL;
}

static void
url_label_class_init (UrlLabelClass *url_label_class)
{
  GObjectClass *object_class = G_OBJECT_CLASS (url_label_class);
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS(url_label_class);

  g_type_class_add_private (url_label_class, sizeof (UrlLabelPrivate));

  object_class->finalize = url_label_finalize;

  widget_class->button_press_event = url_label_button_press_event;
  widget_class->button_release_event = url_label_button_release_event;
  widget_class->size_allocate = url_label_size_allocate;
  widget_class->realize = url_label_realize;
  widget_class->unrealize = url_label_unrealize;
  widget_class->map = url_label_map;
  widget_class->unmap = url_label_unmap;

  url_label_class->url_clicked = NULL;

  signals[URL_CLICKED] =
    g_signal_new ("url-clicked",
                  G_OBJECT_CLASS_TYPE (url_label_class),
                  G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                  0,
                  NULL, NULL,
                  gtk_marshal_VOID__STRING,
                  G_TYPE_NONE, 1, G_TYPE_STRING);
}

static void
url_label_finalize (GObject *object)
{
  UrlLabelPrivate *priv = URL_LABEL_GET_PRIVATE (object);

  free_url_link_slist(priv->urls);

  G_OBJECT_CLASS (url_label_parent_class)->finalize(object);
}

static void
url_label_size_allocate(GtkWidget *widget, GtkAllocation *allocation)
{
  UrlLabelPrivate *priv = URL_LABEL_GET_PRIVATE(widget);

  (* GTK_WIDGET_CLASS (url_label_parent_class)->size_allocate) (widget, allocation);

  if (priv && priv->window)
  {
    gdk_window_move_resize (priv->window,
                            allocation->x, allocation->y,
                            allocation->width, allocation->height);
  }
}

static void
url_label_realize(GtkWidget *widget)
{
  UrlLabelPrivate *priv = URL_LABEL_GET_PRIVATE(widget);

  (* GTK_WIDGET_CLASS (url_label_parent_class)->realize) (widget);

  if (priv && !priv->window)
  {
    GdkWindowAttr attributes;

    attributes.wclass = GDK_INPUT_ONLY;
    attributes.window_type = GDK_WINDOW_TEMP;
    attributes.override_redirect = TRUE;
    attributes.event_mask = GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK;
    attributes.x = widget->allocation.x;
    attributes.y = widget->allocation.y;
    attributes.width = widget->allocation.width;
    attributes.height = widget->allocation.height;

    priv->window = gdk_window_new(widget->window, &attributes,
                                  GDK_WA_X | GDK_WA_Y | GDK_WA_NOREDIR);
    gdk_window_set_user_data (priv->window, widget);
  }
}

static void
url_label_unrealize(GtkWidget *widget)
{
  UrlLabelPrivate *priv = URL_LABEL_GET_PRIVATE(widget);

  if (priv && priv->window)
  {
    gdk_window_set_user_data (priv->window, NULL);
    gdk_window_destroy(priv->window);
    priv->window = NULL;
  }

  (* GTK_WIDGET_CLASS (url_label_parent_class)->unrealize) (widget);
}

static void
url_label_map(GtkWidget *widget)
{
  UrlLabelPrivate *priv = URL_LABEL_GET_PRIVATE(widget);

  (* GTK_WIDGET_CLASS (url_label_parent_class)->map) (widget);

  if (priv && priv->window)
    gdk_window_show(priv->window);
}

static void
url_label_unmap(GtkWidget *widget)
{
  UrlLabelPrivate *priv = URL_LABEL_GET_PRIVATE(widget);

  if (priv && priv->window)
    gdk_window_hide(priv->window);

  (* GTK_WIDGET_CLASS (url_label_parent_class)->unmap) (widget);
}

static URLLink *
url_label_find_link_xy(GtkWidget *widget, int x, int y)
{
  UrlLabelPrivate *priv = URL_LABEL_GET_PRIVATE (URL_LABEL(widget));
  PangoLayout *layout = gtk_label_get_layout(GTK_LABEL(widget));
  gint ox, oy, idx;

  if (!priv)
    return NULL;

  gtk_label_get_layout_offsets(GTK_LABEL(widget), &ox, &oy);

  if (pango_layout_xy_to_index(layout,
                               (x - ox + widget->allocation.x)*PANGO_SCALE,
                               (y - oy + widget->allocation.y)*PANGO_SCALE,
                               &idx, NULL))
  {
    GSList *l;

    for (l = priv->urls; l && l->data && ((URLLink *)l->data)->start <= idx; l = l->next)
    {
      URLLink *link = (URLLink *)l->data;
      if (link->start <= idx && link->end >= idx)
        return link;
    }
  }

  return NULL;
}

static gint
url_label_button_press_event(GtkWidget *widget, GdkEventButton * event)
{
  UrlLabelPrivate *priv = URL_LABEL_GET_PRIVATE (URL_LABEL(widget));

  if (event->button == 1 && priv)
  {
    priv->last_url = url_label_find_link_xy(widget, event->x, event->y);

    return priv->last_url != NULL;
  }

  if (priv)
    priv->last_url = NULL;

  return FALSE;
}

static gint
url_label_button_release_event(GtkWidget *widget, GdkEventButton * event)
{
  UrlLabelPrivate *priv = URL_LABEL_GET_PRIVATE (URL_LABEL(widget));

  if (event->button == 1 && priv && priv->last_url &&
      priv->last_url == url_label_find_link_xy(widget, event->x, event->y))
  {
    g_signal_emit(widget, signals[URL_CLICKED], 0, priv->last_url->url);
    priv->last_url = NULL;

    return TRUE;
  }

  if (priv)
    priv->last_url = NULL;


  return FALSE;
}

/* quick and dirty url finder */
static gchar *
_find_and_replace_urls(gchar *text, GSList **urls)
{
	gchar *p = text, *q = text, *t, *a, *b;
	GString *res;

  res = g_string_new(NULL);
  if (*urls)
    *urls = free_url_link_slist(*urls);

	while ((p = strstr(p, "<a")))
	{
    g_string_append_len(res, q, p-q);

    if ((t = strchr(p, '>')))
    {
      a = strstr(p, "href");
      if (a && a < t)
      {
        a = strchr(a, '"');
        if (a && (b = strchr(a+1, '"')))
        {
          g_string_append(res, "<span underline=\"single\" color=\"green\">");
          *urls = g_slist_prepend(*urls, new_url_link(a+1, b-a-1, 0, 0));
        }
      }

      q = t+1;
      if ((p = strstr(q, "</a>")))
      {
        g_string_append_len(res, q, p-q);
        p = q = p+4;
        g_string_append(res, "</span>");
      }
    } else {
      q = NULL;
      break;
    }
	}

  if (q)
    g_string_append(res, q);

  *urls = g_slist_reverse(*urls);

  return g_string_free(res, FALSE);
}

static void
url_label_update_url_pos(UrlLabel *label)
{
  UrlLabelPrivate *priv = URL_LABEL_GET_PRIVATE(label);

  if (priv->urls)
  {
    PangoLayout *layout = gtk_label_get_layout(GTK_LABEL(label));
    PangoAttrList *list = pango_layout_get_attributes(layout);
    PangoAttrIterator *iter = pango_attr_list_get_iterator(list);
    PangoAttribute *attr;
    GSList *urls = priv->urls;

    while (urls)
    {
      if (pango_attr_iterator_get(iter, PANGO_ATTR_UNDERLINE) &&
          (attr = pango_attr_iterator_get(iter, PANGO_ATTR_FOREGROUND)))
      {
        URLLink *url = (URLLink *)urls->data;
        url->start = attr->start_index;
        url->end = attr->end_index;
        urls = urls->next;
      }

      if (!pango_attr_iterator_next(iter))
        break;
    }
    pango_attr_iterator_destroy (iter);
  }
}

/* URLLink implementation */

static URLLink *
new_url_link(gchar *s, gint len, gint start, gint end)
{
  URLLink *link = g_new(URLLink, 1);
  link->url = g_strndup(s, len);
  link->start = start;
  link->end = end;
  return link;
}

static void
free_url_link(URLLink *link)
{
  if (link)
  {
    g_free(link->url);
    g_free(link);
  }
}

static GSList *
free_url_link_slist(GSList *list)
{
  GSList *link;

  while (list)
  {
    free_url_link((URLLink *)list->data);
    link = list;
    list = list->next;
    g_slist_free1(link);
  }
  return list;
}

/* Public functions */

GtkWidget *url_label_new()
{
  return g_object_new(TYPE_URL_LABEL, NULL);
}

void url_label_set_text(UrlLabel *label, gchar *text)
{
  UrlLabelPrivate *priv = URL_LABEL_GET_PRIVATE(label);
  g_assert(priv);

  gchar *ntext = _find_and_replace_urls(text, &priv->urls);

  gtk_label_set_markup(GTK_LABEL(label), ntext);
  url_label_update_url_pos(label);

  g_free(ntext);
}

