/**
 * Copyright (C) 2008-09 Tan Miaoqing
 * Contact: Tan Miaoqing <rabbitrun84@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <hildon/hildon-edit-toolbar.h>
#include <hildon/hildon-note.h>
#include <hildon/hildon-defines.h>

#include <rtcom-eventlogger/eventlogger.h>
#include <rtcom-eventlogger-ui/rtcom-log-columns.h>

#include "delete-status-window.h"
#include "timeline-view.h"

#define GET_PRIVATE(o) ((DeleteStatusWindowPrivate *)       \
                        ((DeleteStatusWindow *)(o))->priv)

typedef struct _DeleteStatusWindowPrivate DeleteStatusWindowPrivate;
struct _DeleteStatusWindowPrivate
{
  GtkWidget *timeline_view;
  GtkWidget *toolbar;

  RTComEl *eventlogger;
};

G_DEFINE_TYPE (DeleteStatusWindow, delete_status_window, HILDON_TYPE_STACKABLE_WINDOW);

/**************************************/
/* Private functions                  */
/**************************************/

static void
confirmation_dialog_response_cb (GtkDialog *dialog,
                                 gint       response,
                                 gpointer   data)
{
  DeleteStatusWindowPrivate *priv = GET_PRIVATE (data);
  GtkWidget *window = data;

  gtk_widget_destroy (GTK_WIDGET (dialog));

  g_debug (G_STRLOC ": deleting selected statuses");

  if (response == GTK_RESPONSE_YES) {
    GtkTreeModel *model = NULL;
    GtkTreeSelection *selection;
    GList *items, *l;
    GError *log_error = NULL;

    selection = timeline_view_get_selection (TIMELINE_VIEW (priv->timeline_view));
    items = gtk_tree_selection_get_selected_rows (selection, &model);

    for (l = items; l != NULL; l = l->next) {
      GtkTreePath *path = (GtkTreePath *) l->data;
      GtkTreeIter iter;
      gint event_id = 0;

      gtk_tree_model_get_iter (model, &iter, path);
      gtk_tree_model_get (
        model,  &iter,
        RTCOM_LOG_VIEW_COL_EVENT_ID, &event_id,
        -1);

      rtcom_el_delete_event (priv->eventlogger, event_id, &log_error);

      if (log_error) {
          g_error ("%s: Unable to delete an event: %s",
                   G_STRFUNC, log_error->message);
          g_clear_error (&log_error);
      }
    }

    g_list_foreach (items, (GFunc) gtk_tree_path_free, NULL);
    g_list_free (items);

    gtk_widget_destroy (GTK_WIDGET (window));
  }
}

static void
toolbar_delete_button_clicked_cb (GtkButton  *button,
                                  GtkWindow  *window)
{
  GtkWidget *dialog = hildon_note_new_confirmation_add_buttons (
      window,
      "Delete selected statuses?",
      "Yes", GTK_RESPONSE_YES,
      "No",  GTK_RESPONSE_NO,
      NULL);

  g_signal_connect (dialog, "response",
                    G_CALLBACK (confirmation_dialog_response_cb),
                    window);

  gtk_window_set_transient_for (GTK_WINDOW (dialog), window);
  gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);

  gtk_widget_show (GTK_WIDGET (dialog));
}

static void
toolbar_arrow_button_clicked_cb (GtkButton  *button,
                                 GtkWindow  *window)
{
    gtk_widget_destroy (GTK_WIDGET (window));
}

/**************************************/
/* GObject functions                  */
/**************************************/

static void
delete_status_window_init (DeleteStatusWindow *window)
{
  DeleteStatusWindowPrivate *priv;
  GtkWidget *vbox;

  priv = window->priv = G_TYPE_INSTANCE_GET_PRIVATE (window, DELETE_TYPE_STATUS_WINDOW,
                                                     DeleteStatusWindowPrivate);

  vbox = gtk_vbox_new (FALSE, HILDON_MARGIN_DEFAULT);

  priv->toolbar = hildon_edit_toolbar_new_with_text (
      "Select statuses", "Delete");

  priv->timeline_view = g_object_ref_sink (timeline_view_new (FALSE));
  timeline_view_add_search_bar (TIMELINE_VIEW (priv->timeline_view),
                                GTK_WIDGET (window));
  timeline_view_set_mode_selection_multiple (TIMELINE_VIEW (priv->timeline_view));
  priv->eventlogger = timeline_view_get_eventlogger (
      TIMELINE_VIEW (priv->timeline_view));

  /* Packing */
  gtk_box_pack_start (GTK_BOX (vbox), priv->toolbar, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER (vbox), priv->timeline_view);
  gtk_container_add (GTK_CONTAINER(window), vbox);

  /* Signals */
  g_signal_connect (priv->toolbar, "button-clicked",
                    G_CALLBACK (toolbar_delete_button_clicked_cb), window);
  g_signal_connect (priv->toolbar, "arrow-clicked",
                    G_CALLBACK (toolbar_arrow_button_clicked_cb), window);

  gtk_window_fullscreen (GTK_WINDOW (window));
  gtk_widget_show_all (GTK_WIDGET (window));
  timeline_view_hide_search_bar (TIMELINE_VIEW (
                                 priv->timeline_view));
}

/* TODO dispose!!!!!!!!!!!!!!!!!!!!!!!!!!!! */
static void
delete_status_window_dispose (GObject *obj)
{
/*  DeleteStatusWindowPrivate *priv = GET_PRIVATE (obj);*/

  G_OBJECT_CLASS (delete_status_window_parent_class)->dispose (obj);
}

static void
delete_status_window_finalize (GObject *obj)
{

  G_OBJECT_CLASS (delete_status_window_parent_class)->finalize (obj);
}

static void
delete_status_window_class_init (DeleteStatusWindowClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  object_class->dispose = delete_status_window_dispose;
  object_class->finalize = delete_status_window_finalize;

  g_type_class_add_private (klass, sizeof (DeleteStatusWindowPrivate));
}

/**************************************/
/* Public functions                  */
/**************************************/

GtkWidget *
delete_status_window_new (OssoABookRoster *aggregator)
{
  GtkWidget *del_window = g_object_new (DELETE_TYPE_STATUS_WINDOW, NULL);
  DeleteStatusWindowPrivate *priv = GET_PRIVATE (del_window);

  timeline_view_set_abook_aggregator (TIMELINE_VIEW (
                                      priv->timeline_view), aggregator);
  return del_window;
}

void
delete_status_window_populate_friends_status (DeleteStatusWindow *del_window)
{
  DeleteStatusWindowPrivate *priv = GET_PRIVATE (del_window);
  timeline_view_populate_friends_status (TIMELINE_VIEW (
                                         priv->timeline_view));
}

void
delete_status_window_populate_my_status (DeleteStatusWindow *del_window)
{
  DeleteStatusWindowPrivate *priv = GET_PRIVATE (del_window);
  timeline_view_populate_my_status (TIMELINE_VIEW (
                                    priv->timeline_view));
}

void
delete_status_window_populate_contact_status (DeleteStatusWindow *del_window,
                                              const gchar *account,
                                              const gchar *contact)
{
  DeleteStatusWindowPrivate *priv = GET_PRIVATE (del_window);
  timeline_view_populate_contact_status (TIMELINE_VIEW (priv->timeline_view),
                                         account, contact);
}
