/**
 * Copyright (C) 2008-09 Tan Miaoqing
 * Contact: Tan Miaoqing <rabbitrun84@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <libosso-abook/osso-abook.h>
#include <rtcom-eventlogger/eventlogger.h>
#include <rtcom-eventlogger-ui/rtcom-log-view.h>
#include <rtcom-eventlogger-ui/rtcom-log-model.h>
#include <rtcom-eventlogger-ui/rtcom-log-columns.h>

#include "contact-status-view.h"
#include "contact-history-view.h"
#include "one-status-view.h"
#include "eventlogger-util.h"

#define CONTACT_ICON_SIZE_48 48

#define GET_PRIVATE(o) ((ContactStatusViewPrivate *)       \
                        ((ContactStatusView *)(o))->priv)

typedef struct _ContactStatusViewPrivate ContactStatusViewPrivate;
struct _ContactStatusViewPrivate
{
  /* selected status message with avatar and presence icon */
  GtkWidget *selected_status_view;
  GtkWidget *panarea;
  GtkWidget *frame; /* shows "Status history" title */
  GtkWidget *contact_history_view;
  GdkPixbuf *service_icon;

  RTComEl *eventlogger;
  OssoABookRoster *aggregator;

  GList *contacts; /* contacts->data is OssoABookContact object */
  gchar *account; /* local account name (local_uid) */
  gchar *im_field; /* contact IM field (remote_uid) */
  gchar *name; /* contact display name */

  /* 2nd subview window for showing one status history */
  GtkWidget *one_status_window;
  GtkWidget *one_status_view;
  GtkWidget *delete_button;
  GtkWidget *copy_button;
};

G_DEFINE_TYPE (ContactStatusView, contact_status_view, GTK_TYPE_ALIGNMENT);


/**************************************/
/* 2nd Sub view: one status window */
/**************************************/

static void
copy_button_clicked_cb (GtkButton *button,
                        gpointer   data)
{
  ContactStatusViewPrivate *priv = GET_PRIVATE (data);
  GtkClipboard* clipboard = gtk_clipboard_get (GDK_SELECTION_CLIPBOARD);

  gtk_clipboard_set_text (clipboard,
      one_status_view_get_status_text (
          ONE_STATUS_VIEW (priv->one_status_view)), -1);
  hildon_banner_show_information (
      GTK_WIDGET (priv->one_status_window), NULL, "Status copied");
}

static void
confirmation_dialog_response_cb (GtkDialog *dialog,
                                 gint       response,
                                 gpointer   data)
{
  ContactStatusViewPrivate *priv = GET_PRIVATE (data);
  gint event_id = one_status_view_get_event_id (
      ONE_STATUS_VIEW (priv->one_status_view));

  gtk_widget_destroy (GTK_WIDGET (dialog));

  if (response != GTK_RESPONSE_YES)
    return ;

  if (event_id > 0) {
    rtcom_el_delete_event (priv->eventlogger, event_id, NULL);
  }

  gtk_widget_hide (priv->one_status_window);
}

static void
delete_button_clicked_cb (GtkButton *button,
                          gpointer   data)
{
  ContactStatusViewPrivate *priv = GET_PRIVATE (data);

  GtkWidget *dialog = hildon_note_new_confirmation_add_buttons (
      GTK_WINDOW (priv->one_status_window),
      "Delete this status?",
      "Yes", GTK_RESPONSE_YES,
      "No",  GTK_RESPONSE_NO,
      NULL);

  g_signal_connect (dialog, "response",
                    G_CALLBACK (confirmation_dialog_response_cb),
                    data);

  gtk_window_set_transient_for (GTK_WINDOW (dialog),
      GTK_WINDOW (priv->one_status_window));
  gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);

  gtk_widget_show (GTK_WIDGET (dialog));
}

static gboolean
app_window_delete_event_cb (GtkWidget *widget,
                            GdkEvent  *event,
                            gpointer   data)
{
  /* TODO i need a cleanup function? */
  gtk_widget_hide (widget);
  return TRUE;
}

static GtkWidget *
create_one_status_window (ContactStatusView *view)
{
  ContactStatusViewPrivate *priv = GET_PRIVATE (view);
  GtkWidget *window = hildon_stackable_window_new ();
  GtkWidget *vbox = gtk_vbox_new (FALSE, 0);
  GtkWidget *hbox = gtk_hbox_new (TRUE, 0);
  GtkWidget *valign;
  GtkWidget *button_align;

  priv->one_status_view = g_object_ref_sink (one_status_view_new ());
  g_object_set (priv->one_status_view,
                "top-padding", HILDON_MARGIN_HALF,
                "left-padding", HILDON_MARGIN_DOUBLE,
                "right-padding", HILDON_MARGIN_DOUBLE,
                NULL);

  valign = gtk_alignment_new (0, 1, 0, 0); /* put child widget to the bottom */
  button_align = gtk_alignment_new (1, 0, 0, 0); /* put child widget to the right */
  priv->delete_button = hildon_gtk_button_new (HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label (GTK_BUTTON (priv->delete_button), "Delete");
  gtk_widget_set_size_request (priv->delete_button, 150, 70);

  priv->copy_button = hildon_gtk_button_new (HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label (GTK_BUTTON (priv->copy_button), "Copy");
  gtk_widget_set_size_request (priv->copy_button, 200, 70);

  gtk_box_pack_start (GTK_BOX (vbox), priv->one_status_view, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), priv->delete_button, TRUE, TRUE, HILDON_MARGIN_DOUBLE);
  gtk_box_pack_start (GTK_BOX (hbox), priv->copy_button, TRUE, TRUE, HILDON_MARGIN_DOUBLE);
  gtk_container_add (GTK_CONTAINER (button_align), hbox);
  gtk_container_add (GTK_CONTAINER (vbox), valign);
  gtk_box_pack_start (GTK_BOX (vbox), button_align, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER (window), vbox);

  g_signal_connect (GTK_WIDGET(window), "delete-event",
                    G_CALLBACK (app_window_delete_event_cb),
                    NULL);

  g_signal_connect (priv->delete_button, "clicked",
                    G_CALLBACK (delete_button_clicked_cb), view);

  g_signal_connect (priv->copy_button, "clicked",
                    G_CALLBACK (copy_button_clicked_cb), view);

  return window;
}

/**************************************/
/* Private functions                  */
/**************************************/

static void
contact_history_row_activated_cb (ContactHistoryView *history_view,
                                  gpointer status,
                                  gpointer data)
{
  ContactStatusViewPrivate *priv = GET_PRIVATE (data);
  HistoryStatus *history_status = (HistoryStatus *)status;

  OssoABookContact *contact = NULL;

  /* contact may not exist (deleted or account disabled) */
  if (priv->contacts)
    contact = priv->contacts->data;

  /* one_status_view will free g_strdup of status and timestamp */
  one_status_view_update_contact (ONE_STATUS_VIEW (priv->one_status_view),
                                  contact, g_strdup (history_status->status),
                                  g_strdup (history_status->timestamp),
                                  priv->service_icon);

  one_status_view_set_event_id (ONE_STATUS_VIEW (priv->one_status_view),
                                history_status->event_id);

  gtk_window_set_title (GTK_WINDOW (priv->one_status_window), priv->name);

  gtk_widget_show_all (priv->one_status_window);
  gtk_widget_grab_focus (priv->one_status_view);
}

/**************************************/
/* GObject functions                  */
/**************************************/

static void
contact_status_view_init (ContactStatusView *view)
{
  ContactStatusViewPrivate *priv;
  GtkWidget *alignment;
  GtkWidget *vbox;

  priv = view->priv = G_TYPE_INSTANCE_GET_PRIVATE(view, CONTACT_TYPE_STATUS_VIEW,
                                                  ContactStatusViewPrivate);

  alignment = gtk_alignment_new (0, 0, 1, 1);
  g_object_set (alignment,
                "top-padding", HILDON_MARGIN_HALF,
                "left-padding", HILDON_MARGIN_DOUBLE,
                "right-padding", HILDON_MARGIN_DOUBLE,
                NULL);

  priv->panarea = hildon_pannable_area_new ();
  g_object_set (priv->panarea,
                "mov-mode", HILDON_MOVEMENT_MODE_VERT,
                "hscrollbar-policy", GTK_POLICY_NEVER,
                NULL);

  vbox = gtk_vbox_new (FALSE, HILDON_MARGIN_DEFAULT);

  /* Selected status of the contact */
  priv->selected_status_view = g_object_ref_sink (one_status_view_new ());

  /* Status history of the contact */
  priv->frame = gtk_frame_new ("Status history");

  gtk_box_pack_start (GTK_BOX(vbox), priv->selected_status_view, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER (vbox), priv->frame);
  hildon_pannable_area_add_with_viewport (HILDON_PANNABLE_AREA (priv->panarea), vbox);

  /* Add all UI elements to ContactStatusView object */
  gtk_container_add (GTK_CONTAINER (alignment), priv->panarea);
  gtk_container_add (GTK_CONTAINER (view), alignment);

  /* One status window for single history status (now hidden) */
  priv->one_status_window = create_one_status_window (view);
}

/* TODO */
static void
contact_status_view_dispose (GObject *obj)
{

}

static void
contact_status_view_finalize (GObject *obj)
{
/*  ContactStatusViewPrivate *priv = GET_PRIVATE (obj); */

  /* widget destory, g_object_unref */

  G_OBJECT_CLASS (contact_status_view_parent_class)->finalize (obj);
}

static void
contact_status_view_class_init (ContactStatusViewClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  object_class->dispose = contact_status_view_dispose;
  object_class->finalize = contact_status_view_finalize;

  g_type_class_add_private (klass, sizeof (ContactStatusViewPrivate));
}

/**************************************/
/* Public functions                  */
/**************************************/

GtkWidget *
contact_status_view_new (void)
{
  return g_object_new (CONTACT_TYPE_STATUS_VIEW, NULL);
}

void
contact_status_view_set_abook_aggregator (ContactStatusView *view,
                                          OssoABookRoster *aggr)
{
  ContactStatusViewPrivate *priv = GET_PRIVATE (view);

  g_return_if_fail (priv->aggregator == NULL);

  priv->aggregator = aggr;
}

void
contact_status_view_set_eventlogger (ContactStatusView *view,
                                     RTComEl *eventlogger)
{
  ContactStatusViewPrivate *priv = GET_PRIVATE (view);

  priv->eventlogger = eventlogger;
}

/**
 * contact_status_view_update_contact:
 *
 * Contact status window will be shown when a status row is tabbed.
 * Before showing it, update the selected contact info, including
 * its uid and the selected status
 */
void
contact_status_view_update_contact (ContactStatusView *view,
                                    const gchar *local_account,
                                    gchar *remote_account,
                                    gchar *name,
                                    gchar *text,
                                    gchar *start_time,
                                    GdkPixbuf *service_icon)
{
  ContactStatusViewPrivate *priv = GET_PRIVATE (view);
  OssoABookContact *contact = NULL;
  gchar *uid;

  g_free (priv->account);
  priv->account = g_strdup (local_account);

  g_free (priv->im_field);
  priv->im_field = remote_account;

  g_free (priv->name);
  priv->name = name;

  /* Update presence icon, avatar and selected status of the contact */
  if (priv->contacts)
    g_list_free (priv->contacts);

  /* uid example: gabble/jabber/account0-someone@gmail.com */
  uid = g_strdup_printf ("%s-%s", local_account, remote_account);
  priv->contacts = osso_abook_aggregator_lookup (
      OSSO_ABOOK_AGGREGATOR (priv->aggregator), uid);
  if (priv->contacts)
    contact = priv->contacts->data;

  if (priv->service_icon)
    g_object_unref (priv->service_icon);
  priv->service_icon = service_icon;

  /* one_status_view will free text string and start_time */
  one_status_view_update_contact (ONE_STATUS_VIEW (priv->selected_status_view),
                                  contact, text, start_time, service_icon);

  /* History view should have been cleaned already
   * when last time going back to main view.
   * Here I just double check it */
  contact_status_view_clean_history (view);

  priv->contact_history_view = contact_history_view_new ();
  gtk_container_add (GTK_CONTAINER (priv->frame), priv->contact_history_view);
  contact_history_view_set_eventlogger (
      CONTACT_HISTORY_VIEW (priv->contact_history_view),
      priv->eventlogger);
  contact_history_view_update_contact (
      CONTACT_HISTORY_VIEW (priv->contact_history_view),
      local_account, remote_account);

  /* Connect to row-activated signal.
   * Show one_status_window in the callback */
  g_signal_connect (priv->contact_history_view, "row-activated",
      G_CALLBACK (contact_history_row_activated_cb), view);

  /* Jump to the top. Every time when showing contact-status-view,
   * we need show the select_status_view on top */
  hildon_pannable_area_jump_to_child (HILDON_PANNABLE_AREA (priv->panarea),
                                      priv->selected_status_view);

  g_free (uid);
}

void
contact_status_view_clean_history (ContactStatusView *view)
{
  ContactStatusViewPrivate *priv = GET_PRIVATE (view);

  if (priv->contact_history_view) {
    /* Disconnect the "row-activated" signal */
    g_signal_handlers_disconnect_by_func (priv->contact_history_view,
        contact_history_row_activated_cb, view);

    gtk_container_remove (GTK_CONTAINER (priv->frame),
        priv->contact_history_view);
    priv->contact_history_view = NULL;
  }
}

const gchar *
contact_status_view_get_contact_im_field (ContactStatusView *view)
{
  ContactStatusViewPrivate *priv = GET_PRIVATE (view);
  return priv->im_field;
}

const gchar *
contact_status_view_get_contact_account (ContactStatusView *view)
{
  ContactStatusViewPrivate *priv = GET_PRIVATE (view);
  return priv->account;
}
