#ifndef TDICTIONARY_H
#define TDICTIONARY_H

#include <QObject>
#include <QList>
#include <QString>
#include <QFileInfo>
#include <QFile>
#include <QPointer>
#include <QPair>
#include <QUuid>

#include "TCard.h"
#include "CSVData.h"


class TField
    {
    public:
    enum TFieldStyle
        {
        EDefaultFieldStyle,
        EExampleFieldStyle
        };
        
    TField()
        {
        id = QUuid::createUuid();
        }
        
    TField( QString aName, TFieldStyle aStyle, TDictionary* aDict ):
        name( aName ), style( aStyle ), iDict( aDict )
        {
        id = QUuid::createUuid();
        }
    
    QString ToXMLString( int aIx ) const;
    bool operator<( const TField& anotherField ) { return name < anotherField.name; }
    
    QUuid id;
    QString name;
    TFieldStyle style;
    
    private:
    TDictionary* iDict;
    };
    
/** \brief The dictionary, containing a list of cards.
 *
 * The cards are kept in #iCards.
 * The property #iIsContentModified shows if the dictionary contents are modified and not saved. Different modifications are possible:
 * \li A single card was modified. Means contents of a card were modified. Signal CardModified(int) is emitted.
 * \li The Content Modified state is changed. Means that iIsContentModified has changed. Signal ContentModifiedChanged(bool) is emitted.
 * \li Some cards were inserted or removed. Independently of whether the cards are valid, signal
 * CardsInserted(int, int) or CardsRemoved(int,int) is emitted.
 * \li The file path is modified. Means that the dictionary file was renamed, e.g. after "Save as" operation.
 * Signal FilePathChanged() is emitted.
 *
 * All card-modifying functions and signals call SetModified().
 */

class TDictionary: public QObject
{
Q_OBJECT

friend class TDictParser;
friend class TStudyFileParser;

public:

enum TFieldRole
    {
    EQuestionFieldRole,
    EAnswerFieldRole,
    ENoneFieldRole,
    KFieldRolesNum
    };

/// Data of unrepeated cards: date of last addition and number of used unrepeated cards in that day.
struct TUnrepCardsData
    {
    QDateTime lastAddition;
    int usedNum;
    };

// Methods
TDictionary( const QString aFilePath = "" );
~TDictionary();

void SetDefaultFields();
QFile::FileError Load(const QString aFilePath);
QFile::FileError Save(const QString aFilePath, bool aChangeFilePath = true );
QFile::FileError Save() { return Save( iFilePath ); }
QFile::FileError SaveContent( const QString aFilePath );
QFile::FileError SaveStudy();
QFile::FileError ImportCSV( const QString aFilePath, const TCSVImportData& aImportData );
QString ExportToCSVString( const TCSVExportData& aExportData ) const;

// Getters and setters

QString FilePath() const { return iFilePath; }
QString ShortName( bool aMarkModified = true ) const;
QString StudyFilePath( bool aCreateIfDoesntExist = false ) const;
QString ErrorString() const { return iErrorStr; }
bool IsContentModified() const { return iIsContentModified; }
bool IsStudyModified() const { return iIsStudyModified; }
bool IsImported() const { return iIsImported; }
bool IsEmpty() const { return iCards.isEmpty(); }
const QMap<int, TField> Fields() const { return iFields; }
TField Field( int aIx ) const { return iFields[aIx]; }
int FieldsNum() const { return iFields.size(); }
int QuestionField() const {return iQuestionField; }
QString QuestionFieldName() const {return iFields[iQuestionField].name; }
/// Uses special font for EExampleFieldStyle
QString FormattedQuestionFieldName() const;
QMap<int, int> AnswerFields() const {return iAnswerFields; }
QString AnswerFieldsNames() const;
    /// @return QPair: role and, if it is an answer field, its index in the answer fields array
QPair<TFieldRole, int> FieldRole( int aIx );
    /// @return -1, if the given UUID was not found
int FieldIdToIx( QUuid aId ) const;
int RepetitionRecordsNum() const;
TQstnAnsrPair CurQstnAnsrPair() const { return qMakePair( iQuestionField, iAnswerFields[0] ); }

/// Gets total number of unrepeated cards for the current question-answer pair
int UnrepCardsNum() const { return iUnrepCardsNum[ CurQstnAnsrPair() ]; } // The values must always exist!

/// Gets last addition time of unrepeated cards for the current question-answer pair
QDateTime UnrepCardsLastAddition() const;

/// Gets number of today's used unrepeated cards for the current question-answer pair
int UsedUnrepCardsNum() const { return iUnrepCardsData[ CurQstnAnsrPair() ].usedNum; } // Returns default-constructed value if doesn't exist

/// Gets active cards deadline for the current question-answer pair
QDateTime ActiveCardsDeadline() const;

/** Decrease the total number of unrepeated cards for the current question-answer pair.
  * Increases the number of today's used unrepeated cards */
void DecreaseUnrepCardsNumBy( int aDiff );

/// Sets last addition time of unrepeated cards for the current question-answer pair
void SetUnrepCardsLastAddition( const QDateTime& aValue ) { iUnrepCardsData[ CurQstnAnsrPair() ].lastAddition = aValue; }

/// Sets number of today's used unrepeated cards for the current question-answer pair
void SetUsedUnrepCardsNum( int aValue ) { iUnrepCardsData[ CurQstnAnsrPair() ].usedNum = aValue; }

/// Sets active cards deadline for the current question-answer pair
void SetActiveCardsDeadline( const QDateTime& aValue ) { iActiveCardsDeadline[ CurQstnAnsrPair() ] = aValue; }

// Card getters
const QPointer<TCard> Card(int aIndex) const;
const QPointer<TCard> operator[](int aIndex) const { return Card(aIndex); }
int CardsNum() const { return iCards.size(); }

const QPointer<TCard> ValidCard(int aIndex) const;
const QList< QPointer<TCard> > ValidCards() const { return iValidCards; }
int ValidCardsNum() const { return iValidCards.size(); }

const QPointer<TCard> FindCardByUuid(QUuid aId) const;
int FindCardIxByUuid(QUuid aId) const;

// Modify cards

/** Sets question of the card aIndex. Checks for limits.
 * Emits signal CardModified(int), if the new value is different.
 * \return true, if the question was changed.
 */
bool SetQuestion( int aIndex, QString aQuestion ) { return SetCardField( aIndex, 0, aQuestion ); }

/** Sets answer of the card aIndex. Checks for limits.
 * Emits signal CardModified(int), if the new value is different.
 * \return true, if the answer was changed.
 */
bool SetAnswer( int aIndex, QString aAnswer ) { return SetCardField( aIndex, 1, aAnswer ); }

/** Sets example of the card aIndex. Checks for limits.
 * Emits signal CardModified(int), if the new value is different.
 * \return true, if the example was changed.
 */
bool SetExample( int aIndex, QString aExample ) { return SetCardField( aIndex, 2, aExample ); }

/** Sets field @p aFieldId of card @p aCardIx to value @p aField. Checks for card index limits.
 * Emits signal CardModified(int), if the new value is different.
 * @return true, if the field value was really changed.
 */
bool SetCardField( int aCardIx, int aFieldId, QString aField );

/** Sets all field related data.
    @return true, if the field ids were correct and the data was set successfully.
*/
bool SetCardFields( QMap<int, TField> aFields, int aQstnIx, QMap<int, int> aAnsrIxs, QList<int> aDeletedIxs );

void SetStudyModified( bool aModified = true );

/** Add one card to the end of the list. By default, adds an empty card.
 * Emits CardsInserted(int,int) signal.
 */
void AddCard( TCard* aCard );

/** Insert one card. By default, inserts an empty card.
 * Emits CardsInserted(int,int) signal.
 */
void InsertCard(int aIndex, TCard* aCard );

/** Inserts cards from the input string list.
 * Emits CardsInserted(int,int) signal.
 */
void InsertCards(int aIndex, const QStringList& aStringList);

/** Inserts empty aNum cards at aIndex.
 * Emits CardsInserted(int,int) signal.
 */
void InsertCards(int aIndex, int aNum);

int InsertXMLCards(int aIndex, const QString& aString);

/** Remove one card at aIndex.
 * Emits CardsRemoved(int,int) signal.
 */
void RemoveCard(int aIndex);

/**
 * Removes aNum cards starting from aIndex.
 * Emits CardsRemoved(int,int) signal.
 */
void RemoveCards(int aIndex, int aNum);

// XML
static QString XMLIndent( int aDepth );
static QString XMLEscapedStr( const QString aString );

protected:
static const int KXMLIndentSize = 4; // Number of indenting spaces

signals:

/// A card at \p aIndex was modified.
void CardModified( int aIndex );

/// \p aNum cards were inserted starting at \p aIndex.
void CardsInserted( int aIndex, int aNum );

/// \p aNum cards were removed starting at \p aIndex.
void CardsRemoved( int aIndex, int aNum );

/// A valid card at dictionary index @p aIndex was added. Add it to your valid card list.
void ValidCardAdded( int aIndex );

/// Some valid cards were removed. Check your valid card lists for NULL.
void ValidCardsRemoved();

void UnrepCardsChanged();

/** The Content Modified state was changed. Means that #iIsContentModified was changed.
 * \param aModified The new value of the Content Modified state.
 */
void ContentModifiedChanged( bool aModified );
void StudyModifiedChanged( bool aModified );

void FilePathChanged();

void FieldsChanged();

public slots:
/** Sets the Content Modified state (#iIsContentModified).
 * By default, sets the Content Modified state to true. 
 * If the new value is different from the old one, emits ContentModifiedChanged(bool).
 * \param aModified The contents are modified. The default is true.
 */
void SetContentModified( bool aModified = true );

void SwapQuestionAnswer();

private slots:
void AddValidCards( int aIndex, int aNum );

private:
void UpdateValidCards();
void SetUnrepCardsNum( int aValue );
QString UnrepDataToXMLStudy( int aDepth );

public:

static QString KNoName;
static QString KDictFileExtension;
static QString KStudyFileExtension;

private:

QString iFilePath;
QUuid iId;

/// Card fields: names and styles.
QMap<int, TField> iFields;

/// Index of the question field
int iQuestionField;

/// Indices of answer fields. The order is important.
QMap<int, int> iAnswerFields;

/// List of all cards. Own.
QList< TCard* > iCards;

/// List of valid cards. Not own.
QList< QPointer<TCard> > iValidCards;   /// @todo Implement filtering of valid cards as a proxy model: QSortFilterProxyModel

/// The dictionary contents are modified and not yet saved.
bool iIsContentModified;

/// Study data modified
bool iIsStudyModified;

/** The dictionary has just been imported from a CSV file. It has the name of the original file, but it must be changed. */
bool iIsImported;

QString iErrorStr;

/// Total number of unrepeated cards. For every current question-answer pair.
QMap<TQstnAnsrPair, int> iUnrepCardsNum;    /// @todo Implement filtering of unrep cards as a proxy model over the valid cards
/// Unrepeated cards data. For every question-answer pair.
QMap<TQstnAnsrPair, TUnrepCardsData> iUnrepCardsData;
/// Deadline for active cards. Initially it's the current date/time.
QMap<TQstnAnsrPair, QDateTime> iActiveCardsDeadline;
};

#endif
