#include "TDictParser.h"
#include "TCard.h"
#include "version.h"

#include <QtDebug>
#include <QUuid>
#include <QMessageBox>

TDictParser::TDictParser( TDictionary* aDict )
     :iDict( aDict ), iCurCard( NULL ), iDictContentModified( false ),
     iCurFieldIx( 0 ), iInFields( false )
{
iCurText.clear();
iErrorStr.clear();
iRootExists = iDict->CardsNum() > 0;
srand( time(NULL) );
}

bool TDictParser::startElement(const QString& /* namespaceURI */, const QString& /* localName */,
                                const QString& qName, const QXmlAttributes& attributes)
{
if( !iRootExists && qName != "dict")
    {
    iErrorStr = QObject::tr("The parsed file is not a dictionary");
    return false;
    }
if( !iDict )
    {
    iErrorStr = "No dictionary";
    return false;
    }

if( qName == "c" || qName == "card" )
    {
    iCurCard = new TCard( iDict );
    QString uuidStr = attributes.value("id");
    if( !uuidStr.isEmpty() )
        {
        QUuid id = QUuid( uuidStr );
        if( !id.isNull() )
            iCurCard->SetId( id );
        }
    QString commentStr = attributes.value("comment");
    if( commentStr == "yes" )
        iCurCard->SetCommented();
    iCurFieldIx = 0;
    }
else if( qName == "dict" )
    {
    // Version
    QString versionStr = attributes.value("version");
    if( versionStr < DIC_VERSION )
        {
        QMessageBox::warning( NULL, "Old dictionary",
            QObject::tr("The dictionary uses obsolete format %1.\n"
                        "It will be converted to the current format "DIC_VERSION".")
                        .arg( versionStr ) );
        iDict->SetDefaultFields();
        iDictContentModified = true;
        }
        
    // UUID
    QString uuidStr = attributes.value("id");
    if( !uuidStr.isEmpty() )
        {
        QUuid id = QUuid( uuidStr );
        if( !id.isNull() )
            iDict->iId = id;
        }
    iRootExists = true;
    }
else if( qName == "fields" )
    iInFields = true;
else if( qName == "field" )
    {
    if( iInFields )
        {   // <fields><field>
        QString uuidStr = attributes.value("id");
        if( !uuidStr.isEmpty() )
            {
            QUuid id = QUuid( uuidStr );
            if( !id.isNull() )
                iFieldId = id;
            }
        
        QString qstStr = attributes.value("question");
        if( qstStr == "yes" )
            iDict->iQuestionField = iCurFieldIx;
            
        QString ansStr = attributes.value("answer");
        if( !ansStr.isEmpty() )
            {
            int ansIx = ansStr.toInt();
            if( ansIx >= 0 )
                iDict->iAnswerFields[ansIx] = iCurFieldIx;
            }
            
        QString styleStr = attributes.value("style");
        if( styleStr == "example" )
            iFieldStyle = TField::EExampleFieldStyle;
        else
            iFieldStyle = TField::EDefaultFieldStyle;
        }
    }
else if (qName == "f")
    {   // <cards><c><f>
    QString uuidStr = attributes.value("id");
    if( !uuidStr.isEmpty() )
        iFieldId = QUuid( uuidStr );
    if( uuidStr.isEmpty() || iFieldId.isNull() )
        iFieldId = iDict->Field( iCurFieldIx ).id;
    iCurFieldIx++;
    }
    
iCurText.clear();
return true;
}

bool TDictParser::endElement(const QString & /* namespaceURI */, const QString & /* localName */,
                            const QString &qName)
{
if ( !iDict )
    {
    iErrorStr = "No dictionary";
    return false;
    }
    
iCurText = iCurText.trimmed();
if( qName == "fields" )
    iInFields = false;
else if( qName == "field" )
    {
    if( iInFields )
        {
        iDict->iFields[ iCurFieldIx ] = TField( iCurText, iFieldStyle, iDict );
        iDict->iFields[ iCurFieldIx ].id = iFieldId;
        iCurFieldIx++;
        }
    }
else if( qName == "f" )
    {
    if( iCurCard )
        {
        int fieldIx = iDict->FieldIdToIx( iFieldId );
        iCurCard->SetField( fieldIx, iCurText );
        }
    }
else if( qName == "c"  || qName == "card" )
    iDict->iCards << iCurCard;
else if( iCurCard )
    {
    if( qName == "qst" || qName =="ans" || qName =="xmp" )
        {
        QString text = iCurText;
        // Replace obsolete comment character (#) to a commented card
        if( text.startsWith("#") )
            {
            iCurCard->SetCommented();
            text = text.remove( 0, 1 ).trimmed(); // remove comment character
            }
        int fieldIx = 0;
        if (qName == "qst")
            fieldIx = 0;
        else if (qName == "ans")
            fieldIx = 1;
        else if (qName == "xmp")
            fieldIx = 2;
        iDictContentModified |= iCurCard->SetField( fieldIx, text );
        }
    }
return true;
}

bool TDictParser::characters(const QString &str)
{
iCurText += str;
return true;
}

bool TDictParser::fatalError(const QXmlParseException &exception)
{
iErrorStr = QObject::tr("XML parse error at line %1, column %2:\n"
    "%3.").arg(exception.lineNumber()).arg(exception.columnNumber())
    .arg(exception.message());
return false;
}
