#include "TMainWindow.h"

#include <QApplication>
#include <QAction>
#include <QMenu>
#include <QMenuBar>
#include <QToolBar>
#include <QStatusBar>
#include <QPushButton>
#include <QToolButton>
#include <QTabWidget>
#include <QSignalMapper>
#include <QClipboard>
#include <QCloseEvent>
#include <QFileDialog>
#include <QMessageBox>
#include <QSettings>
#include <QVBoxLayout>

#include "TAppModel.h"
#include "TDictTableModel.h"
#include "TFindDialog.h"
#include "TCSVImportDialog.h"
#include "TCSVExportDialog.h"
#include "version.h"

// ============== TMainWindow::TDictTab ==================================================

TMainWindow::TDictTab::TDictTab(TDictTableModel* aDictTableModel,  QWidget* aParent ):
    QWidget( aParent )
{
iCardSidesLabel = new QLabel;
iQstnLabel = new QLabel;
iAnsrLabel = new QLabel;
iCardSidesLabel->setMargin( 5 );
iQstnLabel->setFrameStyle( QFrame::Box | QFrame::Sunken );
iAnsrLabel->setFrameStyle( QFrame::Box | QFrame::Sunken );

QHBoxLayout* cardSidesLt = new QHBoxLayout;
cardSidesLt->addWidget( iCardSidesLabel );
cardSidesLt->addWidget( iQstnLabel );
cardSidesLt->addWidget( iAnsrLabel );
cardSidesLt->addStretch();

iDictTable = new TDictTableView( aDictTableModel );
iDictTable->ResizeColumns();

QVBoxLayout* tabLayout = new QVBoxLayout( this );
tabLayout->setContentsMargins( 0, 0, 0, 0 );
tabLayout->addLayout( cardSidesLt );
tabLayout->addWidget( iDictTable );
}

void TMainWindow::TDictTab::SetCardSidesText( QString aQstn, QString aAnsr )
{
iCardSidesLabel->setText( tr("Card sides:") );
iQstnLabel->setText( aQstn );
iAnsrLabel->setText( aAnsr );
}

// ============== TMainWindow ============================================================

TMainWindow::TMainWindow(TAppModel* aModel):
    iAppTitle( tr("Fresh Memory") ),
    iModel( aModel )
{
QCoreApplication::setOrganizationName("freshmemory");
QCoreApplication::setApplicationName("freshmemory");
QSettings::setDefaultFormat( QSettings::IniFormat );
iErrorTitle = tr("Error - ") + iAppTitle;

#ifdef Q_OS_LINUX
char *home = getenv("HOME");
if ( home )
    iWorkPath = home;
else
#endif
    iWorkPath = tr("./");

setWindowTitle( iAppTitle );
setWindowIcon( QIcon(":/images/mainicon.png") );

iSMStartTestTriggered = new QSignalMapper( this );
connect( iSMStartTestTriggered, SIGNAL(mapped(int)), this, SLOT(StartTest(int)) );
connect( QApplication::clipboard(), SIGNAL(dataChanged()), this, SLOT(UpdatePasteAction()) );

CreateActions();
CreateMenus();
CreateToolBars();
CreateStatusBar();
CreateCentralWidget();

ReadSettings();
UpdateRecentFileActs();
}

TMainWindow::~TMainWindow()
{
}

// MVC pattern
void TMainWindow::UpdateDictTab()
{
int curIndex = iDictTabs->currentIndex();
const TDictionary* curDict = iModel->CurDictionary();

if( !curDict->IsContentModified() )
    iDictTabs->setTabIcon( curIndex, QIcon(":/images/openbook-24.png") );
else
    iDictTabs->setTabIcon( curIndex, QIcon(":/images/filesave.png") );
iDictTabs->setTabText( curIndex, curDict->ShortName() );
CurDictTab()->SetCardSidesText( curDict->FormattedQuestionFieldName(), curDict->AnswerFieldsNames() );
}

void TMainWindow::UpdateCardsNumLabels()
{
const TDictionary* curDict = iModel->CurDictionary();
if( curDict)
    {
    iTotalCardsLabel->show();
    iValidCardsLabel->show();
    iUnrepCardsLabel->show();
    iTotalCardsLabel->setText( tr("Total: %1").arg(curDict->CardsNum()) );
    iValidCardsLabel->setText( tr("Valid: %1").arg(curDict->ValidCardsNum()) );
    iUnrepCardsLabel->setText( tr("Unrepeated: %1").arg(curDict->UnrepCardsNum()) );
    }
else
    {
    iTotalCardsLabel->hide();
    iValidCardsLabel->hide();
    iUnrepCardsLabel->hide();
    }
}

void TMainWindow::UpdateActions()
{
const TDictionary* curDict = iModel->CurDictionary();
bool isCurDictModified = curDict && curDict->IsContentModified();

iSaveAct->setEnabled( isCurDictModified );
iSaveAsAct->setEnabled( curDict );
iSaveCopyAct->setEnabled( curDict );
iSaveStudyAct->setEnabled( curDict && curDict->IsStudyModified() );
iExportAct->setEnabled( curDict );
iRemoveTabAct->setEnabled( curDict );
UpdatePasteAction();
iInsertCardsAct->setEnabled( curDict );
iInsertCardsAfterAct->setEnabled( curDict );
iFindAct->setEnabled( curDict );
iFindAgainAct->setEnabled( TFindDialog::CanFindAgain( CurTableView() ) );
iWordDrillAct->setEnabled( curDict );
iSpacedRepetitionAct->setEnabled( curDict );
iFieldManagerAct->setEnabled( curDict );
iSwapQstnAnsrAct->setEnabled( curDict );
}

void TMainWindow::UpdateSelectionActions()
{
bool selectionNotEmpty = iModel->CurDictionary() &&
    CurTableView()->selectionModel()->hasSelection();

iCutAct->setEnabled( selectionNotEmpty );
iCopyAct->setEnabled( selectionNotEmpty );
iCommentAct->setEnabled( selectionNotEmpty );
iRemoveCardsAct->setEnabled( selectionNotEmpty );
}

void TMainWindow::UpdatePasteAction()
{
bool isCurTabValid = iDictTabs->currentIndex() >= 0;
bool hasClipboardText = !QApplication::clipboard()->text().isEmpty();
iPasteAct->setEnabled( isCurTabValid && hasClipboardText );
}

// QMainWindow
void TMainWindow::closeEvent(QCloseEvent *event)
{
if( ProposeToSave() )
    {
    WriteSettings();
    event->accept();
    qApp->quit();
    }
else
    event->ignore();
}

void TMainWindow::NewFile()
{
TDictionary* dict = iModel->NewDictionary();
AddDictTab( dict );
}

void TMainWindow::Load()
{
QString filePath = QFileDialog::getOpenFileName( this, tr("Load dictionary"), iWorkPath,
                                                 tr("Dictionaries (*.fmd);;All files (*)") );
if( filePath.isEmpty() )
    return;
iWorkPath = QFileInfo( filePath ).path();
LoadFile( filePath );
}

void TMainWindow::LoadFile(const QString& aFilePath)
{
QFile::FileError error = iModel->OpenDictionary(aFilePath);
switch( error )
    {
    case QFile::NoError:
        break;
    case QFile::ResourceError:
        QMessageBox::warning( this, iErrorTitle, iModel->ErrorString() );
        return;
    default:
        QMessageBox::warning( this, iErrorTitle, tr("Cannot open file \"%1\".").
            arg(ShortFileName(aFilePath)) );
        return;
    }
QApplication::setOverrideCursor(Qt::WaitCursor);
TDictionary* dict = iModel->CurDictionary();
AddDictTab( dict );
QString studyRecordsStr;
int studyRecordsNum = dict->RepetitionRecordsNum();
if( studyRecordsNum > 0 )
    studyRecordsStr = tr("Loaded %n study record(s)", "", studyRecordsNum );
else
    studyRecordsStr = tr("No study records found");
statusBar()->showMessage( studyRecordsStr, 15000 );
AddToRecentFiles( aFilePath );
QApplication::restoreOverrideCursor();
}

void TMainWindow::AddDictTab(TDictionary* aDict)
{
connect( aDict, SIGNAL(ContentModifiedChanged(bool)), this, SLOT(UpdateDictTab()) );
connect( aDict, SIGNAL(ContentModifiedChanged(bool)), this, SLOT(UpdateActions()) );
connect( aDict, SIGNAL(StudyModifiedChanged(bool)), this, SLOT(UpdateDictTab()) );
connect( aDict, SIGNAL(StudyModifiedChanged(bool)), this, SLOT(UpdateActions()) );
connect( aDict, SIGNAL(FilePathChanged()), this, SLOT(UpdateDictTab()) );
connect( aDict, SIGNAL(FieldsChanged()), this, SLOT(UpdateDictTab()) );

connect( aDict, SIGNAL(CardsInserted(int,int)), this, SLOT(UpdateCardsNumLabels()) );
connect( aDict, SIGNAL(CardsRemoved(int,int)), this, SLOT(UpdateCardsNumLabels()) );
connect( aDict, SIGNAL(ValidCardAdded(int)), this, SLOT(UpdateCardsNumLabels()) );
connect( aDict, SIGNAL(ValidCardsRemoved()), this, SLOT(UpdateCardsNumLabels()) );
connect( aDict, SIGNAL(UnrepCardsChanged()), this, SLOT(UpdateCardsNumLabels()) );
connect( aDict, SIGNAL(FieldsChanged()), this, SLOT(UpdateCardsNumLabels()) );

TDictTableModel* tableModel = new TDictTableModel( aDict );
TDictTab* dictTab = new TDictTab( tableModel );
iDictTabs->addTab( dictTab, "" );
iDictTabs->setCurrentWidget( dictTab );
connect( CurTableView()->selectionModel(), SIGNAL(selectionChanged(const QItemSelection&, const QItemSelection&)),
    this, SLOT(UpdateSelectionActions()) );
    
UpdateDictTab();
UpdateCardsNumLabels();
UpdateSelectionActions();
}

QString TMainWindow::ShortFileName(const QString& aFilePath)
{
return QFileInfo( aFilePath ).fileName();
}

/// Fork to SaveAs() or really save in DoSave()
bool TMainWindow::Save()
{
TDictionary* dict = iModel->CurDictionary();
if( !dict )
    return false;
QString filePath = dict->FilePath();
if( filePath.isEmpty() || dict->IsImported() )
    return SaveAs();
else
    return DoSave( filePath );
}

bool TMainWindow::SaveAs( bool aChangeFilePath )
{
TDictionary* dict = iModel->CurDictionary();
if( !dict )
    return false;
QString filePath = dict->FilePath();
if( filePath.isEmpty() )
    filePath = iWorkPath + "/" + TDictionary::KNoName;
filePath = QFileDialog::getSaveFileName(this, tr("Save dictionary as ..."), filePath,
                                        tr("Dictionaries (*.fmd);;All files (*)"));
if( filePath.isEmpty() )
    return false;
iWorkPath = QFileInfo( filePath ).path();
if( DoSave( filePath, aChangeFilePath ) )
    {
    AddToRecentFiles( filePath );
    return true;
    }
else
    return false;
}

void TMainWindow::SaveCopy()
{
SaveAs( false );    // Do not change the file name
}

void TMainWindow::Import()
{
QString filePath = QFileDialog::getOpenFileName(this, tr("Import CSV file"), iWorkPath,
                                                tr("CSV files (*.csv *.txt);;All files (*)"));
if( filePath.isEmpty() )
    return;
iWorkPath = QFileInfo( filePath ).path();
TCSVImportDialog importDialog( this, filePath );
if( importDialog.exec() )
    {
    iModel->AddDictionary( importDialog.Dictionary() );
    AddDictTab( importDialog.Dictionary() );
    }
}

void TMainWindow::Export()
{
TDictionary* dict = iModel->CurDictionary();
if( !dict )
    return;
TCSVExportDialog exportDialog( this, dict );
if( exportDialog.exec() )
    {
    QString dictFilePath( dict->FilePath() );
    if( dictFilePath.isEmpty() )
        dictFilePath = iWorkPath + "/" + TDictionary::KNoName;
    QString filePath = QFileDialog::getSaveFileName( this, tr("Export to CSV file"), dictFilePath + ".txt",
                                                     tr("CSV files (*.csv *.txt);;All files (*)") );
    if( filePath.isEmpty() )
        return;
    iWorkPath = QFileInfo( filePath ).path();
    exportDialog.SaveCSVToFile( filePath );
    }
}

void TMainWindow::LoadRecentFile()
{
QAction *action = qobject_cast<QAction*>( sender() );
if( action )
    LoadFile( action->data().toString() );
}

bool TMainWindow::DoSave( const QString& aFilePath, bool aChangeFilePath )
{
QFile::FileError error = iModel->CurDictionary()->Save( aFilePath, aChangeFilePath );
if( error != QFile::NoError )
    {
    QMessageBox::warning( this, iErrorTitle, tr("Cannot save dictionary %1.").
        arg(ShortFileName(aFilePath)) );
    return false;
    }
UpdateDictTab();
UpdateActions();
return true;
}

bool TMainWindow::SaveStudy()
{
TDictionary* dict = iModel->CurDictionary();
if( !dict )
    return false;
QFile::FileError error = dict->SaveStudy();
if( error != QFile::NoError )
    {
    QMessageBox::warning( this, iErrorTitle, tr("Cannot save study file %1.").
        arg(ShortFileName( iModel->CurDictionary()->StudyFilePath() )) );
    return false;
    }
UpdateDictTab();
UpdateActions();
return true;
}

/**
 * Checks if all files are saved. If not, proposes to the user to save them.
 * This function is called before closing the application.
 * @return true, if all files are now saved and the application can be closed
 */
bool TMainWindow::ProposeToSave()
{
for(int i=0; i<iDictTabs->count(); i++)
    if( !ProposeToSave(i) )
        return false;
return true;
}

/**
 * Checks if the specified file is modified and if it is, proposes to the user to save it.
 * If the specified file is not modified, the function just returns true.
 * 
 * Function silently saves the study data, if it is modified.
 *
 * @param aTabIndex Index of the tab containing the modified file
 * @return true, if the file is saved and the application can be closed.
 *      false, if the user declined the proposition to save the file.
 */
bool TMainWindow::ProposeToSave(int aTabIndex)
{
const TDictionary* dict = iModel->Dictionary( aTabIndex );

if( dict->IsContentModified() )
    {
    iDictTabs->setCurrentIndex( aTabIndex );
    QMessageBox::StandardButton pressedButton;
    pressedButton = QMessageBox::warning( this, tr("Save dictionary?"), tr("Dictionary %1 was modified.\n"
        "Save changes?").arg( dict->ShortName(false) ),
        QMessageBox::Yes | QMessageBox::No | QMessageBox::Cancel );
    if( pressedButton == QMessageBox::Yes )
        {
        bool saved = Save();
        if( !saved )
            return false;
        }
    else if( pressedButton == QMessageBox::Cancel )
        return false;
    }
if( dict->IsStudyModified() )
    SaveStudy();
return true;
}

void TMainWindow::AddToRecentFiles( const QString& aFilePath)
{
iRecentFiles.removeAll( aFilePath );
iRecentFiles.prepend( aFilePath );
while( iRecentFiles.size() > KMaxRecentFiles )
    iRecentFiles.removeLast();
UpdateRecentFileActs();
}

void TMainWindow::UpdateRecentFileActs()
{
while( iRecentFiles.size() > KMaxRecentFiles )
    iRecentFiles.removeLast();
int recentFilesNum = iRecentFiles.size();
for (int i = 0; i < recentFilesNum; i++)
    {
    QString text = tr("&%1 %2").arg(i + 1).arg( ShortFileName( iRecentFiles[i] ) );
    iRecentFileActs[i]->setText( text );
    iRecentFileActs[i]->setData( iRecentFiles[i] );
    iRecentFileActs[i]->setStatusTip( iRecentFiles[i] );
    iRecentFileActs[i]->setVisible(true);
    }
for (int j = recentFilesNum; j < KMaxRecentFiles; j++)
    iRecentFileActs[j]->setVisible(false);
iNoRecentFilesAct->setVisible( recentFilesNum == 0 );
}

void TMainWindow::About()
{
QMessageBox::about( this, tr("About Fresh Memory"),
    tr("<p><h2>Fresh Memory "FM_VERSION"</h2></p>"
    "<p>A utility for learning foreign words</p>"
    "<p>Author: Mykhaylo Kopytonenko<br/>"
    "Lincense: GPL 2</p>"
    "<p><a href=\"http://freshmemory.sourceforge.net\"> http://freshmemory.sourceforge.net </a></p>") );
}

void TMainWindow::RemoveCurTab()
{
int oldIndex = iDictTabs->currentIndex();
if( !ProposeToSave( oldIndex ) )
    return;
iModel->RemoveDictionary( oldIndex );
delete CurDictTab();    // Also magicly removes the tab from QTabWidget* iDictTabs
}

void TMainWindow::InsertCards()
{
TDictTableView*  tableView = CurTableView();
if( !tableView )
    return;
QAbstractItemModel* tableModel = tableView->model();
QItemSelectionModel* selectionModel = tableView->selectionModel();

// Insert number of rows equal to the selected rows, before the selection 
// If no selection, insert 1 row at the bottom
QList<int> rowsList = tableView->SelectedRows();
int minSelectedRow = tableModel->rowCount();
foreach( int row, rowsList )
    if( row < minSelectedRow )
        minSelectedRow = row;
int numRowsToInsert = rowsList.size() > 0? rowsList.size() : 1;
tableModel->insertRows( minSelectedRow, numRowsToInsert );

// Select the inserted rows and set the current index
QModelIndex topLeftCell = tableModel->index( minSelectedRow, 0, QModelIndex() );
QModelIndex bottomRightCell = tableModel->index( minSelectedRow + numRowsToInsert - 1, 1, QModelIndex() );  // the second column
selectionModel->select( QItemSelection(topLeftCell, bottomRightCell), QItemSelectionModel::ClearAndSelect );
selectionModel->setCurrentIndex( topLeftCell, QItemSelectionModel::Current );
}

void TMainWindow::InsertCardsAfter()
{
TDictTableView*  tableView = CurTableView();
if( !tableView )
    return;
QAbstractItemModel* tableModel = tableView->model();
QItemSelectionModel* selectionModel = tableView->selectionModel();

// Insert number of rows equal to the selected rows, after the selection 
// If no selection, insert 1 row at the bottom.
QList<int> rowsList = tableView->SelectedRows();
int maxSelectedRow = 0;
int numRowsToInsert;
if( !rowsList.isEmpty() )
    {
    foreach( int row, rowsList )
        if( row > maxSelectedRow )
            maxSelectedRow = row;
    numRowsToInsert = rowsList.size();
    }
else
    {
    maxSelectedRow = tableModel->rowCount();
    numRowsToInsert = 1;
    }
tableModel->insertRows( maxSelectedRow + 1, numRowsToInsert );

// Select the inserted rows and set the current index
QModelIndex topLeftCell = tableModel->index( maxSelectedRow + 1, 0, QModelIndex() );
QModelIndex bottomRightCell = tableModel->index( maxSelectedRow + 1+ numRowsToInsert - 1, 1, QModelIndex() );  // the second column
selectionModel->select( QItemSelection(topLeftCell, bottomRightCell), QItemSelectionModel::ClearAndSelect );
selectionModel->setCurrentIndex( topLeftCell, QItemSelectionModel::Current );
}

void TMainWindow::RemoveCards()
{
TDictTableView*  tableView = CurTableView();
if( !tableView )
    return;
QAbstractItemModel* tableModel = tableView->model();
QItemSelectionModel* selectionModel = tableView->selectionModel();

// Remove all selection ranges and find the minimal row
QItemSelectionRange range;
int minSelectedRow = tableModel->rowCount();
while( selectionModel->hasSelection() )
    {
    range = selectionModel->selection().first();
    int topRow = range.top();
    tableModel->removeRows( topRow, range.height() );
    if( topRow < minSelectedRow )
        minSelectedRow = topRow;
    }

// Select the highest (minimal) row
QModelIndex indexToSelect = tableModel->index( minSelectedRow, 0, QModelIndex() );
selectionModel->setCurrentIndex( indexToSelect, QItemSelectionModel::Select );
}

void TMainWindow::Find()
{
TFindDialog::Execute( CurTableView(), this );
iFindAgainAct->setEnabled(true);
}

void TMainWindow::FindAgain()
{
TFindDialog::FindAgain( CurTableView() );
}

void TMainWindow::CreateActions()
{
// File

iNewAct = new QAction( QIcon(":/images/filenew.png"), tr("&New"), this);
iNewAct->setShortcut( tr("Ctrl+N") );
iNewAct->setStatusTip( tr("Create a new dictionary") );
connect( iNewAct, SIGNAL(triggered()), this, SLOT(NewFile()) );

iLoadAct = new QAction( QIcon(":/images/fileopen.png"), tr("&Load ..."), this);
iLoadAct->setShortcut( tr("Ctrl+L") );
iLoadAct->setStatusTip( tr("Load an existing dictionary") );
connect( iLoadAct, SIGNAL(triggered()), this, SLOT(Load()) );

iSaveAct = new QAction( QIcon(":/images/filesave.png"), tr("&Save"), this);
iSaveAct->setShortcut( tr("Ctrl+S") );
iSaveAct->setStatusTip( tr("Save the current dictionary") );
connect( iSaveAct, SIGNAL(triggered()), this, SLOT(Save()) );

iSaveAsAct = new QAction( QIcon(":/images/filesaveas.png"), tr("Save &as ..."), this);
iSaveAsAct->setStatusTip( tr("Save to another file...") );
connect( iSaveAsAct, SIGNAL(triggered()), this, SLOT(SaveAs()) );

iSaveCopyAct = new QAction( QIcon(":/images/filesaveas.png"), tr("Save &copy ..."), this);
iSaveCopyAct->setStatusTip( tr("Save a copy to another file. This doesn't change the file name of the dictionary.") );
connect( iSaveCopyAct, SIGNAL(triggered()), this, SLOT(SaveCopy()) );

iSaveStudyAct = new QAction( tr("Save study &data"), this);
iSaveStudyAct->setShortcut( tr("Ctrl+T") );
iSaveStudyAct->setStatusTip( tr("Save the data about flashcard repetitions") );
connect( iSaveStudyAct, SIGNAL(triggered()), this, SLOT(SaveStudy()) );

iImportAct = new QAction( QIcon(":/images/fileopen.png"), tr("&Import from CSV ..."), this);
iImportAct->setStatusTip( tr("Import from CSV file...") );
connect( iImportAct, SIGNAL(triggered()), this, SLOT(Import()) );

iExportAct = new QAction( QIcon(":/images/filesave.png"), tr("&Export to CSV ..."), this);
iExportAct->setStatusTip( tr("Export to CSV file...") );
connect( iExportAct, SIGNAL(triggered()), this, SLOT(Export()) );

iNoRecentFilesAct = new QAction( tr("None"), this);
iNoRecentFilesAct->setEnabled( false );
for (int i = 0; i < KMaxRecentFiles; i++)
    {
    iRecentFileActs[i] = new QAction(this);
    iRecentFileActs[i]->setVisible(false);
    connect( iRecentFileActs[i], SIGNAL(triggered()), this, SLOT(LoadRecentFile()) );
    }

iRemoveTabAct = new QAction( QIcon(":/images/remove.png"), tr("Close &tab"), this);
iRemoveTabAct->setShortcut( tr("Ctrl+W") );
iRemoveTabAct->setStatusTip( tr("Close the current dictionary") );
connect( iRemoveTabAct, SIGNAL(triggered()), this, SLOT(RemoveCurTab()) );

iQuitAct = new QAction( QIcon(":/images/exit.png"), tr("&Quit"), this);
iQuitAct->setShortcut( tr("Ctrl+Q") );
iQuitAct->setStatusTip( tr("Quit Fresh Memory") );
connect( iQuitAct, SIGNAL(triggered()), this, SLOT(close()) );


// Edit

iCopyAct = new QAction( QIcon(":/images/editcopy.png"), tr("&Copy"), this);
iCopyAct->setShortcut( tr("Ctrl+Insert") );
iCopyAct->setStatusTip( tr("Copy the selected cards to the clipboard") );
connect( iCopyAct, SIGNAL(triggered()), this, SLOT(CopyCards()) );

iCutAct = new QAction( QIcon(":/images/editcut.png"), tr("Cu&t"), this);
iCutAct->setShortcut( tr("Shift+Delete") );
iCutAct->setStatusTip( tr("Cut the selected cards from the clipboard") );
connect( iCutAct, SIGNAL(triggered()), this, SLOT(CutCards()) );

iPasteAct = new QAction( QIcon(":/images/editpaste.png"), tr("&Paste"), this);
iPasteAct->setShortcut( tr("Shift+Insert") );
iPasteAct->setStatusTip( tr("Paste cards from the clipboard") );
connect( iPasteAct, SIGNAL(triggered()), this, SLOT(PasteCards()) );

iCommentAct = new QAction( QIcon(":/images/green-check.png"), tr("Toggle c&omment"), this);
iCommentAct->setShortcut( tr("Ctrl+D") );
iCommentAct->setStatusTip( tr("Comment/Uncomment the selected cards") );
connect( iCommentAct, SIGNAL(triggered()), this, SLOT(ToggleCommentInCards()) );

iInsertCardsAct = new QAction( QIcon(":/images/add.png"), tr("&Insert cards before"), this);
iInsertCardsAct->setShortcut( tr("Ctrl+I") );
iInsertCardsAct->setStatusTip( tr("Insert empty cards before the selected one(s)") );
connect( iInsertCardsAct, SIGNAL(triggered()), this, SLOT(InsertCards()) );

iInsertCardsAfterAct = new QAction( QIcon(":/images/add-after.png"), tr("I&nsert cards after"), this);
iInsertCardsAfterAct->setShortcut( tr("Ctrl+J") );
iInsertCardsAfterAct->setStatusTip( tr("Insert empty cards after the selected one(s)") );
connect( iInsertCardsAfterAct, SIGNAL(triggered()), this, SLOT(InsertCardsAfter()) );

iRemoveCardsAct = new QAction( QIcon(":/images/delete.png"), tr("&Remove cards"), this);
iRemoveCardsAct->setShortcut( Qt::Key_Delete );
iRemoveCardsAct->setStatusTip( tr("Remove the selected cards") );
connect( iRemoveCardsAct, SIGNAL(triggered()), this, SLOT(RemoveCards()) );

iFindAct = new QAction( QIcon(":/images/find.png"), tr("&Find..."), this);
iFindAct->setShortcut( tr("Ctrl+F") );
iFindAct->setStatusTip( tr("Find a card...") );
connect( iFindAct, SIGNAL(triggered()), this, SLOT(Find()) );

iFindAgainAct = new QAction( QIcon(":/images/next.png"), tr("Find &again"), this);
iFindAgainAct->setShortcut( tr("F3") );
iFindAgainAct->setStatusTip( tr("Find again the same text") );
connect( iFindAgainAct, SIGNAL(triggered()), this, SLOT(FindAgain()) );

// Fields

iFieldManagerAct = new QAction( QIcon(":/images/field-manager.png"), tr("&Field manager"), this);
iFieldManagerAct->setShortcut( tr("Ctrl+M") );
iFieldManagerAct->setStatusTip( tr("Edit the card fields") );
connect( iFieldManagerAct, SIGNAL(triggered()), this, SLOT(OpenFieldManager()) );

iSwapQstnAnsrAct = new QAction( QIcon(":/images/vswap.png"), tr("&Swap questions and answers"), this);
iSwapQstnAnsrAct->setShortcut( tr("Ctrl+P") );
iSwapQstnAnsrAct->setStatusTip( tr("Swap question and answer fields") );
connect( iSwapQstnAnsrAct, SIGNAL(triggered()), this, SLOT(SwapQuestionAnswer()) );

// Study

iWordDrillAct = new QAction( QIcon(":/images/flashcards-24.png"), tr("&Word drill"), this);
iWordDrillAct->setShortcut( tr("F5") );
iWordDrillAct->setStatusTip( tr("Word drill: random browsing of flashcards") );
connect( iWordDrillAct, SIGNAL(triggered()), iSMStartTestTriggered, SLOT(map()) );
iSMStartTestTriggered->setMapping(iWordDrillAct, TAppModel::EWordDrill);

iSpacedRepetitionAct = new QAction( QIcon(":/images/clock.png"), tr("&Spaced repetition"), this);
iSpacedRepetitionAct->setShortcut( tr("F6") );
iSpacedRepetitionAct->setStatusTip( tr("Spaced repetition: automatic scheduling of flashcard repetitions") );
connect( iSpacedRepetitionAct, SIGNAL(triggered()), iSMStartTestTriggered, SLOT(map()) );
iSMStartTestTriggered->setMapping(iSpacedRepetitionAct, TAppModel::ESpacedRepetition);

// Help
iAboutAct = new QAction( QIcon(":/images/mainicon.png"), tr("About &Fresh Memory"), this);
iAboutAct->setStatusTip( tr("Show information about Fresh Memory") );
connect( iAboutAct, SIGNAL(triggered()), this, SLOT(About()) );

iAboutQtAct = new QAction( QIcon(":/images/qt.png"), tr("About &Qt"), this);
iAboutQtAct->setStatusTip( tr("Show information about the Qt library") );
connect( iAboutQtAct, SIGNAL(triggered()), qApp, SLOT(aboutQt()) );

// Disable some actions
iSaveAct->setEnabled(false);
iSaveAsAct->setEnabled(false);
iSaveCopyAct->setEnabled(false);
iSaveStudyAct->setEnabled(false);
iExportAct->setEnabled(false);
iRemoveTabAct->setEnabled(false);
iCutAct->setEnabled(false);
iCopyAct->setEnabled(false);
iPasteAct->setEnabled(false);
iCommentAct->setEnabled(false);
iInsertCardsAct->setEnabled(false);
iInsertCardsAfterAct->setEnabled(false);
iRemoveCardsAct->setEnabled(false);
iFindAct->setEnabled(false);
iFindAgainAct->setEnabled(false);
iWordDrillAct->setEnabled(false);
iSpacedRepetitionAct->setEnabled(false);
iFieldManagerAct->setEnabled( false );
iSwapQstnAnsrAct->setEnabled( false );
}

void TMainWindow::CreateMenus()
{
iFileMenu = menuBar()->addMenu( tr("&File") );
iFileMenu->addAction(iNewAct);
iFileMenu->addAction(iLoadAct);
iFileMenu->addAction(iSaveAct);
iFileMenu->addAction(iSaveAsAct);
iFileMenu->addAction(iSaveCopyAct);
iFileMenu->addAction(iSaveStudyAct);
iFileMenu->addAction(iImportAct);
iFileMenu->addAction(iExportAct);

QMenu* recentFilesMenu = iFileMenu->addMenu( tr("&Recent files") );
recentFilesMenu->addAction(iNoRecentFilesAct);
for (int i = 0; i < KMaxRecentFiles; i++)
    recentFilesMenu->addAction(iRecentFileActs[i]);

iFileMenu->addSeparator();
iFileMenu->addAction(iRemoveTabAct);
iFileMenu->addAction(iQuitAct);

iEditMenu = menuBar()->addMenu( tr("&Edit") );
iEditMenu->addAction(iCutAct);
iEditMenu->addAction(iCopyAct);
iEditMenu->addAction(iPasteAct);
iEditMenu->addSeparator();
iEditMenu->addAction(iCommentAct);
iEditMenu->addAction(iInsertCardsAct);
iEditMenu->addAction(iInsertCardsAfterAct);
iEditMenu->addAction(iRemoveCardsAct);
iEditMenu->addSeparator();
iEditMenu->addAction(iFindAct);
iEditMenu->addAction(iFindAgainAct);

iViewMenu = menuBar()->addMenu( tr("&View") );
iToolbarsMenu = iViewMenu->addMenu( tr("&Toolbars") );
iToolbarsMenu->menuAction()->setStatusTip( tr("Show/hide toolbars") );

iFieldsMenu = menuBar()->addMenu( tr("&Fields") );
iFieldsMenu->addAction(iFieldManagerAct);
iFieldsMenu->addAction(iSwapQstnAnsrAct);

iStudyMenu = menuBar()->addMenu( tr("&Study") );
iStudyMenu->addAction(iWordDrillAct);
iStudyMenu->addAction(iSpacedRepetitionAct);

menuBar()->addSeparator();

iHelpMenu = menuBar()->addMenu( tr("&Help") );
iHelpMenu->addAction(iAboutAct);
iHelpMenu->addAction(iAboutQtAct);
}

void TMainWindow::CreateToolBars()
{
iFileToolBar = addToolBar( tr("File") );
iFileToolBar->setObjectName("File");
iFileToolBar->addAction(iNewAct);
iFileToolBar->addAction(iLoadAct);
iFileToolBar->addAction(iSaveAct);
iFileToolBar->addAction(iWordDrillAct);
iFileToolBar->addAction(iSpacedRepetitionAct);

iEditToolBar = addToolBar( tr("Edit") );
iEditToolBar->setObjectName("Edit");
iEditToolBar->addAction(iCutAct);
iEditToolBar->addAction(iCopyAct);
iEditToolBar->addAction(iPasteAct);
iEditToolBar->addSeparator();
iEditToolBar->addAction(iInsertCardsAct);
iEditToolBar->addAction(iRemoveCardsAct);
iEditToolBar->addSeparator();
iEditToolBar->addAction(iFindAct);

iFieldsToolBar = addToolBar( tr("Fields") );
iFieldsToolBar->setObjectName("Fields");
iFieldsToolBar->addAction(iFieldManagerAct);
iFieldsToolBar->addAction(iSwapQstnAnsrAct);

iToolbarsMenu->addAction( iFileToolBar->toggleViewAction() );
iToolbarsMenu->addAction( iEditToolBar->toggleViewAction() );
iToolbarsMenu->addAction( iFieldsToolBar->toggleViewAction() );
}

void TMainWindow::CreateStatusBar()
{
statusBar()->showMessage( tr("Ready") );
statusBar()->addPermanentWidget( iTotalCardsLabel = new QLabel );
statusBar()->addPermanentWidget( iValidCardsLabel = new QLabel );
statusBar()->addPermanentWidget( iUnrepCardsLabel = new QLabel );
UpdateCardsNumLabels();
}

void TMainWindow::CreateCentralWidget()
{
// Tab widget
iDictTabs = new QTabWidget;
connect( iDictTabs, SIGNAL(currentChanged(int)), iModel, SLOT(SetCurDictionary(int)) );
connect( iDictTabs, SIGNAL(currentChanged(int)), this, SLOT(UpdateActions()) );
connect( iDictTabs, SIGNAL(currentChanged(int)), this, SLOT(UpdateSelectionActions()) );
connect( iDictTabs, SIGNAL(currentChanged(int)), this, SLOT(UpdateCardsNumLabels()) );

iDictTabs->addAction(iCutAct);
iDictTabs->addAction(iCopyAct);
iDictTabs->addAction(iPasteAct);
iDictTabs->addAction(iCommentAct);
iDictTabs->addAction(iInsertCardsAct);
iDictTabs->addAction(iInsertCardsAfterAct);
iDictTabs->addAction(iRemoveCardsAct);
iDictTabs->setContextMenuPolicy( Qt::ActionsContextMenu );

// Delete tab button
QToolButton* delTabButton = new QToolButton();
delTabButton->setIcon( iRemoveTabAct->icon() );
delTabButton->setToolTip( iRemoveTabAct->toolTip() );
delTabButton->setStatusTip( iRemoveTabAct->statusTip() );
iDictTabs->setCornerWidget( delTabButton );
connect( delTabButton, SIGNAL(clicked()), this, SLOT(RemoveCurTab()) );

setCentralWidget( iDictTabs );
}

void TMainWindow::CopyCards()
{
TDictTableView*  tableView = CurTableView();
if( !tableView )
    return;

// Gather the copied content from the selected cards
QList<int> rowsList = tableView->SelectedRows();
TDictionary* dict = iModel->CurDictionary();
QStringList copiedStrings;
foreach( int row, rowsList )
    copiedStrings << dict->Card( row )->ToCSVString();
tableView->HighlightSelectedRows();

// Set the clipboard
QClipboard *clipboard = QApplication::clipboard();
clipboard->setText( copiedStrings.join("\n") );
}

void TMainWindow::CutCards()
{
CopyCards();
RemoveCards();
}

void TMainWindow::PasteCards()
{
TDictTableView*  tableView = CurTableView();
if( !tableView )
    return;
QAbstractItemModel* tableModel = tableView->model();
QItemSelectionModel* selectionModel = tableView->selectionModel();

// Get content from the clipboard
QClipboard *clipboard = QApplication::clipboard();
QString pastedContent = clipboard->text();
QStringList pastedStrings = pastedContent.split("\n");

// Find where to paste
// No selection -> paste to the end
QList<int> rowsList = tableView->SelectedRows();
int rowToInsert = tableModel->rowCount();
foreach( int row, rowsList )
    if( row < rowToInsert )
        rowToInsert = row;

// Insert rows filled with the clipboard content
//int pastedRowsNum = static_cast<TDictTableModel*>(tableModel)->InsertXMLRows( rowToInsert, pastedContent );
int pastedRowsNum = pastedStrings.size();
static_cast<TDictTableModel*>(tableModel)->InsertRows( rowToInsert, pastedStrings );

// Select the inserted rows and set the current index
QModelIndex topLeftCell = tableModel->index( rowToInsert, 0, QModelIndex() );
QModelIndex bottomRightCell = tableModel->index( rowToInsert + pastedRowsNum - 1, 2, QModelIndex() );  // the second column
selectionModel->select( QItemSelection(topLeftCell, bottomRightCell), QItemSelectionModel::ClearAndSelect );
selectionModel->setCurrentIndex( topLeftCell, QItemSelectionModel::Current );
}

void TMainWindow::ToggleCommentInCards()
{
TDictTableView*  tableView = CurTableView();
if( !tableView )
    return;

QList<int> rowsList = tableView->SelectedRows();
TDictionary* dict = iModel->CurDictionary();
foreach( int row, rowsList )
    dict->Card( row )->ToggleCommented();
tableView->reset(); // Update the table font
}

void TMainWindow::ReadSettings()
{
QSettings settings;
move( settings.value("main-pos", QPoint(100, 100)).toPoint() );
resize( settings.value("main-size", QSize(600, 500)).toSize() );
restoreState( settings.value("main-state").toByteArray(), 0 );
iRecentFiles = settings.value("main-recent-files").toStringList();
}

void TMainWindow::WriteSettings()
{
QSettings settings;
settings.setValue("main-pos", pos());
settings.setValue("main-size", size());
settings.setValue("main-state", saveState( 0 ));
settings.setValue("main-recent-files", iRecentFiles);
}

void TMainWindow::StartTest(int aTestType)
{
TAppModel::TError error = iModel->StartTest(aTestType);
QString message;
switch( error )
    {
    case TAppModel::ENoCurDict:
        message = tr("The test cannot be started.\n No dictionary opened.");
        break;
    case TAppModel::EDictIsEmpty:
        message = tr("The test cannot be started.\n The current dictionary is empty.");
        break;
    default:
        return;
    }
QMessageBox::warning(this, iErrorTitle, message);
}

void TMainWindow::OpenFieldManager()
{
TFieldManager fieldManager( iModel->CurDictionary(), this );
fieldManager.exec();
UpdateDictTab();
UpdateCardsNumLabels();
}

void TMainWindow::SwapQuestionAnswer()
{
iModel->CurDictionary()->SwapQuestionAnswer();
}
