#include "TFlashcardsTestWindow.h"

#include <QPushButton>
#include <QLabel>
#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QShortcut>
#include <QToolButton>
#include <QCheckBox>
#include <QPushButton>
#include <QToolBar>
#include <QSettings>
#include <QProgressBar>

#include "TFlashcardsTestModel.h"
#include "TCard.h"
#include "TAppModel.h"
#include "TDictionary.h"

TFlashcardsTestWindow::TFlashcardsTestWindow(TFlashcardsTestModel* aModel):
    MTestWindow( aModel )
{
iCurQuestion.clear();
iCurAnswer.clear();

setWindowTitle("Word drill");
setWindowIcon( QIcon(":/images/flashcards-24.png") );

CreateCentralWidget();

ReadSettings();

Update();
}

void TFlashcardsTestWindow::CreateButtons()
{
iBackBtn = new QPushButton( QIcon(":/images/back.png"), tr("Back"), this);
iBackBtn->setFocusPolicy( Qt::NoFocus );
iBackBtn->setShortcut( Qt::Key_Left );
iBackBtn->setToolTip( tr("Go back in history") + ShortcutToStr( iBackBtn->shortcut() ) );
connect( iBackBtn, SIGNAL(clicked()), dynamic_cast<TFlashcardsTestModel*> (iModel), SLOT(GoBack()) );

iForwardBtn = new QPushButton( QIcon(":/images/forward.png"), tr("Forward"), this);
iForwardBtn->setFocusPolicy( Qt::NoFocus );
iForwardBtn->setShortcut( Qt::Key_Right );
iForwardBtn->setToolTip( tr("Go forward in history") + ShortcutToStr( iForwardBtn->shortcut() ) );
connect( iForwardBtn, SIGNAL(clicked()), dynamic_cast<TFlashcardsTestModel*> (iModel), SLOT(GoForward()) );

iAnsrBtn = new QPushButton( QIcon(":/images/info.png"), tr("Answer"), this);
iAnsrBtn->setFocusPolicy( Qt::NoFocus );
iAnsrBtn->setShortcut( tr("A") );
iAnsrBtn->setToolTip( tr("Toggle answer") + ShortcutToStr( iAnsrBtn->shortcut() ) );
connect( iAnsrBtn, SIGNAL(clicked()), this, SLOT(ToggleAnswer()) );

iNextBtn = new QPushButton( QIcon(":/images/next.png"), tr("Next"), this);
iNextBtn->setFocusPolicy( Qt::NoFocus );
iNextBtn->setShortcut( Qt::Key_Space );
iNextBtn->setToolTip( tr("Next question") + ShortcutToStr( iNextBtn->shortcut() ) );
connect( iNextBtn, SIGNAL(clicked()), iModel, SLOT(PickNextCard()) );
}

QString TFlashcardsTestWindow::ShortcutToStr( QKeySequence aShortcut )
{
return " (" + aShortcut.toString() + ")";
}

void TFlashcardsTestWindow::CreateCentralWidget()
{
CreateButtons();

QWidget* centralWidget = new QWidget;
QVBoxLayout* mainLayout = new QVBoxLayout( centralWidget );

UpdateFieldLabels();

QFrame* upperPanel = new QFrame;
upperPanel->setFrameStyle( QFrame::Box | QFrame::Sunken );
QHBoxLayout* dictLt = new QHBoxLayout( upperPanel );
QLabel* dicIconLabel = new QLabel;
dicIconLabel->setPixmap( QPixmap(":/images/openbook-24.png") );
iDicNameLabel = new QLabel( iModel->Dictionary()->ShortName() );
iQuitBtn = new QPushButton( QIcon(":/images/exit.png"), tr("Close"), this);
iQuitBtn->setFocusPolicy( Qt::NoFocus );
iQuitBtn->setShortcut( tr("Q") );
iQuitBtn->setToolTip( tr("Close Word drill") + ShortcutToStr( iQuitBtn->shortcut() ) );
connect( iQuitBtn, SIGNAL(clicked()), iModel, SLOT(Stop()) );
connect( iQuitBtn, SIGNAL(clicked()), this, SLOT(close()) );
dictLt->addWidget(dicIconLabel);
dictLt->addWidget(iDicNameLabel, 1);
dictLt->addWidget(iQuitBtn);

iQstnLabel = new QLabel();
iQstnLabel->setFrameStyle( QFrame::Plain | QFrame::Box );
iQstnLabel->setFont(QFont("Times New Roman", 18, QFont::Bold));
iQstnLabel->setAlignment(Qt::AlignCenter);
iQstnLabel->setTextFormat(Qt::RichText);
iQstnLabel->setWordWrap(true);
iQstnLabel->setPalette( iCardBackgroundColor );
iQstnLabel->setAutoFillBackground(true);

iAnsrLabel = new QLabel();
iAnsrLabel->setFrameStyle( QFrame::Plain | QFrame::Box );
iAnsrLabel->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
iAnsrLabel->setMinimumHeight(40);
iAnsrLabel->setFont(QFont("Times New Roman", 18));
iAnsrLabel->setAlignment(Qt::AlignCenter);
iAnsrLabel->setTextFormat(Qt::RichText);
iAnsrLabel->setWordWrap(true);
iAnsrLabel->setPalette( iCardBackgroundColor );
iAnsrLabel->setAutoFillBackground(false);

QHBoxLayout* cardNumLayout = new QHBoxLayout;
QLabel* cardIconLabel = new QLabel;
cardIconLabel->setPixmap( QPixmap(":/images/flashcards-24.png") );
iCardNumLabel = new QLabel;
iCardNumLabel->setAlignment( Qt::AlignVCenter );
iCardNumLabel->setFont(QFont("Times New Roman", 14));
iCardNumLabel->setMargin( 5 );
iPassesIconLabel = new QLabel;
iPassesIconLabel->setPixmap( QPixmap(":/images/passes-24.png") );
iPassesIconLabel->hide();
iPassesLabel = new QLabel;
iPassesLabel->setAlignment( Qt::AlignVCenter );
iPassesLabel->setFont(QFont("Times New Roman", 14));
iPassesLabel->setMargin( 5 );
iPassesLabel->hide();
iProgressBar = new QProgressBar;
iShowAnswersCB = new QCheckBox(tr("Show answers"));
iShowAnswersCB->setFocusPolicy( Qt::NoFocus );
iShowAnswersCB->setShortcut( tr("S") );
iShowAnswersCB->setToolTip( tr("Always show answers") + ShortcutToStr( iShowAnswersCB->shortcut() ) );
connect( iShowAnswersCB, SIGNAL(stateChanged(int)), this, SLOT(ToggleAnswer()) );
cardNumLayout->addWidget( cardIconLabel );
cardNumLayout->addWidget( iCardNumLabel );
cardNumLayout->addWidget( iPassesIconLabel );
cardNumLayout->addWidget( iPassesLabel );
cardNumLayout->addWidget( iProgressBar );
cardNumLayout->addWidget( iShowAnswersCB );

/* Add each card side into its own layout to not allow
to dynamically resize the question label, when showing/hiding the answer. */
QVBoxLayout* qstnLt = new QVBoxLayout;
qstnLt->addWidget(iQstnFieldLabel);
qstnLt->addWidget(iQstnLabel, 1);
QVBoxLayout* ansrLt = new QVBoxLayout;
ansrLt->addWidget(iAnsrFieldLabel);
ansrLt->addWidget(iAnsrLabel, 1);

QHBoxLayout* controlLt = new QHBoxLayout;
controlLt->addWidget( iBackBtn );
controlLt->addWidget( iForwardBtn );
controlLt->addWidget( iAnsrBtn );
controlLt->addWidget( iNextBtn );

mainLayout->addWidget( upperPanel );
mainLayout->addLayout( qstnLt );
mainLayout->addLayout( ansrLt );
mainLayout->addLayout( cardNumLayout );
mainLayout->addLayout( controlLt );

setCentralWidget( centralWidget );
}

void TFlashcardsTestWindow::Update()
{
if( !iModel->IsStarted() )
    return;
UpdateDictName();
DisplayCardNum();
UpdateCard(true);
UpdateButtons();
}

void TFlashcardsTestWindow::UpdateDictName()
{
if( !iModel->IsStarted() )
    return;
iDicNameLabel->setText( iModel->Dictionary()->ShortName() );
}

void TFlashcardsTestWindow::SetAnswerMode( TTestState aState )
{
iState = aState;
switch( iState )
    {
    case EAnswerVisible:
        if( iCurAnswer.isEmpty() )
            {
            SetAnswerMode( ENoAnswerVisibleExample );
            return;
            }
        ShowAnswer(true);
        iAnsrLabel->setEnabled(true);
        break;
    case EAnswerHidden:
        if( iCurAnswer.isEmpty() )
            {
            SetAnswerMode( ENoAnswerHiddenExample );
            return;
            }
        ShowAnswer(false);
        break;
    case ENoAnswerHiddenExample:
        if( !iCurAnswer.isEmpty() )
            {
            SetAnswerMode( EAnswerHidden );
            return;
            }
        iAnsrLabel->setAutoFillBackground(false);
        iAnsrLabel->setEnabled(false);
        iAnsrLabel->setText( tr("No answer") );
        iAnsrBtn->setEnabled( false );
        break;
    case ENoAnswerVisibleExample:
        if( !iCurAnswer.isEmpty() )
            {
            SetAnswerMode( EAnswerHidden );
            return;
            }
        ShowAnswer(true);
        break;
    case ENoCurCard:
    case ENoTestCards:
        ShowAnswer(false);
        break;
    }
}

void TFlashcardsTestWindow::ShowAnswer( bool aVisible )
{
iAnsrLabel->setAutoFillBackground( aVisible );
iAnsrLabel->setLineWidth( aVisible? 1 : 0 );
iAnsrLabel->setText( aVisible? iCurAnswer: "" );
iAnsrFieldLabel->setEnabled( aVisible );
bool curCardIsValid = iModel->CurPackSize() > 0 && !iCurQuestion.isEmpty();
iAnsrBtn->setEnabled( curCardIsValid && !iShowAnswersCB->isChecked() );
iAnsrFieldLabel->setVisible( curCardIsValid );
}

void TFlashcardsTestWindow::SetQuestionMode( TTestState aState )
{
iQstnLabel->setAutoFillBackground( iCurCard );
iQstnLabel->setLineWidth( iCurCard? 1 : 0 );
iQstnFieldLabel->setVisible( iCurCard );
switch( aState )
    {
    case ENoCurCard:
        iQstnLabel->setText( tr("No current card") );
        iAnsrBtn->setEnabled(false);
        break;
    case ENoTestCards:
        iQstnLabel->setText( tr("No cards available") );
        break;
    default:
        iQstnLabel->setText( iCurQuestion );
    }
}

void TFlashcardsTestWindow::UpdateCard( bool aNewCard )
{
/// @todo Do we need this iCurCard and aNewCard argument?
if( aNewCard )
    iCurCard = iModel->CurCard();
TTestState state( iState );
if( iModel->CurPackSize() <= 0 )
    {
    iCurQuestion.clear();
    state = ENoTestCards;
    }
else if( !iCurCard )
    {
    iCurQuestion.clear();
    state = ENoCurCard;
    }
else
    {
    iCurQuestion = iCurCard->Question();
    iCurAnswer = iCurCard->FormattedAnswer();
    if( aNewCard )
        state = iShowAnswersCB->isChecked()? EAnswerVisible: EAnswerHidden;
    }
SetQuestionMode( state );
SetAnswerMode( state );
}

void TFlashcardsTestWindow::DisplayCardNum()
{
int curCardNum = iModel->CurCardNum() + 1;
int historySize = iModel->HistorySize();
int sessionSize = iModel->SessionCardsNum();
int curPackSize = iModel->CurPackSize();
int curCardIxInCurPack = iModel->CurCardIxInCurPack() + 1;
int passNum;
if( curPackSize > 0 )
    passNum = sessionSize / curPackSize;
else
    passNum = 1;

QString cardNumStr = QString::number( curCardNum );
if( curCardNum < historySize )
    cardNumStr += QString(" (%1)").arg(historySize);
cardNumStr += QString(" / %1").arg( sessionSize );
iCardNumLabel->setText( cardNumStr );

iPassesIconLabel->setVisible( passNum > 1 && curCardNum == historySize );
iPassesLabel->setVisible( passNum > 1 && curCardNum == historySize );
if( passNum > 1 )
    iPassesLabel->setNum( passNum );
iProgressBar->setMaximum( curCardNum == historySize? curPackSize : sessionSize );
iProgressBar->setValue( curCardNum == historySize? curCardIxInCurPack : curCardNum );
}

void TFlashcardsTestWindow::UpdateButtons()
{
iBackBtn->setEnabled( iModel->CanGoBack() );
iForwardBtn->setEnabled( iModel->CanGoForward() );
iNextBtn->setEnabled( iModel->CurPackSize() > 0 );
iShowAnswersCB->setEnabled( iModel->CurPackSize() > 0 );
}

void TFlashcardsTestWindow::ToggleAnswer()
{
if( !iCurCard )
    return;
switch( iState )
    {
    case EAnswerHidden:
        SetAnswerMode( EAnswerVisible );
        break;
    case EAnswerVisible:
        SetAnswerMode( iShowAnswersCB->isChecked()? EAnswerVisible: EAnswerHidden );
        break;
    case ENoAnswerHiddenExample:
        SetAnswerMode( ENoAnswerVisibleExample );
        break;
    case ENoAnswerVisibleExample:
        SetAnswerMode( iShowAnswersCB->isChecked()? ENoAnswerVisibleExample: ENoAnswerHiddenExample );
        break;
    default:
        return;
    }
}

void TFlashcardsTestWindow::ReadSettings()
{
QSettings settings;
move( settings.value("worddrill-pos", QPoint(200, 200)).toPoint() );
resize( settings.value("worddrill-size", QSize(600, 430)).toSize() );
restoreState( settings.value("worddrill-state").toByteArray(), 0 );
}

void TFlashcardsTestWindow::WriteSettings()
{
QSettings settings;
settings.setValue("worddrill-pos", pos());
settings.setValue("worddrill-size", size());
settings.setValue("worddrill-state", saveState( 0 ));
}
