#include "TFindDialog.h"

#include <QLabel>
#include <QComboBox>
#include <QPushButton>
#include <QToolButton>
#include <QCheckBox>
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QGridLayout>
#include <QShowEvent>
#include <QMessageBox>
#include <QGroupBox>
#include <QMenu>
#include <QAction>
#include <QLineEdit>

#include "TDictTableView.h"

TFindDialog* TFindDialog::iInstance = NULL;

TFindDialog::TFindDialog( QWidget* aParent ):
    QDialog( aParent ), iSearchText(""), iSearchRegExp("")
{
QLabel* textLabel = new QLabel(tr("&Text to find:"));

iTextEdit = new QComboBox;
textLabel->setBuddy(iTextEdit);
iTextEdit->setEditable(true);
iTextEdit->setInsertPolicy( QComboBox::NoInsert );
iTextEdit->setMaxCount(10);
iTextEdit->setFocus();

iFindButton = new QPushButton(tr("&Find"));
iFindButton->setDefault(true);  // Enter
iFindButton->setEnabled(false);

iCloseButton = new QPushButton(tr("Close"));    // Esc
iCloseButton->setFocusPolicy( Qt::TabFocus );

iRegExpCB = new QCheckBox(tr("&Regular expression"));
iRegExpCB->setFocusPolicy( Qt::TabFocus );
iAddRegexpButton = new QToolButton();
iAddRegexpButton->setText(tr("&Add regexp..."));
iAddRegexpButton->setEnabled(false);
connect( iRegExpCB, SIGNAL(stateChanged(int)), this, SLOT(UpdateAddRegexpButton()) );
QHBoxLayout* regexpLayout = new QHBoxLayout;
regexpLayout->addWidget( iRegExpCB, 1 );
regexpLayout->addWidget( iAddRegexpButton );

QStringList regexpDescrList, regexpDataList;
regexpDescrList << tr(". (any character)");
regexpDataList << QString( "." );
regexpDescrList << tr("\\d (digit)");
regexpDataList << QString( "\\d" );
regexpDescrList << tr("\\D (non-digit)");
regexpDataList << QString( "\\D" );
regexpDescrList << tr("\\s (whitespace)");
regexpDataList << QString( "\\s" );
regexpDescrList << tr("\\S (non-whitespace)");
regexpDataList << QString( "\\S" );
regexpDescrList << tr("\\w (word character)");
regexpDataList << QString( "\\w" );
regexpDescrList << tr("\\W (non-word character)");
regexpDataList << QString( "\\W" );
regexpDescrList << tr("\\t (tabulation)");
regexpDataList << QString( "\\t" );
regexpDescrList << tr("\\\\ (backslash)");
regexpDataList << QString( "\\\\" );
regexpDescrList << QString( "" );
regexpDescrList << tr("[] (set of characters)");
regexpDataList << QString( "![]" ); // '!' means moving the cursor 1 position back
regexpDescrList << tr("[^] (except the set of characters)");
regexpDataList << QString( "![^]" );
regexpDescrList << tr("- (range of characters)");
regexpDataList << QString( "-" );
regexpDescrList << tr("| (OR)");
regexpDataList << QString( "|" );
regexpDescrList << QString( "" );
regexpDescrList << tr("? (match zero or one)");
regexpDataList << QString( "?" );
regexpDescrList << tr("+ (match one or more)");
regexpDataList << QString( "+" );
regexpDescrList << tr("* (match zero or more)");
regexpDataList << QString( "*" );
regexpDescrList << tr("{} (match N occurences)");
regexpDataList << QString( "!{}" );
regexpDescrList << QString( "" );
regexpDescrList << tr("^ (start of string)");
regexpDataList << QString( "^" );
regexpDescrList << tr("$ (end of string)");
regexpDataList << QString( "$" );
regexpDescrList << tr("\\b (word boundary)");
regexpDataList << QString( "\\b" );
regexpDescrList << tr("\\B (non-word boundary)");
regexpDataList << QString( "\\B" );
regexpDescrList << tr("(?=) (positive lookahead)");
regexpDataList << QString( "!(?=)" );
regexpDescrList << tr("(?!) (negative lookahead)");
regexpDataList << QString( "!(?!)" );

iAddRegexpMenu = new QMenu();
QListIterator< QString > descrIt( regexpDescrList );
QListIterator< QString > dataIt( regexpDataList );
while( descrIt.hasNext() )
    {
    QString actionDescr( descrIt.next() );
    if( actionDescr.isEmpty() )
        iAddRegexpMenu->addSeparator();
    else
        {
        QAction* action = new QAction( actionDescr, this );
        action->setData( dataIt.next() );
        iAddRegexpMenu->addAction( action );
        connect( action, SIGNAL(triggered()), this, SLOT(AddRegexpToEdit()) );
        }
    }
iAddRegexpButton->setMenu( iAddRegexpMenu );
iAddRegexpButton->setPopupMode( QToolButton::InstantPopup );

iCaseSensitiveCB = new QCheckBox(tr("&Case sensitive"));
iCaseSensitiveCB->setFocusPolicy( Qt::TabFocus );
iWholeWordsCB = new QCheckBox(tr("&Whole words only"));
iWholeWordsCB->setFocusPolicy( Qt::TabFocus );
iFromCursorCB = new QCheckBox(tr("Fr&om cursor"));
iFromCursorCB->setFocusPolicy( Qt::TabFocus );
iUpwardsCB = new QCheckBox(tr("&Upwards"));
iUpwardsCB->setFocusPolicy( Qt::TabFocus );
iInSelectionCB = new QCheckBox(tr("In &selection only"));
iInSelectionCB->setFocusPolicy( Qt::TabFocus );
iInQuestionsCB = new QCheckBox(tr("In &questions"));
iInQuestionsCB->setChecked( true );
iInQuestionsCB->setFocusPolicy( Qt::TabFocus );
iInAnswersCB = new QCheckBox(tr("In &answers"));
iInAnswersCB->setFocusPolicy( Qt::TabFocus );
iInAnswersCB->setChecked( true );
iInExamplesCB = new QCheckBox(tr("In &examples"));
iInExamplesCB->setFocusPolicy( Qt::TabFocus );
iInExamplesCB->setToolTip(tr("Examples can be only in answer fields"));
iInCommentsCB = new QCheckBox(tr("In co&mments"));
iInCommentsCB->setToolTip(tr("Comments include both question and answer fields of commented cards"));
iInCommentsCB->setFocusPolicy( Qt::TabFocus );

QGridLayout* optionsLayout = new QGridLayout;
optionsLayout->addWidget( iCaseSensitiveCB, 0, 0 );
optionsLayout->addWidget( iWholeWordsCB, 1, 0 );
optionsLayout->addWidget( iFromCursorCB, 2, 0 );
optionsLayout->addWidget( iUpwardsCB, 3, 0 );
optionsLayout->addWidget( iInSelectionCB, 0, 1 );
optionsLayout->addWidget( iInQuestionsCB, 1, 1 );
optionsLayout->addWidget( iInAnswersCB, 2, 1 );
optionsLayout->addWidget( iInExamplesCB, 3, 1 );
optionsLayout->addWidget( iInCommentsCB, 4, 1 );
QGroupBox* optionsGroup = new QGroupBox(tr("Options"));
optionsGroup->setLayout( optionsLayout );

QHBoxLayout* bottomLayout = new QHBoxLayout;
bottomLayout->addStretch();
bottomLayout->addWidget( iFindButton );
bottomLayout->addWidget( iCloseButton );

QVBoxLayout* mainLayout = new QVBoxLayout;
mainLayout->addWidget( textLabel, 0, Qt::AlignLeft );
mainLayout->addWidget( iTextEdit );
mainLayout->addLayout( regexpLayout );
mainLayout->addWidget( optionsGroup );
mainLayout->addLayout( bottomLayout );
setLayout( mainLayout );

connect( iInSelectionCB, SIGNAL(stateChanged(int)), this, SLOT(UpdateFromCursorCB()) );
connect( iTextEdit, SIGNAL(editTextChanged(const QString&)),
    this, SLOT(UpdateFindButton()) );
connect( iInQuestionsCB, SIGNAL(stateChanged(int)), this, SLOT(UpdateFindButton()) );
connect( iInAnswersCB, SIGNAL(stateChanged(int)), this, SLOT(UpdateFindButton()) );
connect( iInExamplesCB, SIGNAL(stateChanged(int)), this, SLOT(UpdateFindButton()) );
connect( iInCommentsCB, SIGNAL(stateChanged(int)), this, SLOT(UpdateFindButton()) );
connect( iFindButton, SIGNAL(clicked()), this, SLOT(FindClicked()) );
connect( iCloseButton, SIGNAL(clicked()), this, SLOT(reject()) );

setWindowTitle(tr("Find card"));
setFixedHeight( sizeHint().height() );
}

void TFindDialog::FindClicked()
{
accept();

// Save the entered text
iSearchText = iTextEdit->currentText();
int textIndex = iTextEdit->findText( iSearchText );
if( textIndex > -1 ) // Remove duplicates
    iTextEdit->removeItem( textIndex );
iTextEdit->insertItem( 0, iSearchText );   // Fresh text will be always first

// Process search text options
bool regExp = iRegExpCB->isChecked();
Qt::CaseSensitivity caseSensitivity = iCaseSensitiveCB->isChecked()?
    Qt::CaseSensitive : Qt::CaseInsensitive;
bool wholeWords = iWholeWordsCB->isChecked();

// Get the search regular expression
QString searchRegExpStr;
if( !regExp )
    searchRegExpStr = QRegExp::escape( iSearchText );
else
    searchRegExpStr = iSearchText;
if( wholeWords )
    searchRegExpStr = "\\b" + iSearchText + "\\b";
iSearchRegExp = QRegExp( searchRegExpStr, caseSensitivity);

Find( false );
}

void TFindDialog::Find( bool aFindAgain )
{
if( iSearchRegExp.isEmpty() || iSearchText.isEmpty() )
    return;

// Process location and direction options
bool fromCursor = (iFromCursorCB->isChecked() && !iInSelectionCB->isChecked()) || aFindAgain;
bool upwards = iUpwardsCB->isChecked();
bool inSelection = iInSelectionCB->isChecked();
bool inQuestions = iInQuestionsCB->isChecked();
bool inAnswers = iInAnswersCB->isChecked();
bool inExamples = iInExamplesCB->isChecked();
bool inComments = iInCommentsCB->isChecked();

/** @todo Use dictionary view only to set/get the current row. In other cases, use the dictionary model.
Remove duplicate functions from the dictionary view class. */
// Get sorted search range
QModelIndexList searchRange;
if( inSelection )
    searchRange = iTableView->SortedSelectedIndexes();
else
    searchRange = iTableView->Indexes();

// Remove empty and filtered items
QMutableListIterator<QModelIndex> i( searchRange );
while( i.hasNext() )
    {
    QModelIndex index = i.next();
    if( iTableView->IsEmpty(index) ||
        ( ( (!inQuestions && iTableView->model()->IsQuestion(index)) ||
            (!inAnswers && iTableView->model()->IsAnswer(index)) ||
            (!inExamples && iTableView->model()->IsExample(index)) ) &&
            !(inComments && iTableView->Commented(index)) ) ||
        (!inComments && iTableView->Commented(index)) )
        i.remove();
    }

// Get the starting search point (iterator)
QListIterator<QModelIndex> startingPoint( searchRange );
if( fromCursor && !inSelection )
    {
    startingPoint.findNext( iTableView->currentIndex() );
    if( upwards )
        startingPoint.previous();   // Go one item backwards
    }
else
    if( upwards )
        startingPoint.toBack();

// Try to find the regexp
bool found = false;
if( FindRegExp( iSearchRegExp, startingPoint ) )
    found = true;
else if ( fromCursor && !inSelection )
    {
    QMessageBox::StandardButton pressedButton;
    pressedButton = QMessageBox::information( this, windowTitle(), (!upwards)?
        tr("End of dictionary reached.\nContinue from the beginning?") :
        tr("Beginning of dictionary reached.\nContinue from the end?"),
        QMessageBox::Yes, QMessageBox::No );
    if( pressedButton == QMessageBox::Yes )
        { // Continue searching
        // Update the starting point
        if( !upwards )
            startingPoint.toFront();
        else
            startingPoint.toBack();
        // Try to find again
        if( FindRegExp( iSearchRegExp, startingPoint ) )
            found = true;
        }
    else
        found = true;   // Actually not, but user doesn't want to continue
    }
if( !found )
    QMessageBox::information( this, windowTitle(),
        tr("Search string \"%1\" not found.").arg(iSearchText), QMessageBox::Ok );
}

bool TFindDialog::FindRegExp( const QRegExp& aSearchRegExp, QListIterator<QModelIndex> aStartingPoint )
{
const QModelIndex* foundIndexPtr = iTableView->model()->Find(
    aSearchRegExp, aStartingPoint, iUpwardsCB->isChecked() );
if( foundIndexPtr )
    {
    iTableView->setCurrentIndex( *foundIndexPtr );
    return true;
    }
else
    return false;
}

void TFindDialog::UpdateFindButton()
{
iFindButton->setEnabled(
    !iTextEdit->currentText().isEmpty() &&
    (iInQuestionsCB->isChecked() || iInAnswersCB->isChecked() || iInExamplesCB->isChecked() || iInCommentsCB->isChecked())
    );
}

void TFindDialog::showEvent(QShowEvent* /*event*/)
{
iTextEdit->setCurrentIndex(0);

bool existsSelection = iTableView->selectionModel()->selectedIndexes().size() > 1;
iInSelectionCB->setEnabled( existsSelection );
iInSelectionCB->setChecked( existsSelection );

UpdateFromCursorCB();
}

void TFindDialog::UpdateFromCursorCB()
{
iFromCursorCB->setEnabled( !iInSelectionCB->isChecked() );
}

int TFindDialog::Execute(TDictTableView* aTableView, QWidget* aParent)
{
if( !iInstance )
    iInstance = new TFindDialog( aParent );
iInstance->SetTableView( aTableView );
return iInstance->exec();
}

bool TFindDialog::CanFindAgain(TDictTableView* aTableView)
{
if( !aTableView )
    return false;
if( !iInstance )
    return false;
if( iInstance->iTableView != aTableView )
    return false;
if( iInstance->iSearchText.isEmpty() )
    return false;
return true;
}

void TFindDialog::FindAgain(TDictTableView* aTableView)
{
if( !CanFindAgain(aTableView) )
    return;
iInstance->Find( true );
}

void TFindDialog::SetTableView( TDictTableView* aTableView )
{
iTableView = aTableView;
iSearchRegExp.setPattern("");
iSearchText.clear();
}

void TFindDialog::UpdateAddRegexpButton()
{
iAddRegexpButton->setEnabled( iRegExpCB->isChecked() );
}

void TFindDialog::AddRegexpToEdit()
{
QAction *action = qobject_cast<QAction*>( sender() );
if( action )
    {
    QString actData( action->data().toString() );
    bool moveCursorBack = false;
    if( actData[0] == '!' )
        {
        actData.remove( 0, 1 );
        moveCursorBack = true;
        }
    QLineEdit* lineEdit( iTextEdit->lineEdit() );
    lineEdit->insert( actData );
    if( moveCursorBack )
        lineEdit->cursorBackward( false );
    }
}

