#include "TCSVImportDialog.h"

#include <QLabel>
#include <QComboBox>
#include <QToolButton>
#include <QPushButton>
#include <QCheckBox>
#include <QSpinBox>
#include <QLineEdit>
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QGridLayout>
#include <QMessageBox>
#include <QGroupBox>
#include <QAction>
#include <QMenu>
#include <QTextCodec>
#include <QRadioButton>

#include "TDictTableView.h"
#include "TDictTableModel.h"
#include "CSVData.h"

QChar TCSVImportDialog::iSpaceChar( 0x2423 );
QChar TCSVImportDialog::iTabChar( 0x21A6 );

TCSVImportDialog::TCSVImportDialog( QWidget* aParent, QString aFilePath ):
    QDialog( aParent ), iFilePath( aFilePath ),
    iDictionary( NULL )
{
setWindowTitle(tr("Import from CSV"));
resize( 600, 550 );

// Input group box

QLabel* characterSetLabel = new QLabel(tr("C&haracter set:"));
iCharacterSetCombo = new QComboBox;
characterSetLabel->setBuddy( iCharacterSetCombo );
iCharacterSetCombo->setEditable(false);
QList<QByteArray> codecNames = QTextCodec::availableCodecs();
qSort( codecNames );
int systemIx = codecNames.indexOf( "System" );
codecNames.move( systemIx, 0 );
for( int i=0; i<codecNames.size(); i++)
    iCharacterSetCombo->addItem( codecNames[i], QTextCodec::codecForName( codecNames[i] )->mibEnum() );
int utf8Ix = codecNames.indexOf( "UTF-8" );
iCharacterSetCombo->setCurrentIndex( utf8Ix );

QLabel* fromRowLabel = new QLabel(tr("From &row:"));
iFromRowSpin = new QSpinBox();
fromRowLabel->setBuddy( iFromRowSpin );
iFromRowSpin->setValue( 1 );
iFromRowSpin->setMinimum( 1 );
iFromRowSpin->setMaximum( 1000000 );    // infinity

QLabel* colsToImportLabel = new QLabel(tr("Number of co&lumns:"));
iColsToImportSpin = new QSpinBox();
colsToImportLabel->setBuddy( iColsToImportSpin );
iColsToImportSpin->setValue( 3 );
iColsToImportSpin->setMinimum( 1 );

QGridLayout* inputLayout = new QGridLayout;
inputLayout->addWidget( characterSetLabel, 0, 0 );
inputLayout->addWidget( iCharacterSetCombo, 0, 1 );
inputLayout->addWidget( fromRowLabel, 1, 0);
inputLayout->addWidget( iFromRowSpin, 1, 1);
inputLayout->addWidget( colsToImportLabel, 2, 0);
inputLayout->addWidget( iColsToImportSpin, 2, 1);
inputLayout->setColumnStretch( 1, 1 );
QGroupBox* inputGroup = new QGroupBox(tr("Input"));
inputGroup->setLayout( inputLayout );

// Separators group box

QLabel* separatorsLabel = new QLabel(tr("Field &separators:"));
iSeparatorsEdit = new QLineEdit( QString( iTabChar ) + "&" );
separatorsLabel->setBuddy( iSeparatorsEdit );
iAddSeparatorButton = new QToolButton();
iAddSeparatorButton->setText(tr("&Add..."));

QStringList regexpDescrList, regexpDataList;
regexpDescrList << tr("Comma (,)");
regexpDataList << QString( "," );
regexpDescrList << tr("Semicolon (;)");
regexpDataList << QString( ";" );
regexpDescrList << tr("Tab (%1)").arg(iTabChar);
regexpDataList << QString( iTabChar );
regexpDescrList << tr("Space (%1)").arg(iSpaceChar);
regexpDataList << QString( iSpaceChar );
regexpDescrList << tr("Ampersand (%1)").arg("&&");
regexpDataList << QString( "&" );
regexpDescrList << tr("Hash (#)");
regexpDataList << QString( "#" );
iSeparatorsMenu = new QMenu();
QListIterator< QString > descrIt( regexpDescrList );
QListIterator< QString > dataIt( regexpDataList );
while( descrIt.hasNext() )
    {
    QAction* action = new QAction( descrIt.next(), this );
    action->setData( dataIt.next() );
    iSeparatorsMenu->addAction( action );
    connect( action, SIGNAL(triggered()), this, SLOT(AddSeparatorToEdit()) );
    }
iAddSeparatorButton->setMenu( iSeparatorsMenu );
iAddSeparatorButton->setPopupMode( QToolButton::InstantPopup );

QLabel* fieldsSeparatedLabel = new QLabel("Separated by:");
iAnyCharacterRB = new QRadioButton("An&y character");
iAnyCombinationRB = new QRadioButton("Any co&mbination");
iAnyCombinationRB->setChecked( true );
iExactStringRB = new QRadioButton("E&xact string");

QRegExp oneCharRegExp("\\S{1}");
QValidator *oneCharValidator = new QRegExpValidator(oneCharRegExp, this);

iTextDelimiterCB = new QCheckBox(tr("&Text delimiter:"));
iTextDelimiterCB->setChecked( true );
iTextDelimiterCombo = new QComboBox;
iTextDelimiterCombo->setEditable(true);
iTextDelimiterCombo->addItem( "\"" );
iTextDelimiterCombo->addItem( "\'" );
iTextDelimiterCombo->setInsertPolicy( QComboBox::NoInsert );
iTextDelimiterCombo->setValidator( oneCharValidator );

iCommentCharacterCB = new QCheckBox(tr("&Comment:"));
iCommentCharacterCB->setChecked( true );
iCommentCharacterCombo = new QComboBox;
iCommentCharacterCombo->setEditable(true);
iCommentCharacterCombo->addItem( "#" );
iCommentCharacterCombo->addItem( ";" );
iCommentCharacterCombo->setInsertPolicy( QComboBox::NoInsert );
iCommentCharacterCombo->setValidator( oneCharValidator );

QGridLayout* separatorsLayout = new QGridLayout;
separatorsLayout->addWidget( separatorsLabel, 0, 0 );
separatorsLayout->addWidget( iSeparatorsEdit, 0, 1 );
separatorsLayout->addWidget( iAddSeparatorButton, 0, 2 );
separatorsLayout->addWidget( fieldsSeparatedLabel, 1, 0, 3, 1, Qt::AlignTop );
separatorsLayout->addWidget( iAnyCharacterRB, 1, 1, 1, 2 );
separatorsLayout->addWidget( iAnyCombinationRB, 2, 1, 1, 2 );
separatorsLayout->addWidget( iExactStringRB, 3, 1, 1, 2 );
separatorsLayout->addWidget( iTextDelimiterCB, 4, 0 );
separatorsLayout->addWidget( iTextDelimiterCombo, 4, 1, 1, 2 );
separatorsLayout->addWidget( iCommentCharacterCB, 5, 0 );
separatorsLayout->addWidget( iCommentCharacterCombo, 5, 1, 1, 2 );
separatorsLayout->setColumnStretch( 1, 1 );
QGroupBox* separatorsGroup = new QGroupBox(tr("Separators"));
separatorsGroup->setLayout( separatorsLayout );

// Top layout

QHBoxLayout* topLayout = new QHBoxLayout;
topLayout->addWidget( inputGroup );
topLayout->addWidget( separatorsGroup );

// Preview group box

QLabel* previewLabel = new QLabel(tr("&Preview:"));
previewLabel->setBuddy( iPreviewTable );
iDictionary = new TDictionary();
iPreviewModel = new TDictTableModel( iDictionary, false );
UpdatePreview(false);   // import as many columns as possible
iColsToImportSpin->setValue( iDictionary->FieldsNum() );
iPreviewTable = new TDictTableView( iPreviewModel );

// OK - Cancel layout

iOkButton = new QPushButton(tr("&OK"));
iOkButton->setDefault(true);  // Enter

iCancelButton = new QPushButton(tr("Ca&ncel"));    // Esc

QHBoxLayout* okCancelLayout = new QHBoxLayout;
okCancelLayout->addStretch();
okCancelLayout->addWidget( iOkButton );
okCancelLayout->addWidget( iCancelButton );

// Main layout

QVBoxLayout* mainLayout = new QVBoxLayout;
mainLayout->addLayout( topLayout );
mainLayout->addWidget( previewLabel );
mainLayout->addWidget( iPreviewTable );
mainLayout->addLayout( okCancelLayout );
setLayout( mainLayout );

connect( iTextDelimiterCB, SIGNAL(stateChanged(int)), this, SLOT(UpdateTextDelimiterCombo()) );
connect( iCommentCharacterCB, SIGNAL(stateChanged(int)), this, SLOT(UpdateCommentCharacterCombo()) );
connect( iOkButton, SIGNAL(clicked()), this, SLOT(accept()) );
connect( iCancelButton, SIGNAL(clicked()), this, SLOT(reject()) );
connect( this, SIGNAL(rejected()), this, SLOT(DeleteDictionary()) );

connect( iCharacterSetCombo, SIGNAL(currentIndexChanged(int)), this, SLOT(UpdatePreview()) );
connect( iFromRowSpin, SIGNAL(valueChanged(int)), this, SLOT(UpdatePreview()) );
connect( iSeparatorsEdit, SIGNAL(textChanged(QString)), this, SLOT(UpdatePreview()) );
connect( iAnyCharacterRB, SIGNAL(toggled(bool)), this, SLOT(UpdatePreview()) );
connect( iAnyCombinationRB, SIGNAL(toggled(bool)), this, SLOT(UpdatePreview()) );
connect( iExactStringRB, SIGNAL(toggled(bool)), this, SLOT(UpdatePreview()) );
connect( iTextDelimiterCB, SIGNAL(stateChanged(int)), this, SLOT(UpdatePreview()) );
connect( iTextDelimiterCombo, SIGNAL(editTextChanged(QString)), this, SLOT(UpdatePreview()) );
connect( iCommentCharacterCB, SIGNAL(stateChanged(int)), this, SLOT(UpdatePreview()) );
connect( iCommentCharacterCombo, SIGNAL(editTextChanged(QString)), this, SLOT(UpdatePreview()) );
connect( iColsToImportSpin, SIGNAL(valueChanged(int)), this, SLOT(UpdatePreview()) );

}

TCSVImportDialog::~TCSVImportDialog()
{
delete iPreviewModel;
}

void TCSVImportDialog::AddSeparatorToEdit()
{
QAction *action = qobject_cast<QAction*>( sender() );
if( action )
    iSeparatorsEdit->insert( action->data().toString() );
}

void TCSVImportDialog::UpdateTextDelimiterCombo()
{
iTextDelimiterCombo->setEnabled( iTextDelimiterCB->isChecked() );
}

void TCSVImportDialog::UpdateCommentCharacterCombo()
{
iCommentCharacterCombo->setEnabled( iCommentCharacterCB->isChecked() );
}

QTextCodec* TCSVImportDialog::TextCodec()
{
return QTextCodec::codecForMib( iCharacterSetCombo->itemData( iCharacterSetCombo->currentIndex() ).toInt() );
}

QString TCSVImportDialog::FieldSeparators()
{
// Convert the visual characters for space and tab to normal ones.
QString separators( iSeparatorsEdit->text() );
separators.replace( iSpaceChar, ' ' );
separators.replace( iTabChar, '\t' );
return separators;
}

void TCSVImportDialog::UpdatePreview( bool aObeyImportCols )
{
TCSVImportData importData;
importData.iTextCodec = TextCodec();
importData.iFromRow = iFromRowSpin->value();
importData.iFieldSeparators = FieldSeparators();
if( iAnyCharacterRB->isChecked() )
    importData.iFieldSeparationMode = EFieldSeparatorAnyCharacter;
else if ( iAnyCombinationRB->isChecked() )
    importData.iFieldSeparationMode = EFieldSeparatorAnyCombination;
else if ( iExactStringRB->isChecked() )
    importData.iFieldSeparationMode = EFieldSeparatorExactString;
importData.iCommentChar = iCommentCharacterCB->isChecked()? iCommentCharacterCombo->currentText()[0]: QChar(0);
importData.iTextDelimiter = iTextDelimiterCB->isChecked()? iTextDelimiterCombo->currentText()[0]: QChar(0);
if( aObeyImportCols )
    importData.iColsToImport = iColsToImportSpin->value();
else
    importData.iColsToImport = 0;   // infinity
iDictionary->ImportCSV( iFilePath, importData );
iPreviewModel->DictionaryChanged();
}

void TCSVImportDialog::DeleteDictionary()
{
delete iDictionary;
iDictionary = NULL;
}
