#include "TSpacedRepetitionWindow.h"

#include <QPushButton>
#include <QLabel>
#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QShortcut>
#include <QToolButton>
#include <QCheckBox>
#include <QAction>
#include <QActionGroup>
#include <QToolBar>
#include <QSettings>
#include <QSignalMapper>
#include <QProgressBar>

#include "TSpacedRepetitionModel.h"
#include "TCard.h"
#include "TAppModel.h"
#include "TDictionary.h"

TSpacedRepetitionWindow::TSpacedRepetitionWindow(TSpacedRepetitionModel* aModel):
    MTestWindow( aModel )
{
iCurQuestion.clear();
iCurAnswer.clear();

setWindowTitle("Spaced repetition");
setWindowIcon( QIcon(":/images/clock.png") );

CreateCentralWidget();
CreateStatusBar();

ReadSettings();
Update();
}

void TSpacedRepetitionWindow::CreateButtons()
{
iAnswerBtn = new QPushButton( QIcon(":/images/info.png"), tr("Answer"), this );
iAnswerBtn->setFocusPolicy( Qt::NoFocus );
iAnswerBtn->setShortcut( Qt::Key_Space );
iAnswerBtn->setToolTip( tr("Show answer") + ShortcutToStr( iAnswerBtn->shortcut() ) );
connect( iAnswerBtn, SIGNAL(clicked()), this, SLOT(ToggleAnswer()) );

iSMCardGraded = new QSignalMapper( this );
connect( iSMCardGraded, SIGNAL(mapped(int)), iModel, SLOT(ScheduleCard(int)) );
for( int i = 0; i < TSpacedRepetitionModel::KGradesNum; i++ )
	{
	iGradeBtns[i] = new QPushButton( QString::number(i), this );
    iGradeBtns[i]->setFocusPolicy( Qt::NoFocus );
    iGradeBtns[i]->setShortcut( QString::number(i) );
	iSMCardGraded->setMapping( iGradeBtns[i], i );
	connect( iGradeBtns[i], SIGNAL(clicked()), iSMCardGraded, SLOT(map()) );
	}

iGradeBtns[5]->setStatusTip( tr("5: Too easy. The intervals must increase more.") );
iGradeBtns[4]->setStatusTip( tr("4: Correct answer, the interval was just OK.") );
iGradeBtns[3]->setStatusTip( tr("3: Difficult to recall. The interval was too long.") );
iGradeBtns[2]->setStatusTip( tr("2: Not completely correct answer.") );
iGradeBtns[1]->setStatusTip( tr("1: Wrong answer, but the question seemed familiar.") );
iGradeBtns[0]->setStatusTip( tr("0: Unknown card - either new or completely forgotten.") );
    
iAnswerGroup = new QWidget( this );
QHBoxLayout* ansrLt = new QHBoxLayout( iAnswerGroup );
ansrLt->addWidget( iAnswerBtn );
for( int i = 0; i < TSpacedRepetitionModel::KGradesNum; i++ )
    ansrLt->addWidget( iGradeBtns[i] );

iNextBtn = new QPushButton( QIcon(":/images/next.png"), tr("Next card"), this );
iNextBtn->setShortcut( Qt::Key_Space );
iNextBtn->setStatusTip( tr("Move on to the next card") + ShortcutToStr( iNextBtn->shortcut() ) );
connect( iNextBtn, SIGNAL(clicked()), iModel, SLOT(PickNextCard()) );

iNextGroup = iNextBtn;
iNextGroup->setVisible( false );

iContinueBtn = new QPushButton( QIcon(":/images/player_fwd-22.png"), tr("&Next day"), this );
iContinueBtn->setToolTip( tr("Cards scheduled for the next day") + ShortcutToStr( iContinueBtn->shortcut() ) );
connect( iContinueBtn, SIGNAL(clicked()), iModel, SLOT(AddFutureActiveCards()) );

iUnrepeatedBtn = new QPushButton( QIcon(":/images/add-unrep.png"), tr("&More unrepeated"), this );
iUnrepeatedBtn->setToolTip( tr("Add unrepeated cards") + ShortcutToStr( iUnrepeatedBtn->shortcut() ) );
connect( iUnrepeatedBtn, SIGNAL(clicked()), iModel, SLOT(AddMoreUnrepeatedCards()) );

iUpdateBtn = new QPushButton( QIcon(":/images/gear.png"), tr("&Update"), this );
iUpdateBtn->setToolTip( tr("Check for active cards") + ShortcutToStr( iUpdateBtn->shortcut() ) );
connect( iUpdateBtn, SIGNAL(clicked()), iModel, SLOT(UpdateActiveCards()) );

iSwapQstnAnsrBtn = new QPushButton( QIcon(":/images/vswap.png"), tr("&Reversed"), this);
iSwapQstnAnsrBtn->setToolTip( tr("Swap question and answer") + ShortcutToStr( iSwapQstnAnsrBtn->shortcut() ) );
connect( iSwapQstnAnsrBtn, SIGNAL(clicked()), iModel->Dictionary(), SLOT( SwapQuestionAnswer()) );

iContinueGroup = new QWidget( this );
iContinueGroup->setVisible( false );
QHBoxLayout* contLt = new QHBoxLayout( iContinueGroup );
contLt->addWidget( iContinueBtn );
contLt->addWidget( iUnrepeatedBtn );
contLt->addWidget( iUpdateBtn );
contLt->addWidget( iSwapQstnAnsrBtn );
}

QString TSpacedRepetitionWindow::ShortcutToStr( QKeySequence aShortcut )
{
return " (" + aShortcut.toString() + ")";
}

void TSpacedRepetitionWindow::CreateCentralWidget()
{
CreateButtons();

QWidget* centralWidget = new QWidget;
QVBoxLayout* mainLayout = new QVBoxLayout( centralWidget );

UpdateFieldLabels();

QFrame* upperPanel = new QFrame;
upperPanel->setFrameStyle( QFrame::Box | QFrame::Sunken );
QHBoxLayout* dictLt = new QHBoxLayout( upperPanel );
QLabel* dicIconLabel = new QLabel;
dicIconLabel->setPixmap( QPixmap(":/images/openbook-24.png") );
iDicNameLabel = new QLabel( iModel->Dictionary()->ShortName() );
iQuitBtn = new QPushButton( QIcon(":/images/exit.png"), tr("Close"), this);
iQuitBtn->setFocusPolicy( Qt::NoFocus );
iQuitBtn->setShortcut( tr("Q") );
iQuitBtn->setToolTip( tr("Close Spaced repetition") + ShortcutToStr( iQuitBtn->shortcut() ) );
connect( iQuitBtn, SIGNAL(clicked()), iModel, SLOT(Stop()) );
connect( iQuitBtn, SIGNAL(clicked()), this, SLOT(close()) );
dictLt->addWidget(dicIconLabel);
dictLt->addWidget(iDicNameLabel, 1);
dictLt->addWidget(iQuitBtn);

iQstnLabel = new QLabel();
iQstnLabel->setFrameStyle( QFrame::Plain | QFrame::Box );
iQstnLabel->setFont(QFont("Times New Roman", 18, QFont::Bold));
iQstnLabel->setAlignment(Qt::AlignCenter);
iQstnLabel->setTextFormat(Qt::RichText);
iQstnLabel->setWordWrap(true);
iQstnLabel->setPalette( iCardBackgroundColor );
iQstnLabel->setAutoFillBackground(true);

iAnsrLabel = new QLabel();
iAnsrLabel->setFrameStyle( QFrame::Plain | QFrame::Box );
iAnsrLabel->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
iAnsrLabel->setMinimumHeight(40);
iAnsrLabel->setFont(QFont("Times New Roman", 18));
iAnsrLabel->setAlignment(Qt::AlignCenter);
iAnsrLabel->setTextFormat(Qt::RichText);
iAnsrLabel->setWordWrap(true);
iAnsrLabel->setPalette( iCardBackgroundColor );
iAnsrLabel->setAutoFillBackground(false);

QHBoxLayout* cardNumLayout = new QHBoxLayout;
iDeadlineLabel = new QLabel;
iDeadlineLabel->setAlignment( Qt::AlignVCenter );
iDeadlineLabel->setMargin( 5 );
QLabel* cardIconLabel = new QLabel;
cardIconLabel->setPixmap( QPixmap(":/images/flashcards-24.png") );
iCardNumLabel = new QLabel;
iCardNumLabel->setAlignment(Qt::AlignVCenter);
iCardNumLabel->setFont(QFont("Times New Roman", 14));
iCardNumLabel->setMargin( 5 );
iProgressBar = new QProgressBar;
cardNumLayout->addWidget( cardIconLabel );
cardNumLayout->addWidget( iCardNumLabel );
cardNumLayout->addWidget( iProgressBar );
cardNumLayout->addWidget( iDeadlineLabel );

/* Add each card side into its own layout to not allow
to dynamically resize the question label, when showing/hiding the answer. */
QVBoxLayout* qstnLt = new QVBoxLayout;
qstnLt->addWidget(iQstnFieldLabel);
qstnLt->addWidget(iQstnLabel, 1);
QVBoxLayout* ansrLt = new QVBoxLayout;
ansrLt->addWidget(iAnsrFieldLabel);
ansrLt->addWidget(iAnsrLabel, 1);

QHBoxLayout* controlLt = new QHBoxLayout;
controlLt->addWidget(iAnswerGroup);
controlLt->addWidget(iNextGroup);
controlLt->addWidget(iContinueGroup);

mainLayout->addWidget( upperPanel );
mainLayout->addLayout( qstnLt );
mainLayout->addLayout( ansrLt );
mainLayout->addLayout( cardNumLayout );
mainLayout->addLayout( controlLt );

setCentralWidget( centralWidget );
}

void TSpacedRepetitionWindow::CreateStatusBar()
{
statusBar()->addPermanentWidget( iValidCardsLabel = new QLabel );
statusBar()->addPermanentWidget( iUnrepCardsLabel = new QLabel );
UpdateCardsNumLabels();
}

void TSpacedRepetitionWindow::Update()
{
if( !iModel->IsStarted() )
    return;
UpdateDictName();
DisplayCardNum();
UpdateCard(true);
UpdateButtons();
}

void TSpacedRepetitionWindow::UpdateDictName()
{
if( !iModel->IsStarted() )
    return;
iDicNameLabel->setText( iModel->Dictionary()->ShortName() );
UpdateCardsNumLabels();
}

void TSpacedRepetitionWindow::UpdateCardsNumLabels()
{
TDictionary* dict = iModel->Dictionary();
if( !dict )
    return;
iValidCardsLabel->setText( tr("Valid: %1").arg( dict->ValidCardsNum() ) );
iUnrepCardsLabel->setText( tr("Unrepeated: %1").arg( dict->UnrepCardsNum() ) );
}

/// @todo Try to simplify the system of states in the test windows
void TSpacedRepetitionWindow::SetAnswerMode( TTestState aState )
{
iState = aState;
switch( iState )
    {
    case EAnswerVisible:
        if( iCurAnswer.isEmpty() )
            {
            SetAnswerMode( ENoAnswerVisibleExample );
            return;
            }
        ShowAnswer(true);
        iAnsrLabel->setEnabled(true);
        UpdateButtons();
        iAnswerBtn->setEnabled( false );
        break;
    case EAnswerHidden:
        if( iCurAnswer.isEmpty() )
            {
            SetAnswerMode( ENoAnswerHiddenExample );
            return;
            }
        ShowAnswer(false);
        break;
    case ENoAnswerHiddenExample:
        if( !iCurAnswer.isEmpty() )
            {
            SetAnswerMode( EAnswerHidden );
            return;
            }
        iAnsrLabel->setAutoFillBackground(false);
        iAnsrLabel->setEnabled(false);
        iAnsrLabel->setText( tr("No answer") );
        iAnswerBtn->setEnabled( false );
        break;
    case ENoAnswerVisibleExample:
        if( !iCurAnswer.isEmpty() )
            {
            SetAnswerMode( EAnswerHidden );
            return;
            }
        ShowAnswer(true);
        break;
    case ENoCurCard:
    case ENoTestCards:
        ShowAnswer(false);
        break;
    }
}

void TSpacedRepetitionWindow::ShowAnswer( bool aVisible )
{
iAnsrLabel->setAutoFillBackground( aVisible );
iAnsrLabel->setLineWidth( aVisible? 1 : 0 );
iAnsrLabel->setText( aVisible? iCurAnswer: "" );
iAnsrFieldLabel->setEnabled( aVisible );
bool curCardIsValid = iModel->CurPackSize() > 0 && !iCurQuestion.isEmpty();
iAnswerBtn->setEnabled( curCardIsValid );
iAnsrFieldLabel->setVisible( curCardIsValid );
}

void TSpacedRepetitionWindow::SetQuestionMode( TTestState aState )
{
iQstnLabel->setAutoFillBackground( iCurCard );
iQstnLabel->setLineWidth( iCurCard? 1 : 0 );
iQstnFieldLabel->setVisible( iCurCard );
switch( aState )
    {
    case ENoCurCard:
        iQstnLabel->setText( tr("No current card") );
        SwitchToNextControls();
        break;
    case ENoTestCards:
        iQstnLabel->setText( tr("No more cards scheduled for now") );
        SwitchToContinueControls();
        break;
    default:
        iQstnLabel->setText( iCurQuestion );
        SwitchToAnswerControls();
    }
}

void TSpacedRepetitionWindow::UpdateCard( bool aNewCard )
{
if( aNewCard )
    iCurCard = iModel->CurCard();
TTestState state( iState );
if( iModel->CurCardIxInCurPack() < 0 )
    {
    iCurQuestion.clear();
    state = ENoTestCards;
    }
else if( !iCurCard )
    {
    iCurQuestion.clear();
    state = ENoCurCard;
    }
else
    {
    iCurQuestion = iCurCard->Question();
    iCurAnswer = iCurCard->FormattedAnswer();
    if( aNewCard )
        state = EAnswerHidden;
    }
SetQuestionMode( state );
SetAnswerMode( state );
}

void TSpacedRepetitionWindow::DisplayCardNum()
{
int curCardNum = iModel->CurCardNum() + 1;
int historySize = iModel->HistorySize();
int sessionSize = iModel->SessionCardsNum();

QDateTime curDeadline = dynamic_cast<TSpacedRepetitionModel*>(iModel)->ActiveCardsDeadline();
static QDateTime prevDeadline = curDeadline;
QString deadlineColor;
if( prevDeadline == curDeadline )
    deadlineColor="<font>";
else if( curDeadline <= QDateTime::currentDateTime() )
    deadlineColor="<font color=\"blue\">";
else
    deadlineColor="<font color=\"red\">";
iDeadlineLabel->setText( deadlineColor + curDeadline.toString( "yyyy-MMM-dd HH:mm:ss" ) + "</font>");

QString cardNumStr;
if( curCardNum > 0 )
    cardNumStr += QString::number( curCardNum );
else
    cardNumStr += QString(" (%1)").arg(historySize);
cardNumStr += QString(" / %1").arg(sessionSize);
iCardNumLabel->setText( cardNumStr );
    
iProgressBar->setMaximum( sessionSize );
if( iProgressBar->maximum() == 0 )
    iProgressBar->setMaximum( 1 );
iProgressBar->setValue( curCardNum );
}

void TSpacedRepetitionWindow::UpdateButtons()
{
#ifdef DEBUG
    int hideGradesNum = TSpacedRepetitionModel::KGradesNum - 1;
#else
    int hideGradesNum = TSpacedRepetitionModel::KGradesNum;
#endif
for( int i = 0; i < hideGradesNum; i++ )
    iGradeBtns[i]->setEnabled( iState == EAnswerVisible );
}

void TSpacedRepetitionWindow::ToggleAnswer()
{
if( !iCurCard )
    return;
switch( iState )
    {
    case EAnswerHidden:
        SetAnswerMode( EAnswerVisible );
        break;
    case EAnswerVisible:
        SetAnswerMode( EAnswerHidden );
        break;
    case ENoAnswerHiddenExample:
        SetAnswerMode( ENoAnswerVisibleExample );
        break;
    case ENoAnswerVisibleExample:
        SetAnswerMode( ENoAnswerHiddenExample );
        break;
    default:
        return;
    }
}

void TSpacedRepetitionWindow::ReadSettings()
{
QSettings settings;
move( settings.value("spacedrep-pos", QPoint(200, 200)).toPoint() );
resize( settings.value("spacedrep-size", QSize(600, 430)).toSize() );
restoreState( settings.value("spacedrep-state").toByteArray(), 0 );
}

void TSpacedRepetitionWindow::WriteSettings()
{
QSettings settings;
settings.setValue("spacedrep-pos", pos());
settings.setValue("spacedrep-size", size());
settings.setValue("spacedrep-state", saveState( 0 ));
}

void TSpacedRepetitionWindow::SwitchToAnswerControls()
{
if( iAnswerGroup->isVisible() )
    return;
if( !iCurCard )
    return;
iContinueGroup->setVisible( false );
iNextGroup->setVisible( false );
iAnswerGroup->setVisible( true );
}

void TSpacedRepetitionWindow::SwitchToContinueControls()
{
if( iContinueGroup->isVisible() )
    return;
iAnswerGroup->setVisible( false );
iNextGroup->setVisible( false );
iContinueGroup->setVisible( true );
iUnrepeatedBtn->setEnabled( dynamic_cast<TSpacedRepetitionModel*>(iModel)->UnrepeatedCardsNum() > 0 );
}

void TSpacedRepetitionWindow::SwitchToNextControls()
{
if( iNextGroup->isVisible() )
    return;
iAnswerGroup->setVisible( false );
iContinueGroup->setVisible( false );
iNextGroup->setVisible( true );
}

