#include "TDictionary.h"
#include "TDictParser.h"
#include "TStudyFileParser.h"
#include "version.h"

#include <QTextStream>
#include <QStringList>
#include <QtAlgorithms>
#include <QXmlSimpleReader>
#include <QXmlInputSource>
#include <QDateTime>
#include <QSettings>
#include <QDir>
//#include <QtDebug>

/*************************************
 * TField
 *************************************/

QString TField::ToXMLString( int aIx ) const
    {
    QString res = QString("<field id=\"%1\"").arg( id );
    QPair<TDictionary::TFieldRole, int> rolePair = iDict->FieldRole( aIx );
    switch( rolePair.first )
        {
        case TDictionary::EQuestionFieldRole:
            res += " question=\"yes\"";
            break;
        case TDictionary::EAnswerFieldRole:
            res += QString(" answer=\"%1\"").arg( rolePair.second );
            break;
        default:
            break;
        }
    if( style == EExampleFieldStyle )
        res += " style=\"example\"";
    res += "> " + name + " </field>\n";
    return res;
    }


/*************************************
 * TDictionary
 *************************************/

QString TDictionary::KNoName(tr("noname.fmd"));
QString TDictionary::KDictFileExtension(".fmd");
QString TDictionary::KStudyFileExtension(".fms");

TDictionary::TDictionary( const QString aFilePath ):
   iFilePath( aFilePath ), iQuestionField( 0 ), iIsContentModified( true ),
   iIsStudyModified( true ), iIsImported( false )
{
iId = QUuid::createUuid();
connect( this, SIGNAL(CardsInserted(int, int)), this, SLOT(SetContentModified()) );
connect( this, SIGNAL(CardsInserted(int, int)), this, SLOT(AddValidCards(int, int)) );
}

TDictionary::~TDictionary()
{
while (!iCards.isEmpty())
     delete iCards.takeFirst();
iValidCards.clear();
}

void TDictionary::SetDefaultFields()
{
iFields[0] = TField( "Question", TField::EDefaultFieldStyle, this );
iFields[1] = TField( "Answer", TField::EDefaultFieldStyle, this );
iFields[2] = TField( "Example", TField::EExampleFieldStyle, this );
iQuestionField = 0;
iAnswerFields.clear();
iAnswerFields[0] = 1;
iAnswerFields[1] = 2;
}

QFile::FileError TDictionary::Load(const QString aFilePath)
{
/// @todo Optimize error handling. Create the error string here. Other clients will use only the resulting error string.
while (!iCards.isEmpty())
     delete iCards.takeFirst();
QFile file( aFilePath );
if( !file.open( QIODevice::ReadOnly | QFile::Text ) ) // \r\n --> \n
    return file.error();
iFilePath = aFilePath;
TDictParser parser( this );
QXmlSimpleReader reader;
reader.setContentHandler( &parser );
reader.setErrorHandler( &parser );
QXmlInputSource xmlInputSource( &file );
if( reader.parse( xmlInputSource ) )
    {
    SetContentModified( parser.DictContentModified() );
    
    // Load study data
    TStudyFileParser studyParser( this );
    QXmlSimpleReader studyReader;
    studyReader.setContentHandler( &studyParser);
    studyReader.setErrorHandler( &studyParser );
    QFile studyFile( StudyFilePath() );
    QXmlInputSource studySource( &studyFile );
    studyReader.parse( studySource );
    SetStudyModified( false );
    
    UpdateValidCards();
    return QFile::NoError;
    }
iErrorStr = parser.errorString();
return QFile::ResourceError;
}

QFile::FileError TDictionary::Save( const QString aFilePath, bool aChangeFilePath )
{
QFile::FileError error = SaveContent( aFilePath );
if( error != QFile::NoError )
    return error;

if( iIsStudyModified )
    {
    error = SaveStudy();
    if( error != QFile::NoError )
        return error;
    }

if( aChangeFilePath && aFilePath != iFilePath )
    {
    iFilePath = aFilePath;
    emit FilePathChanged();
    }
return QFile::NoError;
}

QFile::FileError TDictionary::SaveContent( const QString aFilePath )
{
QFile file( aFilePath );
if( !file.open( QIODevice::WriteOnly | QFile::Text ) ) // \r\n --> \n
    return file.error();
QTextStream outStream(&file);
outStream.setCodec("UTF-8");
outStream << "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
    << "<!DOCTYPE freshmemory-dict>\n"
    << "<dict id=\"" << iId.toString() << "\" version=\"" << DIC_VERSION << "\">\n";
    
outStream << XMLIndent( 1 ) << "<fields>\n";
QMapIterator<int, TField> f( iFields );
while( f.hasNext() )
    {
    f.next();
    outStream << XMLIndent( 2 ) << f.value().ToXMLString( f.key() );
    }
outStream << XMLIndent( 1 ) << "</fields>\n";

foreach( TCard* card, iCards )
    outStream << card->ToXMLString( 1 );
outStream << "</dict>\n";
file.close();
SetContentModified( false );
iIsImported = false;
return QFile::NoError;
}

QFile::FileError TDictionary::SaveStudy()
{
if( !iIsStudyModified )
    return QFile::NoError;
QFile studyFile( StudyFilePath(true) );
if( !studyFile.open( QIODevice::WriteOnly | QFile::Text ) ) // \r\n --> \n
    return studyFile.error();
QTextStream studyStream( &studyFile );
studyStream.setCodec("UTF-8");
studyStream << "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
    << "<!DOCTYPE freshmemory-study>\n"
    << "<study version=\"" << DIC_VERSION << "\">\n";
studyStream << UnrepDataToXMLStudy( 1 );
foreach( TCard* card, iCards )
    studyStream << card->ToXMLStudy( 1 );
studyStream << "</study>\n";
studyFile.close();
SetStudyModified( false );
return QFile::NoError;
}

QString TDictionary::UnrepDataToXMLStudy( int aDepth )
{
if( iUnrepCardsData.isEmpty() )
    return QString();
QString res;
QMapIterator<TQstnAnsrPair, TUnrepCardsData> i( iUnrepCardsData );
while( i.hasNext() )
    {
    i.next();
    
    /** @todo The same processing of the default and reversed field ids as in TCard::ToXMLStudy().
    Put it to a separate function. */
    TQstnAnsrPair qstnAnsrPair = i.key();
    QUuid qstFieldId = Field( qstnAnsrPair.first ).id;
    QUuid ansFieldId = Field( qstnAnsrPair.second ).id;
    QUuid dictField0Id = Field( 0 ).id;
    QUuid dictField1Id = Field( 1 ).id;
    TUnrepCardsData unrep = i.value();
    res += XMLIndent( aDepth ) + "<unrep";
    bool reversed = false;
    bool fieldIdsAdded = false;
    if( qstFieldId != dictField0Id )
        {
        if( qstFieldId == dictField1Id )    // reversed
            reversed = true;
        else
            {
            res += QString(" qst=\"%1\"").arg( qstFieldId );
            fieldIdsAdded = true;
            }
        }
    if( ansFieldId != dictField1Id )
        {
        if( ansFieldId == dictField0Id )    // reversed
            reversed = true;
        else
            {
            res += QString(" ans=\"%1\"").arg( ansFieldId );
            fieldIdsAdded = true;
            }
        }
    if( reversed )
        res += " rev=\"yes\"";
    if( fieldIdsAdded )
        res += QString("\n") + XMLIndent( aDepth );
    else
        res += " ";
    res += QString("added=\"%1\" used=\"%2\" ")
        .arg( unrep.lastAddition.toString( Qt::ISODate ) )
        .arg( unrep.usedNum );
    res += "/>\n";
    }
return res;
}

QFile::FileError TDictionary::ImportCSV( const QString aFilePath, const TCSVImportData& aImportData )
{
while (!iCards.isEmpty())
     delete iCards.takeFirst();
QFile file( aFilePath );
if( !file.open( QIODevice::ReadOnly | QFile::Text ) ) // \r\n --> \n
    return file.error();
iFilePath = aFilePath + KDictFileExtension;
QTextStream inStream(&file);
inStream.setCodec( aImportData.iTextCodec );
// Ignore first rows
int rowNum = 1;
while(!inStream.atEnd() && rowNum++ < aImportData.iFromRow )
    inStream.readLine();
// Load lines and create cards
int fieldsNum = 0;
while(!inStream.atEnd())
    {
    QString line = inStream.readLine();
    TCard* card = new TCard( this, line, aImportData );
    iCards << card;
    fieldsNum = qMax( fieldsNum, card->FieldsNum() );
    }
iFields.clear();
for( int i=0; i<fieldsNum; i++ )
    iFields[i] = TField( QString::number( i+1 ), TField::EDefaultFieldStyle, this );
iQuestionField = 0;
iAnswerFields.clear();
iAnswerFields[0] = 1;
iIsImported = true;
SetContentModified( true );
UpdateValidCards();
return QFile::NoError;
}

QString TDictionary::ExportToCSVString( const TCSVExportData& aExportData ) const
{
QString outStr;
QTextStream outStream( &outStr );
bool emptyStr = false;
for(int i=aExportData.iFromRow-1; i<iCards.size(); i++)
    {
    QString outStr = iCards[i]->ToCSVString( aExportData );
    if( !(emptyStr && outStr.isEmpty()) )   // Don't print several empty strings in a row
        outStream << outStr << endl;
    emptyStr = outStr.isEmpty();
    }
return outStr;
}

QString TDictionary::ShortName(bool aMarkModified) const
{
QString fileName;
if( !iFilePath.isEmpty() )
    fileName = QFileInfo(iFilePath).fileName();
else
    fileName = KNoName;
if( aMarkModified )
    {
    if( iIsStudyModified )
        fileName = "+" + fileName;
    if( iIsContentModified )
        fileName = "*" + fileName;
    }
return fileName;
}

QString TDictionary::StudyFilePath( bool aCreateIfDoesntExist ) const
{
QSettings settings;
QFileInfo settingsInfo( settings.fileName() );
QString settingsPath = settingsInfo.path();
if( aCreateIfDoesntExist )
    {
    QDir settingsDir( settingsPath );
    settingsDir.mkdir( "study" );   /// @todo Check for true
    }
QString studyFilePath = settingsPath + "/study/" + iId.toString() + KStudyFileExtension;
return studyFilePath;
}

void TDictionary::SetContentModified( bool aModified )
{
if( aModified != iIsContentModified )  // The Content Modified state is changed
    {
    iIsContentModified = aModified;
    emit ContentModifiedChanged( iIsContentModified );
    }
}

void TDictionary::SetStudyModified( bool aModified )
{
if( aModified != iIsStudyModified )  // The Study Modified state is changed
    {
    iIsStudyModified = aModified;
    emit StudyModifiedChanged( iIsStudyModified );
    }
}

QDateTime TDictionary::UnrepCardsLastAddition() const
{
QDateTime time = iUnrepCardsData[ CurQstnAnsrPair() ].lastAddition;
if( time.isNull() )
    return QDateTime::fromTime_t( 0 );  // 1970-01-01T00:00:00
else
    return time;
}

void TDictionary::DecreaseUnrepCardsNumBy( int aDiff )
{
iUnrepCardsNum[ CurQstnAnsrPair() ] -= aDiff;
emit UnrepCardsChanged();
iUnrepCardsData[ CurQstnAnsrPair() ].usedNum += aDiff;
}

QDateTime TDictionary::ActiveCardsDeadline() const
{
QDateTime time = iActiveCardsDeadline[ CurQstnAnsrPair() ];
if( time.isNull() )
    return QDateTime::currentDateTime();
else
    return time;
}

const QPointer<TCard> TDictionary::Card(int aIndex) const
{
if( aIndex >= 0 && aIndex < iCards.size() )
    return iCards[aIndex];
else
    return NULL;
}

const QPointer<TCard> TDictionary::ValidCard(int aIndex) const
{
if( aIndex >= 0 && aIndex < iValidCards.size() )
    return iValidCards[aIndex];
else
    return NULL;
}

const QPointer<TCard> TDictionary::FindCardByUuid(QUuid aId) const
{
foreach( QPointer<TCard> card, iCards )
    if( card->Id() == aId )
        return card;
return NULL;
}

int TDictionary::FindCardIxByUuid(QUuid aId) const
{
for( int i=0; i< iCards.size(); i++ )
    {
    QPointer<TCard> card = iCards[i];
    if( card->Id() == aId )
        return i;
    }
return -1;
}

bool TDictionary::SetCardField( int aCardIx, int aFieldId, QString aField )
{
if( aCardIx < 0 || aCardIx >= iCards.size() )
    return false;
TCard* card = iCards[aCardIx];
bool oldValidity = card->IsValid();
card->SetField( aFieldId, aField );
bool newValidity = card->IsValid();
SetContentModified();
if( !oldValidity && newValidity )
    AddValidCards( aCardIx, 1 );
else if( oldValidity && !newValidity )
    {
    int removedNum = iValidCards.removeAll( card );
    if( !card->Repeated() )
        SetUnrepCardsNum( UnrepCardsNum() - removedNum );
    emit ValidCardsRemoved();
    }
emit CardModified( aCardIx );
return true;
}

bool TDictionary::SetCardFields( QMap<int, TField> aFields, int aQstnIx, QMap<int, int> aAnsrIxs, QList<int> aDeletedIxs )
{
if( aAnsrIxs.values().contains( aQstnIx ) )
    return false;
iFields = aFields;
iQuestionField = aQstnIx;
iAnswerFields = aAnsrIxs;
foreach( int deleted, aDeletedIxs )
    foreach( TCard* card, iCards )
        card->RemoveField( deleted );
UpdateValidCards();
emit FieldsChanged();
SetContentModified();
return true;
}

QPair<TDictionary::TFieldRole, int> TDictionary::FieldRole( int aIx )
{
if( aIx == iQuestionField )
    return qMakePair( EQuestionFieldRole, 0 );
else
    {
    int ansIx = iAnswerFields.key( aIx, -1 );
    if( ansIx > -1 )
        return qMakePair( EAnswerFieldRole, ansIx );
    else
        return qMakePair( ENoneFieldRole, -1 );
    }
}

int TDictionary::FieldIdToIx( QUuid aId ) const
{
QMapIterator<int, TField> f( iFields );
while( f.hasNext() )
    {
    f.next();
    QUuid fieldId = f.value().id;
    if( fieldId == aId )
        return f.key();
    }
return -1;
}

QString TDictionary::FormattedQuestionFieldName() const
{
if( iFields[iQuestionField].style == TField::EExampleFieldStyle )
    return QString("<i>") + iFields[iQuestionField].name + "</i>";
else
    return iFields[iQuestionField].name;
}

QString TDictionary::AnswerFieldsNames() const
{
QStringList list;
foreach( int i, iAnswerFields )
    if( iFields[i].style == TField::EExampleFieldStyle )
        list << QString("<i>") + iFields[i].name + "</i>";
    else
        list << iFields[i].name;
return list.join(", ");
}

void TDictionary::SwapQuestionAnswer()
{
int qstnIx = iQuestionField;
int ansrIx = iAnswerFields[0];
iQuestionField = ansrIx;
iAnswerFields[0] = qstnIx;
UpdateValidCards();
emit FieldsChanged();
}

void TDictionary::AddCard( TCard* aCard )
{
InsertCard( iCards.size(), aCard );
}

/// @todo Try to decrease the number of CardsInserted(int,int) signal emitting functions
void TDictionary::InsertCard(int aIndex, TCard* aCard)
{
iCards.insert( aIndex, aCard );
emit CardsInserted( aIndex, 1 );
}

void TDictionary::InsertCards(int aIndex, const QStringList& aStringList)
{
int cardsNum = aStringList.size();
for(int i=0; i < cardsNum; i++)
    iCards.insert( aIndex + i, new TCard( this, aStringList[i] ) );
emit CardsInserted( aIndex, cardsNum );
}

void TDictionary::InsertCards(int aIndex, int aNum)
{
for(int i=0; i < aNum; i++)
    iCards.insert( aIndex, new TCard( this ) );
emit CardsInserted( aIndex, aNum );
}

/// @todo Exchanging cards in XML-format
int TDictionary::InsertXMLCards(int aIndex, const QString& aString)
{
TDictParser parser( this );
//parser.SetInsertIndex( aIndex );
QXmlSimpleReader reader;
reader.setContentHandler( &parser );
reader.setErrorHandler( &parser );
QXmlInputSource xmlInputSource;
//qDebug() << aString;
xmlInputSource.setData( aString );
int cardsNumBefore = CardsNum();
if( reader.parse( xmlInputSource ) )
    SetContentModified( false );
int cardsNumAfter = CardsNum();
int cardsInserted = cardsNumAfter - cardsNumBefore;
emit CardsInserted( aIndex, cardsInserted );
return cardsInserted;
}

void TDictionary::RemoveCard(int aIndex)
{
RemoveCards( aIndex, 1 );
}

void TDictionary::RemoveCards(int aIndex, int aNum)
{
for( int i=0; i < aNum; i++ )
    {
    TCard* card = iCards.takeAt( aIndex );
    if( !card->Repeated() )
        SetUnrepCardsNum( UnrepCardsNum() - 1 );
    delete card;
    }
SetContentModified();
emit CardsRemoved( aIndex, aNum );
if( iValidCards.removeAll( NULL ) > 0 )
    emit ValidCardsRemoved();
}

void TDictionary::AddValidCards( int aIndex, int aNum )
{
for( int i = aIndex; i < aIndex + aNum; i++ )
    {
    TCard* card = iCards[i];
    if( card->IsValid() )
        {
        iValidCards << card;
        emit ValidCardAdded( i );
        if( !card->Repeated() )
            SetUnrepCardsNum( UnrepCardsNum() + 1 );
        }
    }
}

void TDictionary::UpdateValidCards()
{
iValidCards.clear();
int unrepCardsNum = 0;
foreach( TCard* card, iCards )
    if( card->IsValid() )
        {
        iValidCards << card;
        if( !card->Repeated() )
            unrepCardsNum++;
        }
SetUnrepCardsNum( unrepCardsNum );
}

void TDictionary::SetUnrepCardsNum( int aValue )
{
iUnrepCardsNum[ CurQstnAnsrPair() ] = aValue;
emit UnrepCardsChanged();
}

int TDictionary::RepetitionRecordsNum() const
{
int num = 0;
foreach( TCard* card, iCards )
    num += card->RepetitionRecordsNum();
return num;
}

QString TDictionary::XMLIndent( int aDepth = 0 )
{
return QString( KXMLIndentSize * aDepth, ' ' );
}

QString TDictionary::XMLEscapedStr( const QString aString )
{
QString res = aString;
res.replace("&", "&amp;");
res.replace("<", "&lt;");
res.replace(">", "&gt;");
res.replace("'", "&apos;");
res.replace("\"", "&quot;");
return res;
}
