#include "TCard.h"
#include <QStringList>
#include <QDateTime>
#include <QMapIterator>
#include <QRegExp>

#include "TSpacedRepetitionModel.h"

QChar TCard::iCommentChar( '#' );
QString TCard::iFieldSeparators( "\t&" );
QChar TCard::iTextDelimiter( '\"' );
QChar TCard::iKeywordStartChar( '[' );
QChar TCard::iKeywordEndChar( ']' );
QString TCard::iKeywordStartHtml( "<font color=\"blue\">" );
QString TCard::iKeywordEndHtml( "</font>" );
QString TCard::iExampleStartHtml( "<small>" );
QString TCard::iExampleEndHtml( "</small>" );


TCard::TCard( TDictionary* aDict ):
    iDictionary( aDict ), iIsCommented( false )
{
iId = QUuid::createUuid();
}

TCard::TCard( TDictionary* aDict, const QStringList& aElements ):
    iDictionary( aDict ), iIsCommented( false )
{
iId = QUuid::createUuid();
SetData( aElements );
}

TCard::TCard( TDictionary* aDict, const QString aPlainString, const TCSVImportData& aImportData ):
    iDictionary( aDict ), iIsCommented( false )
{
iId = QUuid::createUuid();
QString plainString = aPlainString.trimmed();
QChar comment = aImportData.iCommentChar;
if( !comment.isNull() && plainString.startsWith( comment ) )
    {
    SetCommented();
    plainString.remove(0, 1);   // first character
    plainString = plainString.trimmed();
    }
QString regExpStr;
switch( aImportData.iFieldSeparationMode )
    {
    case EFieldSeparatorAnyCharacter:
        regExpStr = QString("[") + aImportData.iFieldSeparators + "]";
        break;
    case EFieldSeparatorAnyCombination:
        regExpStr = QString("[") + aImportData.iFieldSeparators + "]+";
        break;
    case EFieldSeparatorExactString:
        regExpStr = aImportData.iFieldSeparators;
        break;
    }
    // CSV regexp:      ,(?=(?:[^"]*"[^"]*")*(?![^"]*"))
QChar delim = aImportData.iTextDelimiter;
regExpStr += QString( "(?=(?:[^" ) + delim + "]*" + delim + "[^" + delim + "]*" +
    delim + ")*(?![^" + delim + "]*" + delim + "))";
QRegExp regExp( regExpStr );
QStringList splittedLine = plainString.split( regExp, QString::KeepEmptyParts );
SetData( splittedLine, aImportData.iTextDelimiter, aImportData.iColsToImport );
}

TCard::TCard( TDictionary* aDict, const QString aPlainString ):
    iDictionary( aDict ), iIsCommented( false )
{
iId = QUuid::createUuid();
QString plainString = aPlainString.trimmed();
QRegExp regExp( QString("[") + iFieldSeparators + "]+" );
QStringList splittedLine = plainString.split( regExp, QString::KeepEmptyParts );
SetData( splittedLine );
}

bool TCard::SetField( int aIx, QString aValue )
{
QString newValue = aValue;
// Replace obsolete keyword character (%) with modern brackets []
if( iDictionary->Field( aIx ).style == TField::EExampleFieldStyle )
    {
    // zero %one %two three --> zero [one two] three
    newValue.replace( QRegExp("((?:%\\w+\\s*)+)(?=\\s|$)"), QString( iKeywordStartChar ) + "\\1" + iKeywordEndChar );
    newValue.remove( '%' );
    }
iFields[aIx] = newValue;
return newValue == aValue;
}

bool TCard::SetCommented( bool aValue )
{
if( iIsCommented != aValue )
    {
    iIsCommented = aValue;
    iDictionary->SetContentModified();
    return true;
    }
return false;
}

void TCard::ToggleCommented()
{
iIsCommented = !iIsCommented;
iDictionary->SetContentModified();
}

QString TCard::Question() const
{
return iFields.value( iDictionary->QuestionField() );
}

QString TCard::Answer() const
{
QStringList list;
foreach( int id, iDictionary->AnswerFields() )
    {
    QString fieldStr = iFields.value( id );
    if( !fieldStr.isEmpty() )
        list << fieldStr;
    }
return list.join("\n");
}

QString TCard::FormattedAnswer() const
{
QStringList list;
foreach( int ix, iDictionary->AnswerFields() )
    {
    QString fieldStr = iFields.value( ix );
    if( fieldStr.isEmpty() )
        continue;
    switch( iDictionary->Field( ix ).style )
        {
        case TField::EDefaultFieldStyle:
            fieldStr = "<b>" + fieldStr + "</b>";
            break;
        case TField::EExampleFieldStyle:
            fieldStr.replace( iKeywordStartChar, iKeywordStartHtml );   // [ --> <font color="blue">
            fieldStr.replace( iKeywordEndChar, iKeywordEndHtml );       // ] --> </font>
            fieldStr.replace( QRegExp( "\\b(\\w*" + QRegExp::escape(Question()) + "\\w*)\\b", Qt::CaseInsensitive ),
                iKeywordStartHtml + "\\1" + iKeywordEndHtml );    // Automatic detection of keywords
            fieldStr = iExampleStartHtml + fieldStr + iExampleEndHtml;  // <small> Example </small>
        }
    list << fieldStr;
    }
return list.join("<br>");
}

TQstnAnsrPair TCard::CurRepetitionIx() const
{
int qst = iDictionary->QuestionField();
int ans = iDictionary->AnswerFields()[0];
return qMakePair( qst, ans );
}

void TCard::SetCurRepetition( const TRepetition& aRep )
{
iRepetitionData[ CurRepetitionIx() ] = aRep;
iDictionary->SetStudyModified();
}

void TCard::SetRepetition( TQstnAnsrPair aIndex, const TRepetition& aRep )
{
iRepetitionData[ aIndex ] = aRep;
iDictionary->SetStudyModified();
}

void TCard::SetData( const QStringList aElements, QChar aTextDelimiter, int aColsToImport )
{

int colsNum;
if( aColsToImport > 0 )
    colsNum = qMin( aElements.size(), aColsToImport );
else
    colsNum = aElements.size(); // infinity
for( int i = 0; i < colsNum; i++ )
    {
    QString str = aElements[i];
    SetField( i, str.trimmed().remove( aTextDelimiter ) );
    }
}

QString TCard::ToXMLString( int aDepth ) const
{
QString res = TDictionary::XMLIndent( aDepth ) + "<c ";
if( iIsCommented )
    res += "comment=\"yes\" ";
QString content;
QMapIterator<int, QString> i( iFields );
int fieldIx = 0;
while( i.hasNext() )
    {
    i.next();
    QString fieldValue = TDictionary::XMLEscapedStr( i.value() );
    if( fieldValue.isEmpty() )
        continue;
    QUuid cardFieldId = iDictionary->Field( i.key() ).id;
    QUuid dictFieldId = iDictionary->Field( fieldIx ).id;
    content += TDictionary::XMLIndent( aDepth + 1 ) + "<f";
    if( cardFieldId != dictFieldId )
        content += QString(" id=\"%1\"").arg( cardFieldId );
    content += QString("> %1 </f>\n").arg( fieldValue );
    fieldIx++;
    }
if( !content.isEmpty() )
    {
    res += QString("id=\"%1\">\n").arg( iId ) + content;
    res += TDictionary::XMLIndent( aDepth ) + "</c>\n";
    }
else
    res += "/>\n";
return res;
}

QString TCard::ToXMLStudy( int aDepth ) const
{
if( iRepetitionData.isEmpty() )
    return QString();
QString res = TDictionary::XMLIndent( aDepth ) + QString("<c id=\"%1\">\n").arg( iId );
QMapIterator<TQstnAnsrPair, TRepetition> i( iRepetitionData );
while( i.hasNext() )
    {
    i.next();
    TQstnAnsrPair qstnAnsrPair = i.key();
    QUuid qstFieldId = iDictionary->Field( qstnAnsrPair.first ).id;
    QUuid ansFieldId = iDictionary->Field( qstnAnsrPair.second ).id;
    QUuid dictField0Id = iDictionary->Field( 0 ).id;
    QUuid dictField1Id = iDictionary->Field( 1 ).id;
    TRepetition rep = i.value();
    res += TDictionary::XMLIndent( aDepth + 1 ) + "<s";
    bool reversed = false;
    bool fieldIdsAdded = false;
    if( qstFieldId != dictField0Id )
        {
        if( qstFieldId == dictField1Id )    // reversed
            reversed = true;
        else
            {
            res += QString(" qst=\"%1\"").arg( qstFieldId );
            fieldIdsAdded = true;
            }
        }
    if( ansFieldId != dictField1Id )
        {
        if( ansFieldId == dictField0Id )    // reversed
            reversed = true;
        else
            {
            res += QString(" ans=\"%1\"").arg( ansFieldId );
            fieldIdsAdded = true;
            }
        }
    if( reversed )
        res += " rev=\"yes\"";
    if( fieldIdsAdded )
        res += QString("\n") + TDictionary::XMLIndent( aDepth + 1 );
    else
        res += " ";
    res += QString("last=\"%1\" int=\"%2\" grd=\"%3\" eas=\"%4\" ")
        .arg( rep.lastRepetition.toString( Qt::ISODate ) )
        .arg( rep.interval )
        .arg( rep.grade )
        .arg( rep.easiness );
    if( rep.afterCards > 0 )
        res += QString("cards=\"%1\" ").arg( rep.afterCards );
    res += "/>\n";
    }
res += TDictionary::XMLIndent( aDepth ) + "</c>\n";
return res;
}

/// @todo Remove this function, use the complete ToCSVString( TCSVExportData&) instead
QString TCard::ToCSVString( const QChar aSeparator ) const
{
QString sepStr;
if( aSeparator.isNull() )
    sepStr = iFieldSeparators + " ";
else 
    sepStr = QString("\t%1 ").arg( aSeparator );
QStringList list;
foreach( QString field, iFields )
    {
    if( !field.isEmpty() )
        list << field;
    }
return list.join( sepStr );
}

QString TCard::ToCSVString( const TCSVExportData& aExportData )
{
QString csv;
if( iIsCommented && aExportData.iCommentChar.isNull() )
    return QString();
QRegExp fieldSepRegExp;
if( aExportData.iQuoteAllFields )
    fieldSepRegExp = QRegExp(".");  // Any character
else
    fieldSepRegExp = QRegExp( QString("[") + aExportData.iFieldSeparators + "]" );   // Any of the separators
QMap<int, QString> fields = iFields;
QStringList sortedFields;
sortedFields << fields.take( iDictionary->QuestionField() ); // Question
foreach( int ansrIx, iDictionary->AnswerFields() )      // Answers
    sortedFields << fields.take( ansrIx );
foreach( QString str, fields )                          // The rest
    sortedFields << str;
for( int i = 0; i < sortedFields.size(); i++ )
    sortedFields[i] = DelimitText( sortedFields[i], fieldSepRegExp, aExportData.iTextDelimiter );
QStringList list;
foreach( int col, aExportData.iUsedCols )
    {
    if( !sortedFields.value(col).isEmpty() )  // QList::value(int) checks for list bounds, returns empty string if out of bounds
        list << sortedFields.value(col);
    }
csv = list.join( aExportData.iFieldSeparators );
    
if( iIsCommented )
        csv.insert( 0, aExportData.iCommentChar );  // Insert the comment character
return csv;
}

QString TCard::DelimitText( const QString& aText, const QRegExp& aFieldSepRegExp, const QChar aTextDelimiter ) const
{
if( aText.isEmpty() || aTextDelimiter.isNull() )
    return aText;
QString res( aText );
if( res.contains( aFieldSepRegExp ) )
    res = aTextDelimiter + res + aTextDelimiter;
return res;
}
