#ifndef TCARD_H
#define TCARD_H

#include <QObject>
#include <QString>
#include <QDateTime>
#include <QStringList>
#include <QMap>
#include <QPair>
#include <QUuid>

#include "CSVData.h"

class TDictionary;

/// Question-answer pair    
typedef QPair<int, int> TQstnAnsrPair;


class TCard: public QObject
{
public:

/** A card can be scheduled for the next repetition either with an interval in days (#interval), or
  * after certain amount of displayed cards (#afterCards). In the latter case, #interval must == 0.
  *
  * If #lastRepetition is null date, the card is unrepeated. A null date is date that has been initialized with QDateTime(),
  * and its isNull() returns true. Use Repeated() function to get this property. @see TCard::Repeated()
  */
struct TRepetition
    {
    TRepetition(): interval ( -1 ), grade( -1 ), easiness( 0 ), afterCards( 0 )
        {
        lastRepetition = QDateTime();   // null date, unrepeated card
        }
    void Set(QDateTime aLastRepetition, float aInterval, int aGrade, float aEasiness, int aAfterCards = 0 )
        {
        lastRepetition = aLastRepetition;
        interval = aInterval;
        grade = aGrade;
        easiness = aEasiness;
        afterCards = aAfterCards;
        }
    bool Repeated() const { return !lastRepetition.isNull(); }
        
    QDateTime lastRepetition;
    float interval;    /// Repetition interval in days, can be fractional.
    int grade;
    float easiness;
    int afterCards;    /// Repetition interval in cards. If #afterCards > 0, #interval == 0.
    };

// Constructors and destructor
//TCard();
TCard( TDictionary* aDict );
TCard( TDictionary* aDict, const QStringList& aElements );
TCard( TDictionary* aDict, const QString aPlainString );
TCard( TDictionary* aDict, const QString aPlainString, const TCSVImportData& aImportData );

// Getters and setters
QUuid Id() const { return iId; }
QString Field( int aIndex ) const { return iFields.value( aIndex ); }
int FieldsNum() const { return iFields.size(); }
QString Question() const;
QString Answer() const;
QString FormattedAnswer() const;

TQstnAnsrPair CurRepetitionIx() const;

/** Convinience function. Returns reference to the current repetition record.
  * If there is no yet given repetition, the underlying QPair::operator[] creates a new one. */
TRepetition& CurRepetition() { return iRepetitionData[ CurRepetitionIx() ]; }
const TRepetition CurRepetition() const { return iRepetitionData[ CurRepetitionIx() ]; }

int RepetitionRecordsNum() const { return iRepetitionData.size(); }

// Set of convenience functions to access fields of the current repetition record
const QDateTime LastRepetition() const { return CurRepetition().lastRepetition; }
float Interval() const { return CurRepetition().interval; }
int Grade() const { return CurRepetition().grade; }
float Easiness() const { return CurRepetition().easiness; }
int AfterCards() const { return CurRepetition().afterCards; }
bool Repeated() const
    {
    if( !iRepetitionData.contains( CurRepetitionIx() ) )
        return false;
    else
        return CurRepetition().Repeated();
    }
    
bool Commented() const { return iIsCommented; }    
    
QString ToXMLString( int aDepth ) const;
QString ToXMLStudy( int aDepth ) const;
QString ToCSVString( const QChar aSeparator = '\0' ) const;
QString ToCSVString( const TCSVExportData& aExportData );

void SetId( QUuid aId ) { iId = aId; }

/// @return true, if the field content was modified before assigning to the card
bool SetField( int aIx, QString aValue );

void RemoveField( int aIx ) { iFields.remove( aIx ); }
void SetData( const QStringList aElements, QChar aTextDelimiter = iTextDelimiter, int aColsToImport = 3 );

void SetRepetition( TQstnAnsrPair aIndex, const TRepetition& aRep );

/** Convinience function. Sets the current repetition record.
  * If there is no yet given repetition, the underlying QPair::operator[] creates a new one. */
void SetCurRepetition( const TRepetition& aRep );

// Set of convenience functions to access fields of the current repetition record
void SetLastRepetition( const QDateTime aValue ) { CurRepetition().lastRepetition = aValue; }
void SetInterval( const float aValue ) { CurRepetition().interval = aValue; }
void SetGrade( const int aValue ) { CurRepetition().grade = aValue; }
void SetEasiness( const float aValue ) { CurRepetition().easiness = aValue; }
void SetAfterCards( const int aValue ) { CurRepetition().afterCards = aValue; }

/** If the commented state changes, sets the dictionary to the content modified state.
  * @return true, if the Commented state changed
  */
bool SetCommented( bool aValue = true );

/// If the commented state changes, sets the dictionary to the content modified state.
void ToggleCommented();

/** The card is valid. It means that the card is not NULL, its question is not empty and it doesn't contain comment.
 */
bool IsValid() const { return !Question().isEmpty() && !Answer().isEmpty() && !Commented(); }

public:

static QChar iCommentChar;
static QString iFieldSeparators;
static QChar iTextDelimiter;
static QChar iKeywordStartChar;
static QChar iKeywordEndChar;
static QString iKeywordStartHtml;
static QString iKeywordEndHtml;
static QString iExampleStartHtml;
static QString iExampleEndHtml;

protected:
QString DelimitText( const QString& aText, const QRegExp& aFieldSepRegExp, const QChar aTextDelimiter ) const;

public:
/** A set of repetition records.
  * A repetition record is specific for each question-answer pair. The first parameter in QPair is question, the second is answer.
  *
  * If there is no repetition record for a given question-answer pair, that means the card has not been repeated yet.
  * You can check the presence of specific repetition record with QMap::contains().
  * Remember that QMap::operator[] always inserts a new record, if it doesn't exist yet.
  */
QMap<TQstnAnsrPair, TRepetition> iRepetitionData;

protected:
QUuid iId;
QMap<int, QString> iFields;
TDictionary* iDictionary;
bool iIsCommented;
};

#endif 
