#include "TCSVExportDialog.h"

#include <QLabel>
#include <QComboBox>
#include <QToolButton>
#include <QPushButton>
#include <QCheckBox>
#include <QSpinBox>
#include <QLineEdit>
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QGridLayout>
#include <QMessageBox>
#include <QGroupBox>
#include <QAction>
#include <QMenu>
#include <QTextCodec>
#include <QTextEdit>
#include <QList>
#include <QTextStream>

#include "TDictionary.h"
// #include "CSVData.h"

QChar TCSVExportDialog::iSpaceChar( 0x2423 );
QChar TCSVExportDialog::iTabChar( 0x21A6 );
QString TCSVExportDialog::iExtendedTab( QString(QChar(0x21A6)) + "\t" );

TCSVExportDialog::TCSVExportDialog( QWidget* aParent, const TDictionary* aDict ):
    QDialog( aParent ), iDictionary( aDict )
{
setWindowTitle(tr("Export to CSV"));
resize( 600, 550 );

// Output group box
QLabel* characterSetLabel = new QLabel(tr("C&haracter set:"));
iCharacterSetCombo = new QComboBox;
characterSetLabel->setBuddy( iCharacterSetCombo );
iCharacterSetCombo->setEditable(false);
QList<QByteArray> codecNames = QTextCodec::availableCodecs();
qSort( codecNames );
int systemIx = codecNames.indexOf( "System" );
codecNames.move( systemIx, 0 );
for( int i=0; i<codecNames.size(); i++)
    iCharacterSetCombo->addItem( codecNames[i], QTextCodec::codecForName( codecNames[i] )->mibEnum() );
int utf8Ix = codecNames.indexOf( "UTF-8" );
iCharacterSetCombo->setCurrentIndex( utf8Ix );        

QLabel* fromRowLabel = new QLabel(tr("From &row:"));
iFromRowSpin = new QSpinBox();
fromRowLabel->setBuddy( iFromRowSpin );
iFromRowSpin->setValue( 1 );
iFromRowSpin->setMinimum( 1 );
iFromRowSpin->setMaximum( 10000000 );

QLabel* usedColsLabel = new QLabel(tr("Used co&lumns:"));
iUsedColsEdit = new QLineEdit("123");
usedColsLabel->setBuddy( iUsedColsEdit );
QRegExp columnsRegExp("[1-3]+");
QValidator *columnsValidator = new QRegExpValidator( columnsRegExp, this );
iUsedColsEdit->setValidator( columnsValidator );

iCommentsCB = new QCheckBox(tr("Export &comments"));
iCommentsCB->setChecked( true );

QGridLayout* outputLayout = new QGridLayout;
outputLayout->addWidget( characterSetLabel, 0, 0 );
outputLayout->addWidget( iCharacterSetCombo, 0, 1 );
outputLayout->addWidget( fromRowLabel, 1, 0);
outputLayout->addWidget( iFromRowSpin, 1, 1);
outputLayout->addWidget( usedColsLabel, 2, 0);
outputLayout->addWidget( iUsedColsEdit, 2, 1);
outputLayout->addWidget( iCommentsCB, 3, 0, 1, 2);
outputLayout->setColumnStretch( 1, 1 );
QGroupBox* outputGroup = new QGroupBox(tr("Output"));
outputGroup->setLayout( outputLayout );

// Separators group box

QLabel* separatorsLabel = new QLabel(tr("Field &separators:"));
iSeparatorsEdit = new QLineEdit( QString( iTabChar ) + "&" + iSpaceChar );
separatorsLabel->setBuddy( iSeparatorsEdit );
iAddSeparatorButton = new QToolButton();
iAddSeparatorButton->setText(tr("&Add..."));

QStringList regexpDescrList, regexpDataList;
regexpDescrList << tr("Comma (,)");
regexpDataList << QString( "," );
regexpDescrList << tr("Semicolon (;)");
regexpDataList << QString( ";" );
regexpDescrList << tr("Tab (%1)").arg(iTabChar);
regexpDataList << QString( iTabChar );
regexpDescrList << tr("Space (%1)").arg(iSpaceChar);
regexpDataList << QString( iSpaceChar );
regexpDescrList << tr("Ampersand (%1)").arg("&&");
regexpDataList << QString( "&" );
regexpDescrList << tr("Hash (#)");
regexpDataList << QString( "#" );
iSeparatorsMenu = new QMenu();
QListIterator< QString > descrIt( regexpDescrList );
QListIterator< QString > dataIt( regexpDataList );
while( descrIt.hasNext() )
    {
    QAction* action = new QAction( descrIt.next(), this );
    action->setData( dataIt.next() );
    iSeparatorsMenu->addAction( action );
    connect( action, SIGNAL(triggered()), this, SLOT(AddSeparatorToEdit()) );
    }
iAddSeparatorButton->setMenu( iSeparatorsMenu );
iAddSeparatorButton->setPopupMode( QToolButton::InstantPopup );

QRegExp oneCharRegExp("\\S{1}");
QValidator *oneCharValidator = new QRegExpValidator(oneCharRegExp, this);

iTextDelimiterCB = new QCheckBox(tr("&Text delimiter:"));
iTextDelimiterCB->setChecked( true );
iTextDelimiterCombo = new QComboBox;
iTextDelimiterCombo->setEditable(true);
iTextDelimiterCombo->addItem( "\"" );
iTextDelimiterCombo->addItem( "\'" );
iTextDelimiterCombo->setInsertPolicy( QComboBox::NoInsert );
iTextDelimiterCombo->setValidator( oneCharValidator );

iQuoteAllFieldsCB = new QCheckBox(tr("&Quote all fields"));

iCommentCharacterLabel = new QLabel("Co&mment character");
iCommentCharacterCombo = new QComboBox;
iCommentCharacterLabel->setBuddy( iCommentCharacterCombo );
iCommentCharacterCombo->setEditable(true);
iCommentCharacterCombo->addItem( "#" );
iCommentCharacterCombo->addItem( ";" );
iCommentCharacterCombo->setInsertPolicy( QComboBox::NoInsert );
iCommentCharacterCombo->setValidator( oneCharValidator );

QGridLayout* separatorsLayout = new QGridLayout;
separatorsLayout->addWidget( separatorsLabel, 0, 0 );
separatorsLayout->addWidget( iSeparatorsEdit, 0, 1 );
separatorsLayout->addWidget( iAddSeparatorButton, 0, 2 );
separatorsLayout->addWidget( iTextDelimiterCB, 1, 0 );
separatorsLayout->addWidget( iTextDelimiterCombo, 1, 1, 1, 2 );
separatorsLayout->addWidget( iQuoteAllFieldsCB, 2, 0, 1, 3 );
separatorsLayout->addWidget( iCommentCharacterLabel, 3, 0 );
separatorsLayout->addWidget( iCommentCharacterCombo, 3, 1, 1, 2 );
separatorsLayout->setColumnStretch( 1, 1 );
QGroupBox* separatorsGroup = new QGroupBox(tr("Separators"));
separatorsGroup->setLayout( separatorsLayout );

// Top layout

QHBoxLayout* topLayout = new QHBoxLayout;
topLayout->addWidget( outputGroup );
topLayout->addWidget( separatorsGroup );

// Preview

QLabel* previewLabel = new QLabel(tr("&Preview:"));
iCSVPreview = new QTextEdit;
previewLabel->setBuddy( iCSVPreview );
iCSVPreview->setReadOnly( true );
iCSVPreview->setLineWrapMode( QTextEdit::NoWrap );
iCSVPreview->setFontPointSize( 10 );

// OK - Cancel layout

iShowInvisibleCharsCB = new QCheckBox(tr("Show &invisible characters"));
iShowInvisibleCharsCB->setChecked( true );

iOkButton = new QPushButton(tr("&OK"));
iOkButton->setDefault(true);  // Enter

iCancelButton = new QPushButton(tr("Ca&ncel"));    // Esc

QHBoxLayout* okCancelLayout = new QHBoxLayout;
okCancelLayout->addWidget( iShowInvisibleCharsCB );
okCancelLayout->addStretch();
okCancelLayout->addWidget( iOkButton );
okCancelLayout->addWidget( iCancelButton );

UpdatePreview();

// Main layout

QVBoxLayout* mainLayout = new QVBoxLayout;
mainLayout->addLayout( topLayout );
mainLayout->addWidget( previewLabel );
mainLayout->addWidget( iCSVPreview );
mainLayout->addLayout( okCancelLayout );
setLayout( mainLayout );

connect( iTextDelimiterCB, SIGNAL(stateChanged(int)), this, SLOT(UpdateTextDelimiterCombo()) );
connect( iCommentsCB, SIGNAL(stateChanged(int)), this, SLOT(UpdateCommentCharacterControls()) );
connect( iShowInvisibleCharsCB, SIGNAL(stateChanged(int)), this, SLOT(UpdateCharVisibility()) );
connect( iOkButton, SIGNAL(clicked()), this, SLOT(accept()) );
connect( iCancelButton, SIGNAL(clicked()), this, SLOT(reject()) );

connect( iFromRowSpin, SIGNAL(valueChanged(int)), this, SLOT(UpdatePreview()) );
connect( iUsedColsEdit, SIGNAL(textChanged(QString)), this, SLOT(UpdatePreview()) );
connect( iSeparatorsEdit, SIGNAL(textChanged(QString)), this, SLOT(UpdatePreview()) );
connect( iTextDelimiterCB, SIGNAL(stateChanged(int)), this, SLOT(UpdatePreview()) );
connect( iTextDelimiterCB, SIGNAL(stateChanged(int)), this, SLOT(UpdateQuoteAllFieldsCB()) );
connect( iQuoteAllFieldsCB, SIGNAL(stateChanged(int)), this, SLOT(UpdatePreview()) );
connect( iTextDelimiterCombo, SIGNAL(editTextChanged(QString)), this, SLOT(UpdatePreview()) );
connect( iCommentsCB, SIGNAL(stateChanged(int)), this, SLOT(UpdatePreview()) );
connect( iCommentCharacterCombo, SIGNAL(editTextChanged(QString)), this, SLOT(UpdatePreview()) );

}

void TCSVExportDialog::AddSeparatorToEdit()
{
QAction *action = qobject_cast<QAction*>( sender() );
if( action )
    iSeparatorsEdit->insert( action->data().toString() );
}

void TCSVExportDialog::UpdateTextDelimiterCombo()
{
iTextDelimiterCombo->setEnabled( iTextDelimiterCB->isChecked() );
}

void TCSVExportDialog::UpdateCommentCharacterControls()
{
bool isCommentsEnabled = iCommentsCB->isChecked();
iCommentCharacterLabel->setEnabled( isCommentsEnabled );
iCommentCharacterCombo->setEnabled( isCommentsEnabled );
}

void TCSVExportDialog::UpdateQuoteAllFieldsCB()
{
iQuoteAllFieldsCB->setEnabled( iTextDelimiterCB->isChecked() );
}

QTextCodec* TCSVExportDialog::TextCodec()
{
return QTextCodec::codecForMib( iCharacterSetCombo->itemData( iCharacterSetCombo->currentIndex() ).toInt() );
}

/// Convert invisible chars like space and tab to visible equivalents or vice versa.
QString TCSVExportDialog::SetCharVisibility( const QString& aInput, bool aVisible ) const
{
QString output( aInput );
if( aVisible )
    {
    output.replace( ' ', iSpaceChar );
    output.replace( '\t', iExtendedTab );
    }
else
    {
    output.replace( iSpaceChar, ' ' );
    output.replace( iExtendedTab, "\t" );
    output.replace( iTabChar, '\t' );
    }
return output;
}

void TCSVExportDialog::UpdatePreview()
{
QString usedColsStr( iUsedColsEdit->text() );
QList<int> usedCols;
for(int k=0; k<usedColsStr.length(); k++)
    {
    int col = usedColsStr[k].digitValue();
    if( col > 0 && col <= iDictionary->FieldsNum() )
        usedCols << col-1;
    }
TCSVExportData exportData;
exportData.iFromRow = iFromRowSpin->value();
exportData.iUsedCols = usedCols;
exportData.iFieldSeparators = SetCharVisibility( iSeparatorsEdit->text(), false );
exportData.iTextDelimiter = iTextDelimiterCB->isChecked()? iTextDelimiterCombo->currentText()[0]: QChar(0);
exportData.iQuoteAllFields = iQuoteAllFieldsCB->isChecked();
exportData.iCommentChar = iCommentsCB->isChecked()? iCommentCharacterCombo->currentText()[0]: QChar(0);
QString csvString = iDictionary->ExportToCSVString( exportData );
iCSVPreview->setPlainText( SetCharVisibility( csvString, iShowInvisibleCharsCB->isChecked() ) );
}

void TCSVExportDialog::SaveCSVToFile( const QString& aFilePath )
{
QFile file( aFilePath );
if( !file.open( QIODevice::WriteOnly | QFile::Text ) ) // \r\n --> \n
    {
    QMessageBox::warning( this, tr("Error - Fresh Memory"), tr("Cannot save to file:\n %1.").arg(aFilePath) );
    return;
    }
QTextStream outStream( &file );
outStream.setCodec( TextCodec() );
outStream << SetCharVisibility( iCSVPreview->toPlainText(), false );
}

void TCSVExportDialog::UpdateCharVisibility()
{
iCSVPreview->setPlainText(
    SetCharVisibility( iCSVPreview->toPlainText(), iShowInvisibleCharsCB->isChecked() )
    );
}
