#ifndef MTESTMODEL_H
#define MTESTMODEL_H

#include <QObject>
#include <QList>
#include <QLinkedList>
#include <QPointer>
#include <QtDebug>

#include "TDictionary.h"

class TCard;
class TAppModel;

/** 
 * @brief The default model for all tests.
 *
 * At the beginning of a test, the model generates a fresh pack of valid cards: #iFreshPack. In course of the test,
 * cards one-by-one are randomly taken from the fresh pack and moved to the history pack #iCardHistory.
 * The part of history, which belong to the current pack (recently generated new pack), is separated from older
 * cards with property #iHistoryCurPackStart.
 * The current card can be obtained with CurCard(), it is taken from the history. When the fresh pack is finished
 * and the next card required, new fresh pack is generated.
 *
 * All card containers, the fresh pack and history, do not own the cards.
 * The cards are always owned by the dictionary. In order to keep track of removed cards, the card containers
 * are implemented using QPointer\<TCard\>.
 *
 * All used cards over all pack iterations are saved to the history #iCardHistory.
 * The user can travel back and forth along the history with GoBack() and GoForward(). In this case #iCurCardNum
 * increases or decreases. Functions CanGoBack() and CanGoForward() can be used to check if it is possible to
 * move in the corresponding direction.
 */
 
/// @todo Implement filtering the valid cards as a proxy model: QSortFilterProxyModel

class MTestModel: public QObject
{
Q_OBJECT

protected:
typedef QList< QPointer<TCard> > TPersistentCardList;

public:
static const int KNoCurrentCard = -1;

public: //Methods

virtual void Start();

/** This method doesn't emit Stopped() signal.
 * Therefore, test model is not destroyed, when restarted.
 */
void Restart();

/** Returns the current card from the history.
 * If the index is out of range, the function returns NULL.
 */
TCard* CurCard();

/** Size of the current card pack. It's value depends on the training tool.
  * For the word drill, it's always equal to number of all valid cards.
  */
virtual int CurPackSize() const { return iFreshPack.size() + iCardHistory.size() - iHistoryCurPackStart; }

/// Number of cards used in the current session. Includes both history and fresh cards.
virtual int SessionCardsNum() const { return iFreshPack.size() + iCardHistory.size(); }

/// Number of all valid cards
int ValidCardsNum() const {return iDictionary->ValidCardsNum();}
int CurCardNum() const { return iCurCardNum; }  /// @todo Change to CurCardIx
int CurCardIxInCurPack() const { return iCurCardNum - iHistoryCurPackStart; }
int HistorySize() const {return iCardHistory.size(); }
bool IsStarted() const { return iIsStarted; } 
TDictionary* Dictionary() const { return iDictionary; }
QString StatusMessage() const { return iStatusMessage; }

bool CanGoBack();
bool CanGoForward();

signals:
void CurCardNumChanged();
void CurPackChanged();
void Stopped();
void ShowStatusMessage();

public slots:
virtual void Stop();

/** Picks a random card. Removes the selected card from the fresh pack and adds it to the history.
 * Thus, the new card is the last entry in the history.
 * This function guarantees that the new card's question will be different from the previous one, unless there is no choice.
 *
 * Updates #iCurCardNum.
 */
virtual void PickNextCard();

/** Go back along the history line.
 * @return true, if the transition was successful
 */
bool GoBack();

/** Go forward along the history line.
 * @return true, if the transition was successful
 */
bool GoForward();

protected slots:
void CheckCards();
void AddCard( int aIndex );

protected:	// Methods
// Constructor
MTestModel(TDictionary* aDictionary);

/** Generates the fresh card pack from only valid cards of the dictionary.
 * Updates #iTestCardsNum.
 *
 * The added cards may be further filtered out due to their properties.
 */
virtual void GenerateFreshPack();

/** Display the next card.
 * Updates the current card number #iCurCardNum and adds the card to the end of history.
 * 
 * Notifies the view that the current card has changed and it needs to be updated.
 */
virtual void DisplayNextCard( TCard* aCard );

virtual void AddNewCards();

/// Returns if the card is valid for this test
virtual bool IsValidCard( const TCard* aCard ) const;

protected:	// Data

// Data source
/** The dictionary pointer. Not own. */
TDictionary* iDictionary;

// Test-related data
bool iIsStarted;

/** Fresh cards. The cards that were not shown yet (in the current iteration). Not own.*/
TPersistentCardList iFreshPack;

/// List of all shown cards. Not own.
TPersistentCardList iCardHistory;

/** Index of the current card in the history list, base=0.
 * Can increase and decrease when moving along the history line.
 */
int iCurCardNum;

/// Previous question. Used in PickNextCard() to check for the same subsequent questions.
QString iPrevQuestion;

/// The start index of the current pack in the history
int iHistoryCurPackStart;

/** A status message for displaying at the view.
  * First, assign a new string to @a iStatusMessage, then emit signal ShowStatusMessage().
  */
QString iStatusMessage;
};

#endif
