# Based in wordpresslib.py WordPress xml-rpc client library
# by Michele Ferretti

__author__ = "Daniel Martin Yerga <dyerga@gmail.com>"
__version__ = "$Revision: 1.3 $"
__date__ = "$Date: 2009/12/21 $"
__copyright__ = "Copyright (c) 2009 Daniel Martin Yerga"
__license__ = "LGPL"

import exceptions
import re
import os
import xmlrpclib
import time

def create_wordpress_client(url, user, password):
    """Quick helper routine used to create the wordpress
    client"""
    # prepare client object
    try:
        wp = WordPressClient(url, user, password)
    except:
        return None

    # select blog id
    wp.selectBlog(0)

    return wp

class WordPressException(exceptions.Exception):
    """Custom exception for WordPress client operations
    """
    def __init__(self, obj):
        if isinstance(obj, xmlrpclib.Fault):
            self.id = obj.faultCode
            self.message = obj.faultString
        else:
            self.id = 0
            self.message = obj

    def __str__(self):
        return '<%s %d: \'%s\'>' % (self.__class__.__name__, self.id, self.message)

class WordPressCategory:
    """Represents category item
    """
    def __init__(self):
        self.id = 0
        self.name = ''
        self.isPrimary = False

class MetaWeblogCategory:
    def __init__(self):
        self.id = 0
        self.parentId = ''
        self.description = ''
        self.name = ''
        self.htmlurl = ''
        self.rssurl = ''

class WordPressPost:
    """Represents post item
    """
    def __init__(self):
        self.id = 0
        self.title = ''
        self.date = None
        self.permaLink = ''
        self.description = ''
        self.textMore = ''
        self.excerpt = ''
        self.link = ''
        self.categories = []
        self.user = ''
        self.trackbacks = ''
        self.allowPings	= False
        self.allowComments = False
        self.password = ''
        self.tags = ''
        self.name = ''

class WordPressClient:
    """Client for connect to WordPress XML-RPC interface
    """

    def __init__(self, url, user, password):
        self.url = url
        self.user = user
        self.password = password
        self.blogId = 0
        self.categories = None
        self._server = xmlrpclib.ServerProxy(self.url)

    def _filterPost(self, post):
        """Transform post struct in WordPressPost instance
        """
        postObj = WordPressPost()
        postObj.permaLink 		= post['permaLink']
        postObj.description 	= post['description']
        postObj.title 			= post['title']
        postObj.excerpt 		= post['mt_excerpt']
        postObj.user 			= post['userid']
        postObj.date 			= time.strptime(str(post['dateCreated']), \
                                                "%Y%m%dT%H:%M:%S")
        postObj.link 			= post['link']
        postObj.textMore 		= post['mt_text_more']
        postObj.allowComments 	= post['mt_allow_comments']
        postObj.id 				= int(post['postid'])
        postObj.categories 		= post['categories']
        postObj.allowPings 		= post['mt_allow_pings']
        postObj.password 		= post['wp_password']
        postObj.tags            = post['mt_keywords']
        postObj.status          = post['post_status']
        postObj.name            = post['wp_slug']

        return postObj

    def _filterCategory(self, cat):
        """Transform category struct in WordPressCategory instance
        """
        catObj = WordPressCategory()
        catObj.id 			= int(cat['categoryId'])
        catObj.name 		= cat['categoryName']
        if cat.has_key('isPrimary'):
            catObj.isPrimary 	= cat['isPrimary']
        return catObj

    def selectBlog(self, blogId):
        self.blogId = blogId


    ###################################
    ##                               ##
    ##     Posting functions         ##
    ##                               ##
    ###################################

    def newPost(self, post):
        """Insert new post
        """
        blogContent = {
            'title' : post.title,
            'description' : post.description,
            'mt_tb_ping_urls' : post.trackbacks,
            'mt_allow_comments' : post.allowComments,
            'mt_text_more' : post.textMore,
            'mt_excerpt' : post.excerpt,
            'mt_allow_pings' : post.allowPings,
            'wp_password' : post.password,
            'mt_keywords' : post.tags,
            'post_status' : post.status,
            'wp_slug': post.name
        }

        if post.date:
            blogContent['dateCreated'] = xmlrpclib.DateTime(post.date)

        # add categories
        i = 0
        categories = []
        for cat in post.categories:
            if i == 0:
                categories.append({'categoryId' : cat, 'isPrimary' : 1})
            else:
                categories.append({'categoryId' : cat, 'isPrimary' : 0})
            i += 1

        # insert new post
        idNewPost = int(self._server.metaWeblog.newPost(self.blogId, self.user, \
                                                self.password, blogContent, 0))

        # set categories for new post
        self.setPostCategories(idNewPost, categories)

        return idNewPost

    def setPostCategories(self, postId, categories):
        """Set post's categories
        """
        self._server.mt.setPostCategories(postId, self.user, self.password, categories)

    ###################################
    ##                               ##
    ##     Categories functions      ##
    ##                               ##
    ###################################

    def getCategories(self):
        try:
            categories = self._server.metaWeblog.getCategories(self.blogId, \
                                                    self.user, self.password)
            for cat in categories:
                catObj = MetaWeblogCategory()
                catObj.id = cat['categoryId']
                catObj.parentId = cat['parentId']
                catObj.description = cat['description']
                catObj.name = cat['categoryName']
                catObj.htmlurl = cat['htmlUrl']
                catObj.rssurl = cat['rssUrl']
                yield catObj
        except xmlrpclib.Fault, fault:
            raise WordPressException(fault)

    def getCategoryList(self):
        """Get blog's categories list
        """
        try:
            if not self.categories:
                self.categories = []
                categories = self._server.mt.getCategoryList(self.blogId,
                                                    self.user, self.password)
                for cat in categories:
                    self.categories.append(self._filterCategory(cat))
            return self.categories
        except xmlrpclib.Fault, fault:
            raise WordPressException(fault)

    def getCategoryIdFromName(self, name):
        """Get category id from category name
        """
        for c in self.getCategoryList():
            if c.name == name:
                return c.id

    ###################################
    ##                               ##
    ##      Other functions          ##
    ##                               ##
    ###################################


    def newMediaObject(self, mediaFileName):
        """Add new media object (image, movie, etc...)
        """
        try:
            f = file(mediaFileName, 'rb')
            mediaBits = f.read()
            f.close()

            mediaStruct = {
                'name' : os.path.basename(mediaFileName),
                'bits' : xmlrpclib.Binary(mediaBits)
            }

            result = self._server.metaWeblog.newMediaObject(self.blogId, \
                                        self.user, self.password, mediaStruct)
            return result['url']

        except xmlrpclib.Fault, fault:
            raise WordPressException(fault)
