/****************************************************************************
**
** Copyright (C) 2007-2007 Trolltech ASA. All rights reserved.
**
** This file is part of the Graphics Dojo project on Trolltech Labs.
**
** This file may be used under the terms of the GNU General Public
** License version 2.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of
** this file.  Please review the following information to ensure GNU
** General Public Licensing requirements will be met:
** http://www.trolltech.com/products/qt/opensource.html
**
** If you are unsure which license is appropriate for your use, please
** review the following information:
** http://www.trolltech.com/products/qt/licensing.html or contact the
** sales department at sales@trolltech.com.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/
#include "CDesktop.h"
#include "CDlgSetup.h"
#include <QSvgRenderer>
#include <QPainter>
#include <QPaintEvent>
#include <QPen>
#include <QBrush>

#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <QtGui/QX11Info>
#include <QtCore/QByteArray>
#include <QtCore/QCoreApplication>

#include <QtGui/QMessageBox>
#include <QLabel>

bool          dirty;

CDesktop::CDesktop()
    : QWidget(0, Qt::FramelessWindowHint),
      state(0)
{
    //To optimize painting a bit you can set the following:
    //   note that child widgets might not be composed
    //   correctly then though
    //setAttribute(Qt::WA_PaintOnScreen);
    //setAttribute(Qt::WA_NoSystemBackground);
    dirty = true;
   
    QCoreApplication::instance()->setApplicationName("FM Rds Notify");
    QCoreApplication::instance()->setOrganizationName("FM Rds Notify");


    // Get required atoms
    Atom winTypeAtom = XInternAtom(QX11Info::display(), "_NET_WM_WINDOW_TYPE", false);
    Atom homeAppletAto = XInternAtom(QX11Info::display(), "_HILDON_WM_WINDOW_TYPE_HOME_APPLET", false);
    Atom appletIDAtom = XInternAtom(QX11Info::display(), "_HILDON_APPLET_ID", false);
    Atom utf8Atom = XInternAtom(QX11Info::display(), "UTF8_STRING", false); 
    Atom appletSettingAtom = XInternAtom(QX11Info::display(), "_HILDON_APPLET_SETTINGS", false);
   
    // Set correct window type
    XChangeProperty(QX11Info::display(), winId(), winTypeAtom, XA_ATOM, 32,
		    PropModeReplace, (unsigned char *) &homeAppletAto, 1);

    // Use application name to fill AppletID
    QByteArray id (QCoreApplication::instance()->applicationName().remove(' ').toUtf8());
    XChangeProperty(QX11Info::display(), winId(), appletIDAtom, utf8Atom, 8, 
    	            PropModeReplace, (unsigned char *)id.constData(), id.length());

    // Add setting button. This button is shown when hildon-desktop is in edit mode. 
    int settings = 0;
    XChangeProperty(QX11Info::display(), winId(), appletSettingAtom, XA_CARDINAL, 32,
		    PropModeReplace, (unsigned char*)&settings, 1);

    if (QFile( QDir::homePath()+"/MyDocs/rds.svg").exists())
        renderer = new QSvgRenderer(QDir::homePath()+"/MyDocs/rds.svg", this);
    else
        renderer = new QSvgRenderer(QLatin1String(":icons/rds.svg"), this);

    resize( renderer->defaultSize());

    // Mafw Meta data connection
    rdsMafw = new CMafwMeta(this);

    // Mafw Meta data connection
    rdsRTCom = new CRTCom(this);

    // Rds Broadcast writer 
    rdsBroadcast = new CRdsBroadcast(this);

    // Rds Broadcast writer
    rdsSms = new CSms(this);

    // Create Radio Tuner
    fmRadio = new CFMRadio(this);

    // Connect signal changed
    connect(rdsMafw, SIGNAL(sigChanged(QString,int)), rdsBroadcast, SLOT(slotSetText(QString,int)));
    connect(rdsRTCom, SIGNAL(sigChanged(QString,int)), rdsBroadcast, SLOT(slotSetText(QString,int)));
    connect(rdsSms, SIGNAL(sigChanged(QString,int)), rdsBroadcast, SLOT(slotSetText(QString,int)));

    QDBusConnection::systemBus().connect( 0, FMTX_PATH, FMTX_IF,  "info", this, SLOT(slotFMTXChanged(QString,QString)));

    // Plug start button
    startButton = new QPushButton(this);
    startButton->setCheckable(true);
    //Transparent button
    startButton->setFlat(true);
    startButton->setUpdatesEnabled(false);
    connect(startButton, SIGNAL(toggled(bool)), this, SLOT(slotToggleFM(bool)));

    nextButton = new QPushButton(this);
    //Transparent button
    nextButton->setFlat(true);
    //nextButton->setUpdatesEnabled(false);
    connect(nextButton, SIGNAL(clicked(bool)), this, SLOT(slotClickedNext(bool)));

    previousButton = new QPushButton(this);
    //Transparent button
    previousButton->setFlat(true);
    //previousButton->setUpdatesEnabled(false);
    connect(previousButton, SIGNAL(clicked(bool)), this, SLOT(slotClickedPrevious(bool)));

}


void CDesktop::paintEvent(QPaintEvent *e)
{
    QPainter p(this);
    p.setRenderHint(QPainter::Antialiasing);
    p.setClipRect(e->rect());

    //make sure you clean your widget with a transparent
    //  color before doing any rendering
    //  note the usage of a composition mode Source
    //  it's important!
    p.save();
    p.setCompositionMode(QPainter::CompositionMode_Source);
    p.fillRect(rect(), Qt::transparent);
    p.restore();
    
    if (dirty) {
        cache.fill(Qt::transparent);
        QPainter p(&cache);
        p.setRenderHint(QPainter::Antialiasing);
        renderer->render(&p);
    	drawDesktop(&p);
        p.end();
        dirty = false;
    }
    p.drawPixmap(0, 0, cache);
        
}

//Filter show setting requests
bool QWidget::x11Event ( XEvent * event )
{
  static Atom appletShowSettingAtom = XInternAtom(QX11Info::display(), "_HILDON_APPLET_SHOW_SETTINGS", false);

  if (event->xclient.message_type == appletShowSettingAtom ) { 
      CDlgSetup *mySetup = new CDlgSetup();
      mySetup->exec();
      dirty = true;
      update();
      return true;
  }
  return false;
}

void CDesktop::resizeEvent(QResizeEvent *e)
{
    if (e->size() != cache.size()) {
        cache = QPixmap(e->size());
        dirty = true;
        int w = e->size().width();
        int h = e->size().height();
	Q_UNUSED(w);
	Q_UNUSED(h);
    }
}

void CDesktop::drawStatus(QPainter *p, QRectF rect, bool status)
{
    p->save();
    QFont localFont = p->font();
    localFont.setPointSize(12);
    p->setFont(localFont);
    if (status) {
       //p->setBrush(Qt::green);
       p->setPen(Qt::green); 
       p->drawText(rect , Qt::AlignCenter, tr("yes") );
    } else {
       //p->setBrush(Qt::ref);
       p->setPen(Qt::red); 
       p->drawText(rect, Qt::AlignCenter, tr("no") );
    }
    p->restore();
}

void CDesktop::drawDesktop(QPainter *p)
{
    if ( CMafwMeta::self().getEnable())
        renderer->render(p, "Music_On_Icon", renderer->boundsOnElement("Music_On_Icon"));
    else
        renderer->render(p, "Music_Off_Icon", renderer->boundsOnElement("Music_Off_Icon"));

    if ( CSms::self().getEnable())
        renderer->render(p, "SMS_On_Icon", renderer->boundsOnElement("SMS_On_Icon"));
    else
        renderer->render(p, "SMS_Off_Icon", renderer->boundsOnElement("SMS_Off_Icon"));

    if ( CRTCom::self().getEnable())
        renderer->render(p, "Phone_On_Icon", renderer->boundsOnElement("Phone_On_Icon"));
    else
        renderer->render(p, "Phone_Off_Icon", renderer->boundsOnElement("Phone_Off_Icon"));

    p->save();
    QFont localFont = p->font();
    localFont.setPointSize(14);
    p->setFont(localFont);
    p->setPen(Qt::white);
    
    QRectF titleRect = renderer->boundsOnElement("Station_Name");
    if ( state)
        if ((titleRect.height() == 0) &&  (titleRect.width() == 0))
            p->drawText(titleRect.x(), titleRect.y(), CRdsBroadcast::self().getDefText());
        else
            p->drawText(titleRect, CRdsBroadcast::self().getDefText());
    else
        if ((titleRect.height() == 0) &&  (titleRect.width() == 0))
            p->drawText(titleRect.x(), titleRect.y(), tr(""));
        else
            p->drawText(titleRect, CRdsBroadcast::self().getDefText());

    QRectF frequency = renderer->boundsOnElement("Station_Freq");

    if ((frequency.height() == 0) &&  (frequency.width() == 0))
        p->drawText(frequency.x(), frequency.y(), QString("%1 Mhz").arg( CFMRadio::self().getFrequency()/1000 ));
    else
        p->drawText(frequency,  QString("%1 Mhz").arg( CFMRadio::self().getFrequency()/1000) );

    localFont.setPointSize(9);
    p->setFont(localFont);

    QRectF freqStepPlus = renderer->boundsOnElement("Amount_Plus");
    p->drawText(freqStepPlus.x(), freqStepPlus.y(),  QString("%1 Mhz").arg( CFMRadio::self().getStep()/1000 ) );

    QRectF freqStepMin = renderer->boundsOnElement("Amount_Min");
    p->drawText(freqStepMin.x(), freqStepMin.y(),  QString("%1 Mhz").arg( CFMRadio::self().getStep()/1000) );

    QRect FMIcon = renderer->boundsOnElement("FM_On_Icon").toAlignedRect();
    FMIcon.adjust(4,4,4,4);
    if ( state )
    {
        // Draw stop button in place of FM icon
        renderer->render(p, "TX_Status_On", renderer->boundsOnElement("TX_Status_On"));
        p->setOpacity(1.0);
        renderer->render(p, "FM_On_Icon", renderer->boundsOnElement("FM_On_Icon"));
        p->setOpacity(0.0);
        startButton->setGeometry(FMIcon);
    }
    else
    {
        // Draw start button in place of FM icon
        renderer->render(p, "TX_Status_Off", renderer->boundsOnElement("TX_Status_Off"));
        p->setOpacity(1.0);
        renderer->render(p, "FM_Off_Icon", renderer->boundsOnElement("FM_Off_Icon"));
        p->setOpacity(0.0);
        startButton->setGeometry(FMIcon);
    }
    QRect next = renderer->boundsOnElement("Plus").toAlignedRect();
    next.adjust(4,4,4,4);
    QRect prev = renderer->boundsOnElement("Min").toAlignedRect();
    prev.adjust(4,4,4,4);
    nextButton->setGeometry(next);
    previousButton->setGeometry(prev);

    p->restore();
}

void CDesktop::slotFMTXChanged(QString type, QString status)
{
    if (type == "connected") {
	rdsMafw->setConnect(status.toInt() );
	rdsRTCom->setConnect(status.toInt() );
        rdsSms->setConnect(status.toInt() );
	dirty = true;
	state = status.toInt();
	update();
    }
}

void CDesktop::slotToggleFM(bool checked){
    CFMRadio::self().setState(checked);
}

void CDesktop::slotClickedNext(bool checked){
    CFMRadio::self().setFreqUp();
    dirty = true;
    update();
    qDebug() << "Current frequency=" << CFMRadio::self().getFrequency();
}

void CDesktop::slotClickedPrevious(bool checked){
    CFMRadio::self().setFreqDown();
    dirty = true;
    update();
    qDebug() << "Current frequency=" << CFMRadio::self().getFrequency();
}
