from com import Component, msgs
from ui.Button import Button
from ui.ImageButton import ImageButton
from ui.Label import Label
from ui.itemview import ThumbableGridView
from ui.Slider import VSlider
from ui.dialog import InputDialog
from ui.dialog import OptionDialog
from ui.dialog import ListDialog
from ui.Pixmap import Pixmap
from ui.Window import Window
from StationItem import StationItem
from FMScale import FMScale
from FMRadio import *
from SoundPipe import SoundPipe
import platforms
import values
from theme import theme
import maemostations

import gtk
import gobject
try:
    import hildon
except:
    pass


_DEVICE_ERRORS = [
    "",
    "Error: Cannot activate Bluetooth/FM combo chip",
    "Error: Cannot establish communication to the FM receiver",
    "Error: Cannot load FM radio driver module",
    "Error: Cannot power up FM radio"
]


class FMRadioUI(Component, Window):
    """
    Player for FM radio stations.
    """

    def __init__(self):

        self.__buffer = None

        self.__radio = None
        self.__keep_alive_handler = None
        self.__device = platforms.get_product_code()

        self.__sound_pipe = SoundPipe()
        self.__use_speaker = False
        self.__was_playing_before_interruption = False
        self.__have_headphones = False
        
        self.__stations = maemostations.get_stations()
        self.__stations_dirty = False
        

        Component.__init__(self)
        Window.__init__(self, Window.TYPE_TOPLEVEL)
        self.set_title("FM Radio")
        self.set_visible(True)
        self.connect_closed(self.__on_close_window)
        self.connect_key_pressed(self.__on_key_pressed)
        
        self.__update_app_menu()
        
        # [Add] button
        self.__btn_add = Button("Save Station")
        self.__btn_add.connect_clicked(self.__on_btn_add)
        self.add(self.__btn_add)

        # station button
        self.__btn_station = Button("-")
        self.__btn_station.connect_clicked(self.__on_btn_station)
        self.__btn_station.set_visible(False)
        self.add(self.__btn_station)

        # [Select] button
        self.__btn_select = ImageButton(theme.mb_btn_bookmark_1,
                                        theme.mb_btn_bookmark_2)
        self.__btn_select.connect_clicked(self.__on_btn_select) #self.__on_toggle_view)
        self.add(self.__btn_select)

        # stations list
        #self.__stations_list = ThumbableGridView()
        #self.__stations_list.set_visible(False)
        #self.add(self.__stations_list)
        
        # slider for scrolling
        #self.__slider = VSlider(theme.mb_list_slider)
        #self.__slider.set_visible(False)
        #self.add(self.__slider)
        #self.__stations_list.associate_with_slider(self.__slider)

        # FM scale
        self.__fmscale = FMScale()
        self.__fmscale.set_range(87500, 108000)
        self.__fmscale.set_stations(*self.__stations)
        self.__fmscale.connect_frequency_changed(self.__on_tune)
        self.__fmscale.connect_scanning_activated(self.__on_scan)
        self.add(self.__fmscale)

        self.__lbl_mhz = Label("87.5", theme.font_fmradio_mhz_label,
                               theme.color_fmradio_mhz_label)
        self.__lbl_mhz.set_alignment(Label.CENTERED)
        self.add(self.__lbl_mhz)
       
        # [Play] button
        self.__btn_play = ImageButton(theme.mb_btn_play_1,
                                      theme.mb_btn_play_2)
        self.__btn_play.connect_clicked(self.__on_btn_play)
        self.add(self.__btn_play)

        self.__update_stations_list()


    def __prepare_buffer(self):
    
        w, h = self.get_size()
        if (not self.__buffer):
            self.__buffer = Pixmap(None, w, h)
        else:
            buf_w, buf_h = self.__buffer.get_size()
            if ((buf_w, buf_h) != (w, h)):
                self.__buffer = Pixmap(None, w, h)

       
    def render_this(self):
    
        x, y = self.get_screen_pos()
        w, h = self.get_size()
        screen = self.get_screen()
        
        screen.fill_area(x, y, w, h, theme.color_mb_background)
        #screen.fill_area((w - 4) / 2, 0, 4, 100,
        #                 theme.color_fmradio_scale_needle)

        #if (self.__stations_list.is_visible()):
        #    self.__stations_list.set_geometry(40, 0, w - 80, h)
        #    self.__slider.set_geometry(0, 0, 40, h)
        
        if (self.__stations):
            self.__btn_select.set_visible(True)
        else:
            self.__btn_select.set_visible(False)

        self.__btn_select.set_geometry(w - 80, h - 80, 80, 80)
        self.__btn_add.set_geometry(80, h - 80, w - 160, 80)
        self.__btn_station.set_geometry(80, h - 80, w - 160, 80)

        self.__btn_play.set_geometry(0, h - 80, 80, 80)
        self.__fmscale.set_geometry(0, 80, w, h - 80 - 80 - 1)
        self.__lbl_mhz.set_geometry(0, 20, w, 48)
        
        screen.draw_pixbuf(theme.fmradio_needle, (w - 24) / 2, h - 80 - 60)


    def __render_buffered(self):
    
        if (self.may_render()):
            x, y = self.get_screen_pos()
            w, h = self.get_size()
            screen = self.get_screen()
            
            self.render_at(self.__buffer)
            screen.copy_buffer(self.__buffer, 0, 0, x, y, w, h)            
        #end if


    def __update_app_menu(self):
    
        if (self.__use_speaker):
            lbl = "Headphones Output"
        else:
            lbl = "Speaker Output"
            
        self.set_window_menu(
          (None, lbl, self.__on_toggle_output),
          (None, "Info", self.__on_menu_info))


    def __update_buttons(self):
    
        name = self.__get_current_station_name()
        changed = False
        if (name):
            if (self.__btn_add.is_visible()):
                changed = True
            self.__btn_station.set_text(name)
            self.__btn_add.set_visible(False)
            self.__btn_station.set_visible(True)

        else:
            if (not self.__btn_add.is_visible()):
                changed = True
            self.__btn_add.set_visible(True)
            self.__btn_station.set_visible(False)

        if (changed and self.__fmscale.is_visible()):
            x, y = self.__btn_add.get_screen_pos()
            w, h = self.__btn_add.get_size()
            self.__prepare_buffer()
            self.render_at(self.__buffer)
            self.fx_slide_vertical(self.__buffer, x, y, w, h, self.SLIDE_UP)
        elif (self.__fmscale.is_visible()):
            self.__btn_station.render()
        #end if


    """
    def __update_stations_list(self):
    
        self.__stations_list.clear_items()
        for f, n in self.__stations:
            item = StationItem(n, f)
            self.__stations_list.append_item(item)
        #end for
        self.__stations_list.invalidate()
    """


    """
    def __toggle_view(self):
    
        if (self.__stations_list.is_visible()):
            self.__update_buttons()
            self.__btn_play.set_visible(True)
            self.__stations_list.set_visible(False)
            self.__slider.set_visible(False)
            self.__btn_select.set_visible(True)
            self.__fmscale.set_visible(True)
            self.__lbl_mhz.set_visible(True)
            slide_direction = self.SLIDE_DOWN
            
        else:
            self.__stations_list.set_visible(True)
            self.__slider.set_visible(True)
            self.__btn_play.set_visible(False)
            self.__btn_add.set_visible(False)
            self.__btn_station.set_visible(False)
            self.__btn_select.set_visible(False)
            self.__fmscale.set_visible(False)
            self.__lbl_mhz.set_visible(False)
            slide_direction = self.SLIDE_UP

        w, h = self.get_size()
        self.__prepare_buffer()
        self.render_at(self.__buffer)
        self.fx_slide_vertical(self.__buffer, 0, 0, w, h, slide_direction)
    """     


    def __on_close_window(self):
    
        def f():
            if (self.__stations_dirty):
                maemostations.save_stations(self.__stations)

            gtk.main_quit()
    
    
        self.__radio_off()
        if (self.__stations_dirty):
            try:
                hildon.hildon_banner_show_information(self.get_gtk_window(), "",
                        "Saving Stations")
            except:
                pass
        #end if
        gobject.idle_add(f)


    def __on_key_pressed(self, key):
    
        if (key == "Left"):
            self.__fmscale.step_down()
            
        elif (key == "Right"):
            self.__fmscale.step_up()

            


    def __on_menu_info(self):
    
        hildon.hildon_banner_show_information(self.get_gtk_window(), "",
                "%s - version %s\n%s" \
                % (values.NAME, values.VERSION, values.COPYRIGHT))
        

    def __on_toggle_output(self):
    
        self.__use_speaker = not self.__use_speaker
        self.__update_app_menu()

        if (self.__radio):
            self.__sound_pipe.use_speaker(self.__use_speaker)
      
        
    def __on_tune(self, v):
    
        if (self.__assert_headphones()):
            if (not self.__radio):
                self.__radio_on()
                
            self.__set_frequency(v)
        #end if
        self.__lbl_mhz.set_text("%0.1f" % (v / 1000.0))
        self.__update_buttons()

        if (self.__fmscale.get_value() != v):
            self.__fmscale.set_value(v)

        # hilight list item
        #self.__stations_list.set_hilight(-1)
        #cnt = 0
        #for item in self.__stations_list.get_items():
        #    if (item.get_frequency() == v):
        #        self.__stations_list.set_hilight(cnt)
        #        break
        #    cnt += 1
        #end for
        #self.__stations_list.invalidate()
        #self.__stations_list.render()

        self.set_title(self.__get_current_station_name() or
                       "%0.1f MHz" % (v / 1000.0))


    def __on_scan(self, direction):
    
        def f(freq):
            self.__fmscale.set_value(freq)
            gtk.main_iteration(False)
    
        if (self.__radio):
            if (direction < 0):
                self.__radio.scan_previous(f)
            else:
                self.__radio.scan_next(f)
            #self.__fmscale.set_value(self.__radio.get_frequency())
        #end if


    """
    def __on_toggle_view(self):
    
        self.__toggle_view()
    """


    def __on_btn_play(self):
    
        if (self.__assert_headphones()):
            if (self.__radio):
                self.__radio_off()

            else:
                self.__radio_on()
                
            freq = self.__radio.get_frequency()
            low, hi = self.__radio.get_frequency_range()
        #end if


    def __on_btn_add(self):
        
        freq = self.__fmscale.get_value()
        dlg = InputDialog("Enter Station Name")
        dlg.add_input("Name", "%0.2f MHz" % (freq / 1000.0))
    
        ret = dlg.run()
        if (ret == 0):
            name = dlg.get_values()[0] or "%0.1f MHz" % (freq / 1000.0)
            self.__stations.append((freq, name))
            self.__fmscale.set_stations(*self.__stations)
            self.__update_stations_list()
            self.__update_buttons()
            self.__render_buffered()
            self.__stations_dirty = True
        #end if
        
        
    def __on_btn_station(self):
    
        dlg = OptionDialog("Options")
        dlg.add_option(None, "Rename Station")
        dlg.add_option(None, "Delete Station")

        ret = dlg.run()
        if (ret == 0):
            choice = dlg.get_choice()
            if (choice == 0):
                self.__on_rename_station()
            elif (choice == 1):
                self.__on_delete_station()


    def __on_btn_select(self):
    
        self.__stations.sort(lambda a,b:cmp(a[1], b[1]))

        dlg = ListDialog("Favorites")
        for freq, name in self.__stations:
            item = StationItem(name, freq)
            dlg.add_item(item)
        #end for
        
        ret = dlg.run()
        if (ret == 0):
            item = dlg.get_choice()
            freq = item.get_frequency()
            self.__on_tune(freq)
        #end if


    def __on_delete_station(self):
    
        dlg = OptionDialog("Delete Station?")
        dlg.add_option(None, "Yes")
        dlg.add_option(None, "No")
        
        ret = dlg.run()
        if (dlg.get_choice() == 0):
            freq = self.__fmscale.get_value()
            new_stations = []
            for f, n in self.__stations:
                if (freq == f):
                    pass
                else:
                    new_stations.append((f, n))
            #end for
        
            self.__stations = new_stations
            self.__fmscale.set_stations(*self.__stations)
            self.__update_stations_list()
            self.__update_buttons()
            self.__render_buffered()
            self.__stations_dirty = True
        #end if

    def __on_rename_station(self):

        dlg = InputDialog("Rename Radio Station")
        dlg.add_input("Name", self.__get_current_station_name())
        
        ret = dlg.run()
        if (ret != 0):
            return
        new_name = dlg.get_values()[0]
    
        new_stations = []
        freq = self.__fmscale.get_value()
        for f, n in self.__stations:
            if (freq == f):
                new_stations.append((f, new_name))
            else:
                new_stations.append((f, n))
        #end for
        
        self.__stations = new_stations
        self.__fmscale.set_stations(*self.__stations)
        self.__update_stations_list()
        self.__update_buttons()
        self.__stations_dirty = True


    def __get_current_station_name(self):
    
        freq = self.__fmscale.get_value()
        stations = [ n for f, n in self.__stations
                     if f == freq ]
        if (stations):
            return stations[0]
        else:
            return ""



    def __assert_headphones(self):
    
        if (not self.__have_headphones):
            try:
                hildon.hildon_banner_show_information(self.get_gtk_window(), "",
                    "Please insert a headphones cable for a FM antenna.")
            except:
                # for testing without hildon bindings
                pass
            return False
        else:
            return True


    def __show_error(self, err):
    
        msg = _DEVICE_ERRORS[err]
    
        try:
            hildon.hildon_banner_show_information(self.get_gtk_window(),
                                                  "", msg)
        except:
            pass


    def __radio_on(self):

        if (self.__radio): return

        if (self.__device == "RX-51"):
            ret, dev = platforms.request_fmradio()
        else:
            ret = 1

        if (ret == 0):
            try:
                self.__radio = FMRadio(self.__device)
            except:
                return
        else:
            self.__show_error(ret)
            return
        #end if
        
        if (self.__device == "RX-51"):
            if (self.__keep_alive_handler):
                gobject.source_remove(self.__keep_alive_handler)
            
            self.__keep_alive_handler = gobject.timeout_add(15000,
                                                    self.__keep_radio_alive)
        #end if

        self.__sound_pipe.on()
        self.__sound_pipe.use_speaker(self.__use_speaker)
        self.__btn_play.set_images(theme.mb_btn_pause_1,
                                   theme.mb_btn_pause_2)
        
        
        
    def __radio_off(self):
    
        if (not self.__radio): return

        self.__sound_pipe.off()
        
        self.__radio.close()
        self.__radio = None

        self.__btn_play.set_images(theme.mb_btn_play_1,
                                   theme.mb_btn_play_2)



    def __set_frequency(self, freq):
    
        low, hi = self.__radio.get_frequency_range()
        
        self.__lbl_mhz.set_text("%0.2f MHz" % (freq / 1000.0))
        self.__radio.set_frequency(freq)
        

    def __keep_radio_alive(self):
    
        if (self.__radio):
            platforms.request_fmradio()
            return True
        else:
            self.__keep_alive_handler = None
            return False


    def __load(self, f):
    
        if (not self.__radio):
            self.__radio_on()
    
        parts = f.path.split("/")
        freq = int(parts[-1])
        self.__set_frequency(freq)
        self.__progress.set_message(f.name)


    def __begin_interruption(self):
        """
        Begins an interruption and switches off the radio, if on.
        """
        
        if (self.__radio):
            self.__was_playing_before_interruption = True
            try:
                hildon.hildon_banner_show_information(self.get_gtk_window(), "",
                        "FM Radio Interrupted")
            except:
                pass
        else:
            self.__was_playing_before_interruption = False
    
        self.__radio_off()


    def __end_interruption(self):
        """
        Resumes after an interruption end.
        """

        if (self.__was_playing_before_interruption):
            self.__radio_on()
 

    def handle_CORE_EV_FOLDER_INVALIDATED(self, f):
    
        self.__browser.reload_current_folder()


    def handle_SYSTEM_EV_HEADPHONES_INSERTED(self):
    
        self.__have_headphones = True
        self.__end_interruption()
        
        
    def handle_SYSTEM_EV_HEADPHONES_REMOVED(self):

        self.__have_headphones = False
        self.__begin_interruption()


    def handle_SYSTEM_EV_SOUND_VOLUME_CHANGED(self, v):
    
        # the hw keys don't affect the mixer for the speaker volume, so we
        # have to set that manually
        self.__sound_pipe.set_speaker_volume(v)


    def handle_SYSTEM_EV_PHONE_RINGING(self):
    
        self.__begin_interruption()


    def handle_SYSTEM_EV_PHONE_DIALING(self):

        self.__begin_interruption()


    def handle_SYSTEM_EV_PHONE_HANGUP(self):

        self.__end_interruption()

