from com import View, msgs
from ui.Toolbar import Toolbar
from ui.ImageButton import ImageButton
from ui.ProgressBar import ProgressBar
from storage import StorageBrowser
from FMRadio import *
from FMRadioDevice import FMRadioDevice
from SoundPipe import SoundPipe
import platforms
import values
from theme import theme

import gtk
import gobject
import hildon


class FMRadioUI(View):
    """
    Player for FM radio stations.
    """

    def __init__(self):
    
        self.__radio = None
        self.__keep_alive_handler = None
        self.__device = platforms.get_product_code()

        self.__sound_pipe = SoundPipe()
        self.__radio_storage = FMRadioDevice()
        self.__use_speaker = False
        

        View.__init__(self)
        self.set_title("FM Radio")
        self.set_visible(True)
        self.connect_closed(self.__on_close_window)
        
        self.__update_app_menu()

        # browser
        self.__browser = StorageBrowser()
        self.__browser.set_root_device(self.__radio_storage)
        self.__browser.connect_file_opened(self.__on_select_station)
        self.add(self.__browser)

        # progress bar
        self.__progress = ProgressBar()
        self.__progress.set_message("- No Station Selected -")
        self.add(self.__progress)
        self.__progress.connect_changed(self.__on_tune)
        
        # toolbar elements
        self.__btn_play = ImageButton(theme.mb_btn_play_1,
                                      theme.mb_btn_play_2)
        self.__btn_play.connect_clicked(self.__on_btn_play)

        btn_previous = ImageButton(theme.mb_btn_previous_1,
                                   theme.mb_btn_previous_2)
        btn_previous.connect_clicked(self.__on_btn_previous)

        btn_next = ImageButton(theme.mb_btn_next_1,
                               theme.mb_btn_next_2)
        btn_next.connect_clicked(self.__on_btn_next)

        btn_add = ImageButton(theme.mb_btn_add_1,
                              theme.mb_btn_add_2)
        btn_add.connect_clicked(self.__on_btn_add)

        
        # toolbar
        self.__toolbar = Toolbar()
        self.add(self.__toolbar)
        self.__toolbar.set_toolbar(btn_previous,
                                   self.__btn_play,
                                   btn_next,
                                   btn_add)

       
    def render_this(self):
    
        x, y = self.get_screen_pos()
        w, h = self.get_size()
        screen = self.get_screen()
        
        screen.fill_area(x, y, w, h, theme.color_mb_background)
        
        if (w < h):
            # portrait mode
            self.__browser.set_geometry(0, 0, w, h - 70 - 70)
            self.__progress.set_geometry(20, h - 70 - 50, w - 40, 48)
            self.__toolbar.set_geometry(0, h - 70, w, 70)

        else:
            # landscape mode
            self.__browser.set_geometry(0, 0, w - 70, h - 48)
            self.__progress.set_geometry(42, h - 48, w - 70 - 84, 48)
            self.__toolbar.set_geometry(w - 70, 0, 70, h)


    def __update_app_menu(self):
    
        if (self.__use_speaker):
            lbl = "Headphones Output"
        else:
            lbl = "Speaker Output"
            
        self.set_window_menu(
          (None, lbl, self.__on_toggle_output),
          (None, "Info", self.__on_menu_info))


    def __on_close_window(self):
    
        self.__radio_off()
        gtk.main_quit()


    def __on_menu_info(self):
    
        hildon.hildon_banner_show_information(self.get_gtk_window(), "",
                "%s - version %s\n%s\n\n" \
                "Don't forget to plug in a headphones cable " \
                "as FM antenna." \
                % (values.NAME, values.VERSION, values.COPYRIGHT))
        

    def __on_toggle_output(self):
    
        self.__use_speaker = not self.__use_speaker
        self.__update_app_menu()

        if (self.__radio):
            self.__sound_pipe.use_speaker(self.__use_speaker)


    def __on_select_station(self, f):
    
        try:
            self.__load(f)
        except:
            import traceback; traceback.print_exc()
        
        
    def __on_tune(self, v):
    
        if (not self.__radio):
            self.__radio_on()
            
        low, hi = self.__radio.get_frequency_range()
        freq = int(low + (hi - low) * v / 100.0)
        self.__set_frequency(freq)


    def __on_scan(self, freq):
    
        low, hi = self.__radio.get_frequency_range()
        self.__progress.set_position(freq - low, hi - low)
        self.__progress.set_message("%0.2f MHz" % (freq / 1000.0))
        gobject.timeout_add(5, lambda :False)
        gtk.main_iteration(True)



    def __on_btn_play(self):
    
        if (self.__radio):
            self.__radio_off()

        else:
            self.__radio_on()
            
        freq = self.__radio.get_frequency()
        low, hi = self.__radio.get_frequency_range()
        self.__progress.set_position(freq - low, hi - low)


    def __on_btn_previous(self):

        if (self.__radio):
            self.__radio.scan_previous(self.__on_scan)

        
    def __on_btn_next(self):
    
        if (self.__radio):
            self.__radio.scan_next(self.__on_scan)


    def __on_btn_add(self):
        
        if (self.__radio):
            try:
                self.__radio_storage.add_station(self.__radio.get_frequency())
            except:
                import traceback; traceback.print_exc()


    def __radio_on(self):

        if (self.__radio): return

        if (self.__device == "RX-51"):
            ret, dev = platforms.request_fmradio()
        else:
            ret = 0

        if (ret == 0):
            try:
                self.__radio = FMRadio(self.__device)
            except:
                return
        #end if
        
        if (self.__device == "RX-51"):
            if (self.__keep_alive_handler):
                gobject.source_remove(self.__keep_alive_handler)
            
            self.__keep_alive_handler = gobject.timeout_add(60000,
                                                    self.__keep_radio_alive)
        #end if

        self.__sound_pipe.on()
        self.__sound_pipe.use_speaker(self.__use_speaker)
        self.__btn_play.set_images(theme.mb_btn_pause_1,
                                   theme.mb_btn_pause_2)
        
        
        
    def __radio_off(self):
    
        if (not self.__radio): return

        self.__sound_pipe.off()
        
        self.__radio.close()
        self.__radio = None

        self.__btn_play.set_images(theme.mb_btn_play_1,
                                   theme.mb_btn_play_2)



    def __set_frequency(self, freq):
    
        low, hi = self.__radio.get_frequency_range()
        
        self.__progress.set_position(freq - low, hi - low)
        self.__progress.set_message("%0.2f MHz" % (freq / 1000.0))
        self.__radio.set_frequency(freq)
        

    def __keep_radio_alive(self):
    
        if (self.__radio):
            platforms.request_fmradio()
            return True
        else:
            self.__keep_alive_handler = None
            return False


    def __load(self, f):
    
        if (not self.__radio):
            self.__radio_on()
    
        parts = f.path.split("/")
        freq = int(parts[-1])
        self.__set_frequency(freq)
        self.__progress.set_message(f.name)



    def handle_CORE_EV_FOLDER_INVALIDATED(self, f):
    
        self.__browser.reload_current_folder()

