#include "flickrdemo.h"

#include <QDomDocument>

static const QSize gridSize(52, 52);
static const QSize thumbnailSize(50, 50);
static const QSize imageSize(150, 150);
static const QString apikey = "e36784df8a03fea04c22ed93318b291c";

#ifdef Q_OS_SYMBIAN
    static const QString savePath = "c:\\Data\\Images\\";  // In S60 Download images to Gallery
#elifdef Q_WS_HILDON
    static const QString savePath = "/home/user/MyDocs/.images/";
//FIXME Why saving stuff in C:\ ?
//#elifdef Q_WS_WINDOWS 
//    static const QString savePath = "c:\\";
#else
    static const QString savePath = QDir::homePath() + QDir::separator ();
#endif

FlickrDemo::FlickrDemo(QWidget *parent)
    : QMainWindow(parent),
      m_file(0),
      m_httpGetId(-1),
      m_httpThumbnailGetId(-1),
      m_pages(0),
      m_page(1),
      m_satellitesInView(0),
      m_satellitesUsed(0),
      m_latitude(-1000),
      m_longitude(-1000),
      m_downloadPictureList(true),
      m_downloadingThumbnails(false)
{
    setupUi(this);

    createMenus();

    listWidget->setGridSize(gridSize);
    listWidget->setIconSize(thumbnailSize);

    connect(&m_http, SIGNAL(requestFinished(int, bool)),
            this, SLOT(httpRequestFinished(int, bool)));
    connect(&m_http, SIGNAL(dataReadProgress(int, int)),
            this, SLOT(updateDataReadProgress(int, int)));
    connect(&m_http, SIGNAL(responseHeaderReceived(const QHttpResponseHeader &)),
            this, SLOT(readResponseHeader(const QHttpResponseHeader &)));

#ifdef Q_OS_SYMBIAN
    connect(&m_location, SIGNAL(locationChanged(double,double,double,float)),
            this, SLOT(valueChanged(double,double,double,float)));
    connect(&m_location, SIGNAL(numberOfSatellitesUsedChanged(int)),
            this, SLOT(numberOfSatellitesUsedChanged(int)));
    connect(&m_location, SIGNAL(numberOfSatellitesInViewChanged(int)),
            this, SLOT(numberOfSatellitesInViewChanged(int)));
#endif

    connect(&m_progressDialog, SIGNAL(canceled()), this, SLOT(cancelDownload()));

#ifdef Q_OS_SYMBIAN  
    if (m_location.open() == XQLocation::NoError)
    {
        m_location.startUpdates();
    }
#endif
   
/*   QNetworkProxy proxy;
   proxy.setType(QNetworkProxy::HttpProxy);
   proxy.setHostName("172.16.42.136");
   proxy.setPort(8080);
   QNetworkProxy::setApplicationProxy(proxy);
*/
}

void FlickrDemo::createMenus()
{
    m_downloadAct = new QAction(tr("Download Picture List"), this);
    menuBar()->addAction(m_downloadAct);
    connect(m_downloadAct, SIGNAL(triggered()), this, SLOT(on_downloadButton_clicked()));
    QAction *exitAct = new QAction(tr("Exit"), this);
    menuBar()->addAction(exitAct);
    connect(exitAct, SIGNAL(triggered()), this, SLOT(close()));
}

void FlickrDemo::valueChanged(double latit, double longit, double /*altitude*/, float /*speed*/)
{
    m_latitude = latit;
    m_longitude = longit;
    locationLabel->setText(tr("lattit: %1, longit: %2").arg(QString::number(latit),QString::number(longit)));
}

void FlickrDemo::numberOfSatellitesInViewChanged(int numSatellites)
{
    m_satellitesInView = numSatellites;
    viewSatelliteInfo();
}

void FlickrDemo::numberOfSatellitesUsedChanged(int numSatellites)
{
    m_satellitesUsed = numSatellites;
    viewSatelliteInfo();
}

void FlickrDemo::viewSatelliteInfo()
{
    satellitesLabel->setText(tr("%1 satellites, using : %2").arg(QString::number(m_satellitesInView),
                                                                 QString::number(m_satellitesUsed)));
}


void FlickrDemo::on_downloadButton_clicked()
{
    if (m_downloadPictureList)
    {
        downloadFlickerPictureList();
    }
    else
    {
        downloadPictureFromFlickr();
    }
}

void FlickrDemo::downloadFlickerPictureList()
{
    if (m_latitude == -1000 || m_longitude == -1000)
    {
        if (QMessageBox::question(this, tr("Flickr Demo"),
                                  tr("No satellite connection.\n"
                                  "Use hard coded coordinates?"),
                                  QMessageBox::Ok|QMessageBox::Cancel, QMessageBox::Cancel)
            == QMessageBox::Cancel)
        {
            return;
        }
        // If GPS signal is not detected, Munich area is used as default location
        m_latitude = 48.1376;
        m_longitude = 11.5752;
    }
#ifdef Q_OS_SYMBIAN 
    if (m_accessPointManager.isSetDefaultAccessPointSupported())
    {
        XQAccessPoint ap = m_accessPointManager.preferredAccessPoint();
        if (!ap.isNull())
        {
            if (m_accessPointManager.setDefaultAccessPoint(ap))
            {
                satellitesLabel->setText(tr("Access Point: %1").arg(ap.name()));
            }
        }
        else
        {
            QMessageBox::information(this, tr("Flickr Demo"),
                                     tr("Available Access Points not found."));
        
            return;
        }
    }

    m_location.stopUpdates();
#endif

    locationLabel->setText(tr("latit: %1, longit: %2").arg(QString::number(m_latitude),QString::number(m_longitude)));

    QString urlstring = "http://api.flickr.com/services/rest/?min_taken_date=2000-01-01+0:00:00&extras=date_taken&method=flickr.photos.search&per_page=30&sort=date-taken-desc";
    urlstring.append("&api_key=");
    urlstring.append(apikey);
    urlstring.append("&lat=");
    urlstring.append(QString::number(m_latitude));
    urlstring.append("&lon=");
    urlstring.append(QString::number(m_longitude));
    urlstring.append("&page=");
    urlstring.append(QString::number(m_page));

    QUrl url(urlstring);

    m_http.setHost(url.host(), QHttp::ConnectionModeHttp, url.port() == -1 ? 0 : url.port());
    m_httpRequestAborted = false;

    m_httpGetId = m_http.get(urlstring);

    m_progressDialog.setWindowTitle(tr("FlicrkDemo"));
    m_progressDialog.setLabelText(tr("Downloading\nPicture List."));
    m_progressDialog.setMaximum(10);
    m_progressDialog.setValue(1);
}

bool FlickrDemo::parsePictureList(QString xmlString)
{
    QString errorStr;
    int errorLine;
    int errorColumn;

    QDomDocument domDocument;
    if (!domDocument.setContent(xmlString, true, &errorStr, &errorLine,
                            &errorColumn))
    {
        QMessageBox::information(window(), tr("Flickr Demo"),
                             tr("XML Parse error at line %1, column %2:\n%3")
                             .arg(errorLine)
                             .arg(errorColumn)
                             .arg(errorStr));
        return false;
    }
    QDomElement root = domDocument.documentElement();

    if (root.tagName() != "rsp")
    {
        QMessageBox::information(window(), tr("Flickr Demo"),
                                           tr("Invalid XML format."));
        return false;
    }
    if (root.attribute("stat") != "ok")
    {
        QMessageBox::information(window(), tr("Flickr Demo"),
                                           tr("Query failed."));
        return false;
    }

    listWidget->clear();
    QListWidgetItem* listItem;
    QDomElement photosElement = root.firstChildElement("photos");
    if (!photosElement.isNull())
    {
        m_pages = (photosElement.attribute("pages")).toInt();
        m_page  = (photosElement.attribute("page")).toInt();

        QDomElement child = photosElement.firstChildElement("photo");
        while (!child.isNull())
        {
            QString url = "http://farm";
            url.append(child.attribute("farm"));
            url.append(".static.flickr.com/");
            url.append(child.attribute("server"));
            url.append("/");
            url.append(child.attribute("id"));
            url.append("_");
            url.append(child.attribute("secret"));

            m_names << url;

            listItem = new QListWidgetItem(child.attribute("title"));
            QImage image(thumbnailSize,QImage::Format_Mono);
            image.fill(1);
            listItem->setIcon(QPixmap::fromImage(image));
            listItem->setData(Qt::UserRole, url);
            listWidget->addItem(listItem);

            child = child.nextSiblingElement("photo");
        }
    } else
    {
        return false;
    }
    m_nameCounter = -1;
    downloadNextThumbnail();

    return true;
}

 void FlickrDemo::downloadPictureFromFlickr()
{
    QListWidgetItem* item = listWidget->currentItem();
    if (!item)
    {
        return;
    }
    QString pictureUrl = item->data(Qt::UserRole).toString();
    pictureUrl.append("_m.jpg");

    QUrl url(pictureUrl);
    QFileInfo fileInfo(url.path());
    QString fileName = fileInfo.fileName();
    if (fileName.isEmpty())
    {
        fileName = "test.jpg";
    }

    m_filePath = savePath;
    m_filePath.append(fileName);

    if (QFile::exists(m_filePath))
    {
        if (QMessageBox::question(this, tr("Flickr Demo"),
                                  tr("File %1 is already downloaded."
                                  "Overwrite?").arg(fileName),
                                  QMessageBox::Ok|QMessageBox::Cancel, QMessageBox::Cancel)
                == QMessageBox::Cancel)
        {
            return;
        }
        QFile::remove(m_filePath);
    }

    m_file = new QFile(m_filePath);
    if (!m_file->open(QIODevice::WriteOnly))
    {
        QMessageBox::information(this, tr("Flickr Demo"),
                                 tr("Unable to save the file %1: %2.")
                                 .arg(m_filePath).arg(m_file->errorString()));
        delete m_file;
        m_file = 0;
        return;
    }

    m_http.setHost(url.host(), QHttp::ConnectionModeHttp, url.port() == -1 ? 0 : url.port());

    m_httpRequestAborted = false;
    QByteArray encodedUrl = QUrl::toPercentEncoding(url.path(), "!$&'()*+,;=:@/");
    if (encodedUrl.isEmpty())
    {
        encodedUrl = "/";
    }
    m_httpGetId = m_http.get(encodedUrl, m_file);

    m_progressDialog.setWindowTitle(tr("Flickr Demo"));
    m_progressDialog.setLabelText(tr("Downloading:\n%1").arg(fileName));
    m_progressDialog.setMaximum(10);
    m_progressDialog.setValue(1);

    downloadButton->setEnabled(false);
}

void FlickrDemo::cancelDownload()
{
    m_httpRequestAborted = true;
    m_downloadingThumbnails = false;
    m_http.abort();
    downloadButton->setEnabled(true);
}

void FlickrDemo::httpRequestFinished(int requestId, bool error)
{
    if (m_downloadingThumbnails && m_httpThumbnailGetId == requestId)
    {
        if (!error)
        {
            QByteArray picture = m_http.readAll();
            if (!picture.isNull() && picture.size() > 0)
            {
                QListWidgetItem* item = listWidget->item(m_nameCounter);
                QImage image;
                if (image.loadFromData(picture,"jpg"))
                {
                    item->setIcon(QPixmap::fromImage(image.scaled(thumbnailSize, Qt::KeepAspectRatio, Qt::SmoothTransformation)));
                    listWidget->update();
                }
            }
        }
        downloadNextThumbnail();
        return;
    }

    if (requestId != m_httpGetId)
    {
        return;
    }

    m_progressDialog.hide();
    
    if (m_httpRequestAborted)
    {
        if (m_file)
        {
            m_file->close();
            m_file->remove();
            delete m_file;
            m_file = 0;
        }

        return;
    }

    if (!m_downloadPictureList)
    {
        m_file->close();
    }

    if (error)
    {
        if (!m_downloadPictureList)
        {
            m_file->remove();
        }
        QMessageBox::information(this, tr("Flickr Demo"),
                                 tr("Download failed: %1.")
                                 .arg(m_http.errorString()));
    }

    if (m_downloadPictureList)
    {
        if (parsePictureList(QString::fromUtf8(m_http.readAll())))
        {
            m_downloadPictureList = false;
            downloadButton->setText(tr("Download Selected Picture"));
            m_downloadAct->setText(tr("Download Selected Picture"));
        }
    }
    else
    {
        PictureDialog dialog(m_filePath,listWidget->currentItem()->text(),this);
        if (!dialog.exec()) {
            if (m_file->exists()) {
                m_file->remove();
            }
        }
        delete m_file;
        m_file = 0;
    }

    downloadButton->setEnabled(true);
}

void FlickrDemo::readResponseHeader(const QHttpResponseHeader &responseHeader)
{
    switch (responseHeader.statusCode())
    {
        case 200: // Ok
        case 301: // Moved Permanently
        case 302: // Found
        case 303: // See Other
        case 307: // Temporary Redirect
            // these are not error conditions
            break;
        default:
            QMessageBox::information(this, tr("Flickr Demo"),
                                     tr("Download failed: %1.")
                                     .arg(responseHeader.reasonPhrase()));
            m_downloadingThumbnails = false;
            m_httpRequestAborted = true;
            m_progressDialog.hide();
            m_http.abort();
    }
}

 void FlickrDemo::updateDataReadProgress(int bytesRead, int totalBytes)
{
    if (m_httpRequestAborted)
    {
        return;
    }

    if (!m_downloadingThumbnails)
    {
        m_progressDialog.setMaximum(totalBytes);
        m_progressDialog.setValue(bytesRead);
    }
}

void FlickrDemo::downloadNextThumbnail()
{
    m_nameCounter++;
    if (m_nameCounter < m_names.count())
    {
        QString pictureUrl = m_names[m_nameCounter];
        pictureUrl.append("_s.jpg");
        QUrl url(pictureUrl);
        m_http.setHost(url.host(), QHttp::ConnectionModeHttp, url.port() == -1 ? 0 : url.port());
        m_downloadingThumbnails = true;
        m_httpThumbnailGetId = m_http.get(pictureUrl);
    }
    else
    {
        m_downloadingThumbnails = false;
    }
}

PictureDialog::PictureDialog(const QString& filePath, const QString& pictureName, QWidget* parent)
    : QDialog(parent)
{
    setupUi(this);

    QString fileName = QFileInfo(filePath).fileName();
    label->setText(tr("Downloaded:\n%1\n%2").arg(pictureName).arg(fileName));

    QImage image;
    image.load(filePath);
    imageLabel->setPixmap(QPixmap::fromImage(image.scaled(imageSize, Qt::KeepAspectRatio, Qt::SmoothTransformation)));
    
#ifdef Q_OS_SYMBIAN
    showFullScreen();
#endif    
}

// End of file

