/* This file is part of "modern computer flatboat", a pdf viewer.
 * Copyright (C) 2012 Frank Fuhlbrück 
 * License: GPLv3 (or any later version, at your option)
 * See the file "LICENSE".
 */
#ifndef QPDFIMAGE_H
#define QPDFIMAGE_H

#include <QDeclarativeItem>
#include <QTimer>
extern "C" {
	#include <fitz.h>
}

class PdfLink : public QObject {
	Q_OBJECT
	
public:
	explicit PdfLink(QObject *parent = 0);
	PdfLink(fz_link* fzl);
	PdfLink(const PdfLink&);
	Q_PROPERTY(QRect position READ position)
	Q_PROPERTY(int type READ type)//0: invalid, 1:PDF page 2: URI
	Q_PROPERTY(QVariant destination READ destination)
	
	QRect position();
	int type();
	QVariant destination();

private:
	QRect position_prv;
	int type_prv;
	QVariant destination_prv;
};
Q_DECLARE_METATYPE (PdfLink)
Q_DECLARE_METATYPE (QRect*)

class QPdfImage : public QDeclarativeItem
{
	Q_OBJECT
public:
	explicit QPdfImage(QDeclarativeItem *parent = 0);
	Q_PROPERTY(QString filepath READ filepath WRITE loadFile())
	Q_PROPERTY(bool textMode READ textMode WRITE setTextMode NOTIFY textModeChanged)
	Q_PROPERTY(int pagenum READ pagenum WRITE gotoPage NOTIFY pagenumChanged)
	Q_PROPERTY(int regionIdx READ regionIdx WRITE setRegionIdx NOTIFY regionIdxChanged)
	Q_PROPERTY(int totalPages READ totalPages)
	Q_PROPERTY(int zoom READ zoom WRITE setZoom NOTIFY zoomChanged)
	Q_PROPERTY(QString highlightPhrase READ highlightPhrase WRITE setHighlightPhrase)
	Q_PROPERTY(QList<PdfLink> links READ links)
	Q_PROPERTY(bool linkHints READ linkHints WRITE setLinkHints NOTIFY linkHintsChanged)
	Q_PROPERTY(int pagenum READ pagenum WRITE gotoPage NOTIFY pagenumChanged)
	Q_PROPERTY(bool caseSensitive READ caseSensitive WRITE setCaseSensitive)
	
	QString filepath();
	bool textMode();
	int totalPages();
	int pagenum();
	int regionIdx();
	int region();
	int zoom();
	int crop();
	QString highlightPhrase();
	QList<PdfLink> links();
	bool linkHints();
	bool caseSensitive();
	 
	
	void paint(QPainter *painter, const QStyleOptionGraphicsItem *, QWidget *);
	void mousePressEvent (QGraphicsSceneMouseEvent *event);
	void mouseReleaseEvent (QGraphicsSceneMouseEvent*);
	void mouseDoubleClickEvent (QGraphicsSceneMouseEvent*);
	
	
signals:
	void regionIdxChanged(int idx);
	void pagenumChanged(int pn);
	void zoomChanged(int zm);
	void textModeChanged(bool tm);
	void clickForward();
	void linkSelected(QVariant destination);
	void linkHintsChanged(bool lh);
	
public slots:
	bool loadFile(QString filename);
	void setTextMode(bool tm);
	bool gotoPage(int pagenum,bool render=true);
	bool setZoom(int percent);
	void setHighlightPhrase(QString phrase);
	void setLinkHints(bool lh);
	void setCaseSensitive(bool cs);
	bool searchNext(QString str);
	bool searchPrev(QString str);
	QRect prevRegion(bool render=true);
	QRect curRegion();//no side effekt, therefore no slot
	QRect nextRegion(bool render=true);
	QRect setRegionIdx(int idx);
	void stopClickTimer();

	
private:
	QString filepath_prv;
	QImage *image;
	QString lastSearchPhrase;
	bool lastSearchCaseSensitive;
	QList<QRect> regions;
	int curRegionIdx;
	QStringList textDocument;
	QList<QRect> highlight;
	QList<int> hitCount;
	int hitOnPage;
	fz_context *context;
	fz_document *document;
	fz_pixmap *pixmap;
	fz_page *page;
	fz_text_page *text_page;
	bool textMode_prv;
	int pagenum_prv;
	int totalpages_prv;
	int zoom_prv;
	QString highlightPhrase_prv;
	QList<PdfLink> links_prv;
	bool linkHints_prv;
	bool caseSensitive_prv;
	QTimer clickTimer;
	
	QStringList document2stringList();
	
};
#endif // QPDFIMAGE_H
