/* This file is part of "modern computer flatboat", a pdf viewer.
 * Copyright (C) 2012 Frank Fuhlbrück 
 * License: GPLv3 (or any later version, at your option)
 * See the file "LICENSE".
 */
import QtQuick 1.1
import flatboat.PdfImage 1.0

Rectangle {
	id: mainRec
	width: 480
	height: 800
	signal showFullScreen()
	signal showNormal()
	property int refWidth: searchPrev.width;
	property int refHeight: searchPrev.height; 
	property int refPtSize: searchPrev.pointSize;
	onWidthChanged: if(flick.columnMode) flick.columnModeChanged();
	onHeightChanged: if(flick.columnMode) flick.columnModeChanged();
	
	FileChooser {
		id:fileChooser
		onOpenFile: {
			pdf.loadFile(sethist.curDoc.filepath);
			restoreSettings();
		}
	}
	
	function restoreSettings(){
		pdf.pagenum = sethist.curDoc.lastPage ? sethist.curDoc.lastPage : 1;
		pdf.zoom = sethist.curDoc.lastZoom ? sethist.curDoc.lastZoom : 100;
		if(sethist.curDoc.lastMode == "textMode")
			txtToggle.state = "active";
		else if(sethist.curDoc.lastMode == "colMode"){
			colToggle.state = "active";
			pdf.regionIdx = sethist.curDoc.lastRegionIdx ? sethist.curDoc.lastRegionIdx : 0;
			var r = pdf.curRegion();
			flick.contentX=r.x-10;
			flick.contentY = sethist.curDoc.lastContentY ? sethist.curDoc.lastContentY : r.y;
		} else {
			colToggle.state = "";
			txtToggle.state = "";
		}
	}
	
	function curDocSet(key,val){
		var curd = sethist.curDoc;
		curd[key] = val;
		sethist.curDoc = curd;
// 		console.log(key+":");
// 		console.log(sethist.curDoc[key]);
	}
	
	function openPreviousDoc(idx){
		if(sethist.documents.length <= idx)
			return;
		//TODO:inefficient
		sethist.newDocument(sethist.documents[idx]);
		pdf.loadFile(sethist.curDoc.filepath);
		restoreSettings();
	}
	
	property variant var_helper;
	function unvariant(v){
		var_helper = v;
		return v;
	}
	
	focus: true
	Keys.onPressed: {
		switch(event.key){
			case Qt.Key_Left:
			case Qt.Key_Up:
			case Qt.Key_F8:
				if(helpScreen.visible)
					return;
				prev.clicked();
				event.accepted = true;
				break;
			case Qt.Key_Right:
			case Qt.Key_Down:
			case Qt.Key_F7:
				if(helpScreen.visible)
					return;
				next.clicked();
				event.accepted = true;
				break;
			case Qt.Key_B://B,N are neighbours in many western keyboards
				first.clicked();
				break;
			case Qt.Key_N:
				last.clicked();
				break;
			case Qt.Key_C:
				colToggle.toggleState();
				break;
			case Qt.Key_T:
				txtToggle.toggleState();
				break;
			case Qt.Key_S:
				flickSel.clicked();
				break;
			case Qt.Key_L:
				linkHintToggle.toggleState();
				break;
			case Qt.Key_Equal:
			case Qt.Key_Plus:
				zoom_p.clicked();
				break;
			case Qt.Key_Minus:
				zoom_m.clicked();
				break;
			case Qt.Key_0://the number
				zoom_0.clicked();
				break;
			case Qt.Key_M:
				flickma.clicked({});
				break;
			case Qt.Key_H:
				if(helpScreen.visible) 
					helpScreen.close();
				else
					showHelp.clicked();
				break;
			case Qt.Key_O://the letter
				if(fileChooser.visible) 
					fileChooser.close();
				else
					choose.clicked();
				break;
			case Qt.Key_F:
				fullsc.clicked();
				break;
			case Qt.Key_Q:
				if(event.modifiers & Qt.ControlModifier)
					Qt.quit();
				break;
			case Qt.Key_1:
			case Qt.Key_2:
			case Qt.Key_3:
			case Qt.Key_4:
			case Qt.Key_5:
			case Qt.Key_6:
			case Qt.Key_7:
			case Qt.Key_8:
			case Qt.Key_9:
				openPreviousDoc(event.key - Qt.Key_0);
				if(fileChooser.visible) 
					fileChooser.close();
				break;
		}
// 		console.log(event.key)
	}

	Splash {}
	Help {id:helpScreen}

	Flickable {
		id: flick
		anchors.fill: parent;
		contentWidth: pdf.width
		contentHeight: pdf.height
		flickableDirection: Flickable.VerticalFlick
		boundsBehavior: Flickable.StopAtBounds
		property bool columnMode : colToggle.bstate;
		onColumnModeChanged: {
			if(columnMode) {
				curDocSet("lastMode","colMode");
				pdf.zoom *= 0.95*width/pdf.curRegion().width;//redraws page=>curRegion changes  
				var r = pdf.curRegion();
				contentX=r.x-10;
				contentY=r.y; 
			} else if (!pdf.textMode)
				curDocSet("lastMode","");
		}
		
		Component.onCompleted: {
			if(!sethist.curDoc.filepath)
				return;
			pdf.loadFile(sethist.curDoc.filepath);
			restoreSettings();
		}
		
		onMovementEnded: {
			if(columnMode)
				curDocSet("lastContentY",contentY);
		}
		
		onMovementStarted : {
			pdf.stopClickTimer();
		}
		
		MouseArea {
			id: flickma
			anchors.fill: parent
			onDoubleClicked: clicked()
			onClicked: {
				if(mainRec.state == "")
					mainRec.state = "menus";
				else
					mainRec.state = "";
			}
		}
		
		PdfImage {
			id: pdf
			highlightPhrase: searchWhat.text
			caseSensitive: caseSensitiveToggle.bstate //state as bool
			textMode: txtToggle.bstate
			linkHints: linkHintToggle.bstate
			onTextModeChanged: {
				if(textMode) 
					curDocSet("lastMode","textMode")
				else if(!flick.columnMode)
					curDocSet("lastMode","");
			}
			onPagenumChanged: {
				curDocSet("lastPage",pn);
				if(!flick.columnMode)/*flick.contentX=*/flick.contentY=0;
			}
			onZoomChanged: {
				curDocSet("lastZoom",zm);
				if(flick.columnMode){
					var r = pdf.curRegion(); 
					flick.contentX=r.x-10;
					//flick.contentY=r.y
				}
			}
			onClickForward: if(!flick.moving) flickma.clicked({});
			onLinkSelected: {
				if(typeof destination == "number"){
					pdf.pagenum = destination > pdf.totalPages ? pdf.totalPages : destination;
					//TODO:dirty (doesn't really belong here)
					flick.columnModeChanged();
				}
			}
		}
		
		function prevRegion_orUp(){
			var r = pdf.curRegion();
			if(contentY > r.y){
				contentY -= 0.8 * flick.height;
				curDocSet("lastContentY",contentY);
			} else {
				r = pdf.prevRegion();
				pdf.zoom *= 0.95*width/r.width//redraws page=>curRegion changes
				r = pdf.curRegion();
				curDocSet("lastRegionIdx",pdf.regionIdx);
				contentX=r.x-10;
				contentY=r.y+r.height-0.9*height;
				curDocSet("lastContentY",contentY);
			}
		}
		function nextRegion_orDown(){
			var r = pdf.curRegion();
			if(contentY + height < r.y + r.height){
				contentY += 0.8 * height;
				curDocSet("lastContentY",contentY);
			} else {
				r = pdf.nextRegion();
				pdf.zoom *= 0.95*width/r.width//redraws page=>curRegion changes
				r = pdf.curRegion();
				curDocSet("lastRegionIdx",pdf.regionIdx);
				contentX=r.x-10;
				contentY=r.y;
				curDocSet("lastContentY",contentY);
			}
		}
	}
	
	Rectangle{
		width: parent.width
		color: Qt.rgba(0, 0, 0, 0.1)
		anchors.top: parent.top
		height: flick.height/40
		MouseArea {
			id: ma_prev
			anchors.fill: parent;
			onClicked: {
				prev.clicked();
			}
		}
	}
	
	Rectangle{
		width: parent.width
		color: Qt.rgba(0, 0, 0, 0.1)
		anchors.bottom: parent.bottom
		height: flick.height/40
		MouseArea {
			id: ma_next
			anchors.fill: parent;
			onClicked: {
				next.clicked();
			}
		}
	}

	
	Flow {
		id: lwrMenu
		z : 1
		visible: false
		spacing: 5
		anchors.left: parent.left
		anchors.leftMargin: 20
		anchors.right: parent.right
		anchors.rightMargin: 20
		anchors.bottom: parent.bottom
		anchors.bottomMargin: 30
		
		InputField {
			id: pageIpt
			height: mainRec.refHeight
			width:40
			text: pdf.pagenum
			onAccepted: {
				pdf.pagenum = pageIpt.text > pdf.totalPages ? pdf.totalPages : pageIpt.text;
				//TODO:dirty (doesn't really belong here)
				flick.columnModeChanged();
			}
			onFocusLost: {mainRec.focus = true}
		}
		
		Button {
			id : first
			text: "|<"
			width: mainRec.refWidth
			onClicked: {pdf.pagenum = 1;}
		}
		Button {
			id : prev
			width: mainRec.refWidth
			text: "<"
			onClicked: { 
				if(flick.columnMode)
					flick.prevRegion_orUp();
				else 
					pdf.pagenum = pdf.pagenum > 1 ? pdf.pagenum - 1 : pdf.pagenum;
			}
		}
		Button {
			id : next
			text: ">"
			width: mainRec.refWidth
			onClicked: { 
				if(flick.columnMode)
						flick.nextRegion_orDown();
				else 
					pdf.pagenum = pdf.pagenum + 1;
			}
		}
		Button {
			id : last
			width: mainRec.refWidth
			text: ">|"
			onClicked: {pdf.pagenum = pdf.totalPages}
		}
		Button {
			id : zoom_p
			text: "z+"
			onClicked: {pdf.zoom = pdf.zoom * 1.03}
		}
		Button {
			id : zoom_0
			text: "z0"
			onClicked: {pdf.zoom = 100}
		}
		Button {
			id : zoom_m
			text: "z-"
			onClicked: {pdf.zoom = pdf.zoom / 1.03}
		}
		InputField {
			id: zoomIpt
			height: mainRec.refHeight
			width:60
			text: pdf.zoom
			onAccepted: {if(text > 50) pdf.zoom = text}
			onFocusLost: {mainRec.focus = true}
		}
		Button {
			id: colToggle
			text: "  col\nmode"
			width: mainRec.refWidth
			height: mainRec.refHeight
			pointSize: mainRec.refPtSize / 2;
			onClicked: toggleState();
			mutex: txtToggle.bstate;
		}
		Button {
			id: txtToggle
			text: " text\nmode"
			width: mainRec.refWidth
			height: mainRec.refHeight
			pointSize: mainRec.refPtSize / 2;
			onClicked: toggleState();
			mutex: colToggle.bstate;
		}
		Button {
			id : close_btn
			text: "quit"
			onClicked: Qt.quit();
		}
	}
	
	Flow {
		id: uprMenu
		z : 1
		visible: false
		spacing: 5
		anchors.left: parent.left
		anchors.leftMargin: 20
		anchors.right: parent.right
		anchors.rightMargin: 20
		anchors.top: parent.top
		anchors.topMargin: 10
		
		
		Button {
			id: choose
			text: "File"
			onClicked: {fileChooser.visible = true}
		}
		Button {
			id: searchPrev
			text: "<?"
			onClicked: pdf.searchPrev(searchWhat.text);
		}
		Button {
			id: caseSensitiveToggle
			text: "a  \n  A"
			height: mainRec.refHeight
			pointSize: searchPrev.pointSize / 2;
			onClicked: toggleState();
		}
		Button {
			id: searchNext
			text: "?>"
			onClicked: pdf.searchNext(searchWhat.text);
		}
		InputField {
			id: searchWhat
			height: searchPrev.height
			onFocusLost: {mainRec.focus = true}
			onAccepted: searchNext.clicked();
		}
		Button {
			id: flickSel
			text: "|"
			width: mainRec.refWidth
			onClicked: {
				switch(text){
					case "|" : text = "-";flick.flickableDirection = Flickable.HorizontalFlick;break;
					case "-" : text = "+";flick.flickableDirection = Flickable.HorizontalAndVerticalFlick;break;
					case "+" : text = "0";flick.interactive=false;break;
					case "0" : text = "|";flick.interactive=true;flick.flickableDirection = Flickable.VerticalFlick;break;
				};
			}
		}
		Button {
			id: linkHintToggle
			text: "show\nlinks"
			width: mainRec.refWidth
			height: mainRec.refHeight
			pointSize: mainRec.refPtSize / 2;
			onClicked: toggleState();
		}
		Button {
			id: fullsc
			text: "[]"
			width: mainRec.refWidth
			onClicked: {
				toggleState();
				if(state == "active")
					mainRec.showFullScreen();
				else 
					mainRec.showNormal();
			}
		}
		Button {
			id: showHelp
			text: "help"
			width: mainRec.refWidth
			height: mainRec.refHeight
			pointSize: mainRec.refPtSize * 2/3;
			onClicked: {helpScreen.visible = true}
		}
		
	}
	
	states: [
	State {
		name: "menus"
		
		PropertyChanges {target: lwrMenu; visible: true }
		PropertyChanges {target: uprMenu; visible: true }
		PropertyChanges {target: ma_prev; enabled: false }
		PropertyChanges {target: ma_next; enabled: false }
	}
	]
}
