/*
 *  Flashlight applet (widget) for Maemo.
 *  Copyright (C) 2009 Roman Moravcik
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/mman.h>

#include <asm/types.h>
#include <linux/videodev2.h>

#include "flashlight_lib.h"


int flashlight_get_status (FlashlightContext_t *flashlight, int *status)
{
	struct v4l2_control ctrl;

	printf ("flashlight_get_status()\n");

	if (flashlight == NULL) {
		printf ("flashlight_get_status: flashlight context is not valid\n");
		return ENOCONTEXT;
	}

	if (flashlight->fd == -1) {
		printf ("flashlight_get_status: device not openned\n");
		return ENODEVICE;
	}

	*status = 0;

	/* check short circuit fault */
	ctrl.id = V4L2_CID_FLASH_ADP1653_FAULT_SCP;
	if (ioctl (flashlight->fd, VIDIOC_G_CTRL, &ctrl) == -1) {
		printf ("flashlight_set_intensity: cannot get circuit fault status (%s)\n", strerror (errno));
		return EGENERROR;
	}

	if (ctrl.value)
		*status |= FLASHLIGHT_STATUS_SHORT_CIRCUT_FAULT;
	else
		*status &= ~FLASHLIGHT_STATUS_SHORT_CIRCUT_FAULT;

	/* check overtemperature fault */
	ctrl.id = V4L2_CID_FLASH_ADP1653_FAULT_OT;
	if (ioctl (flashlight->fd, VIDIOC_G_CTRL, &ctrl) == -1) {
		printf ("flashlight_set_intensity: cannot get overtemperature fault status (%s)\n", strerror (errno));
		return EGENERROR;
	}

	if (ctrl.value)
		*status |= FLASHLIGHT_STATUS_OVERTEMPERATURE_FAULT;
	else
		*status &= ~FLASHLIGHT_STATUS_OVERTEMPERATURE_FAULT;

	/* check timeout fault */
	ctrl.id = V4L2_CID_FLASH_ADP1653_FAULT_TMR;
	if (ioctl (flashlight->fd, VIDIOC_G_CTRL, &ctrl) == -1) {
		printf ("flashlight_set_intensity: cannot get timeout fault status (%s)\n", strerror (errno));
		return EGENERROR;
	}

	if (ctrl.value)
		*status |= FLASHLIGHT_STATUS_TIMEOUT_FAULT;
	else
		*status &= ~FLASHLIGHT_STATUS_TIMEOUT_FAULT;

	/* check overtemperature fault */
	ctrl.id = V4L2_CID_FLASH_ADP1653_FAULT_OV;
	if (ioctl (flashlight->fd, VIDIOC_G_CTRL, &ctrl) == -1) {
		printf ("flashlight_set_intensity: cannot get overvoltage fault status (%s)\n", strerror (errno));
		return EGENERROR;
	}

	if (ctrl.value)
		*status |= FLASHLIGHT_STATUS_OVERVOLTAGE_FAULT;
	else
		*status &= ~FLASHLIGHT_STATUS_OVERVOLTAGE_FAULT;

	return ENOERROR;
}

int flashlight_set_intensity (FlashlightContext_t *flashlight, int intensity)
{
	struct v4l2_control ctrl;
	enum v4l2_buf_type type;
	unsigned int i;

	printf ("flashlight_set_intensity(%d)\n", intensity);

	if (flashlight == NULL) {
		printf ("flashlight_set_intensity: flashlight context is not valid\n");
		return ENOCONTEXT;
	}

	if (flashlight->fd == -1) {
		printf ("flashlight_set_intensity: device not openned\n");
		return ENODEVICE;
	}

	if (intensity > flashlight->max_intensity)
		intensity = flashlight->max_intensity;

	ctrl.id = V4L2_CID_TORCH_INTENSITY;
	ctrl.value = intensity;

	if (ioctl (flashlight->fd, VIDIOC_S_CTRL, &ctrl) == -1) {
		printf ("flashlight_set_intensity: cannot set intensity (%s)\n", strerror (errno));
		return EGENERROR;
	}

	/*
	   WORKAROUND: start/stop i/o streaming to block camera application
	 */
	if (intensity > 0) {
		for (i = 0; i < flashlight->n_buffers; ++i) {
			struct v4l2_buffer buf;

			buf.type	= V4L2_BUF_TYPE_VIDEO_CAPTURE;
			buf.memory	= V4L2_MEMORY_MMAP;
			buf.index	= i;
			if (ioctl (flashlight->fd, VIDIOC_QBUF, &buf) == -1) {
				printf ("flashlight_set_intensity: unable to exchange a buffer %d with driver (%s)\n", i, strerror (errno));
				return EGENERROR;
			}
		}

		type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		if (ioctl (flashlight->fd, VIDIOC_STREAMON, &type)) {
			printf ("flashlight_set_intensity: unable to start i/o streaming (%s)\n", strerror (errno));
			return EGENERROR;
		}
	} else {
		type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		if (ioctl (flashlight->fd, VIDIOC_STREAMOFF, &type) == -1) {
			printf ("flashlight_set_intensity: unable to stop i/o streaming (%s)\n", strerror (errno));
			return EGENERROR;
		}
	}

	return ENOERROR;
}

int flashlight_get_intensity (FlashlightContext_t *flashlight, int *intensity)
{
	struct v4l2_control ctrl;

	printf ("flashlight_get_intensity()\n");

	if (flashlight == NULL) {
		printf ("flashlight_get_intensity: flashlight context is not valid\n");
		return ENOCONTEXT;
	}

	if (flashlight->fd == -1) {
		printf ("flashlight_get_intensity: device not openned\n");
		return ENODEVICE;
	}

	ctrl.id = V4L2_CID_TORCH_INTENSITY;

	if (ioctl (flashlight->fd, VIDIOC_G_CTRL, &ctrl) == -1) {
		printf ("flashlight_get_intensity: cannot get intensity (%s)\n", strerror (errno));
		return EGENERROR;
	}

	*intensity = ctrl.value;
	return ENOERROR;
}

int flashlight_open (FlashlightContext_t *flashlight, const char *device_name)
{
	struct v4l2_queryctrl ctrl;
	struct v4l2_cropcap cropcap;
	struct v4l2_crop crop;
	struct v4l2_format fmt;
	struct v4l2_requestbuffers req;
	struct stat st;

	printf ("flashlight_open(%s)\n", device_name);

	if (flashlight == NULL) {
		printf ("flashlight_open: flashlight context is not valid\n");
		return ENOCONTEXT;
	}

	if (device_name == NULL) {
		printf ("flashlight_open: device name not specified\n");
		return EGENERROR;
	}

	memcpy (flashlight->device_name, device_name, sizeof(flashlight->device_name));

	if (stat (flashlight->device_name, &st) == -1) {
		printf ("flashlight_open: cannot identify '%s' (%s)\n", flashlight->device_name, strerror (errno));
		return EGENERROR;
	}

	/* check it device_name is real device */
	if (!S_ISCHR (st.st_mode)) {
		printf ("flashlight_open: %s is no device\n", flashlight->device_name);
		return EGENERROR;
	}

	flashlight->fd = open (flashlight->device_name, O_RDWR /* required */ | O_NONBLOCK, 0);

	if (flashlight->fd == -1) {
		printf ("flashlight_open: cannot open '%s' (%s)\n", flashlight->device_name, strerror (errno));
		return ENODEVICE;
	}

	/* query from driver minimal and maximal flashlight intensity */
	ctrl.id = V4L2_CID_TORCH_INTENSITY;
	if (ioctl (flashlight->fd, VIDIOC_QUERYCTRL, &ctrl) == -1) {
		printf ("flashlight_open: cannot get minimal and maximal flashlight intensity (%s)\n", strerror (errno));
		return EGENERROR;
	}

	flashlight->min_intensity = ctrl.minimum;
	flashlight->max_intensity = ctrl.maximum;

	/*
	   WORKAROUND: Initialization of camera extracted from v4l2_example.
	   http://v4l2spec.bytesex.org/spec/capture-example.html

	   We need to initialize camera in other to block camera application.
	   (bug 4949: Applet breaks the camera application)
	 */
	/* get crop capabilities */
	cropcap.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	if (ioctl (flashlight->fd, VIDIOC_CROPCAP, &cropcap) == -1) {
		printf ("flashlight_open: unable to get crop capabilities (%s)\n", strerror (errno));
		return EGENERROR;
	}

	/* set crop capabilities */
	crop.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	crop.c = cropcap.defrect; /* reset to default */
	if (ioctl (flashlight->fd, VIDIOC_S_CROP, &crop) == -1) {
		printf ("flashlight_open: unable to set cropping rectangle (%s)\n", strerror (errno));
		return EGENERROR;
	}

	/* set data format */
	fmt.type		= V4L2_BUF_TYPE_VIDEO_CAPTURE;
	fmt.fmt.pix.width	= 640;
	fmt.fmt.pix.height	= 480;
	fmt.fmt.pix.pixelformat	= V4L2_PIX_FMT_YUYV;
	fmt.fmt.pix.field	= V4L2_FIELD_INTERLACED;
	if (ioctl (flashlight->fd, VIDIOC_S_FMT, &fmt) == -1) {
		printf ("flashlight_open: unable to set data format (%s)\n", strerror (errno));
		return EGENERROR;
	}

	req.count	= 4;
	req.type	= V4L2_BUF_TYPE_VIDEO_CAPTURE;
	req.memory	= V4L2_MEMORY_MMAP;
	if (ioctl (flashlight->fd, VIDIOC_REQBUFS, &req) == -1) {
		printf ("flashlight_open: unable to initiate memory mapping (%s)\n", strerror (errno));
		return EGENERROR;
	}

	if (req.count < 2) {
		printf ("flashlight_open: insufficient buffer memory on %s\n", device_name);
		return EGENERROR;
	}

	flashlight->buffers = calloc (req.count, sizeof (*flashlight->buffers));
	if (!flashlight->buffers) {
		printf ("flashlight_open: unable to allocate memory\n");
		return EGENERROR;
	}

	for (flashlight->n_buffers = 0; flashlight->n_buffers < req.count; ++flashlight->n_buffers) {
		struct v4l2_buffer buf;

		buf.type	= V4L2_BUF_TYPE_VIDEO_CAPTURE;
		buf.memory	= V4L2_MEMORY_MMAP;
		buf.index	= flashlight->n_buffers;

		if (ioctl (flashlight->fd, VIDIOC_QUERYBUF, &buf) == -1) {
			printf ("flashlight_open: unable to query the status of a buffer %d (%s)\n",
				flashlight->n_buffers, strerror (errno));
			return EGENERROR;
		}

		flashlight->buffers[flashlight->n_buffers].length = buf.length;
		flashlight->buffers[flashlight->n_buffers].start = mmap (NULL /* start anywhere */,
									 buf.length,
									 PROT_READ | PROT_WRITE /* required */,
									 MAP_SHARED /* recommended */,
									 flashlight->fd,
									 buf.m.offset);

		if (flashlight->buffers[flashlight->n_buffers].start == MAP_FAILED) {
			printf ("flashlight_open: unable to map memory (%s)\n", strerror (errno));
			return EGENERROR;
		}
	}

	return ENOERROR;
}

int flashlight_close (FlashlightContext_t *flashlight)
{
	unsigned int i;

	printf ("flashlight_close()\n");

	if (flashlight == NULL) {
		printf ("flashlight_close: flashlight context is not valid\n");
		return ENOCONTEXT;
	}

	/* unmap memory mapped buffers */
	for (i = 0; i < flashlight->n_buffers; ++i) {
		if (flashlight->buffers[flashlight->n_buffers].start != MAP_FAILED) {
			if (munmap (flashlight->buffers[i].start, flashlight->buffers[i].length) == -1) {
				printf ("flashlight_close: unable to unmap memory (%s)\n", strerror (errno));
				return EGENERROR;
			}
		}
	}
	flashlight->n_buffers = 0;

	/* free buffers */
	if (flashlight->buffers)
		free (flashlight->buffers);
	flashlight->buffers = NULL;

	/* close camera device */
	if (flashlight->fd != -1) {
		if (close (flashlight->fd) == -1) {
			printf ("flashlight_close: cannot close device '%s' (%s)\n", flashlight->device_name, strerror (errno));
			return ENODEVICE;
		}
	}
	flashlight->fd = -1;

	return ENOERROR;
}

int flashlight_init (FlashlightContext_t **pRefContext)
{
	FlashlightContext_t *flashlight = NULL;

	printf ("flashlight_init()\n");

	if (*pRefContext != NULL) {
		printf("flashlight_init: expecting zero pointer context '*pRefContext'\n");
		return EGENERROR;
	}

	/* allocate memory for context structure */
	flashlight = malloc (sizeof (FlashlightContext_t));
	if (flashlight == NULL) {
		printf ("flashlight_init: unable to allocate memory for context\n");
		return ENOCONTEXT;
	}

	*pRefContext = flashlight;

	/* initialize default values */
	memset (flashlight, 0x00, sizeof (FlashlightContext_t));
	flashlight->fd = -1;

	flashlight->n_buffers = 0;
	flashlight->buffers = NULL;

	/* from adp1653.c */
	flashlight->min_intensity = 0;
	flashlight->max_intensity = 11;

	return ENOERROR;
}

int flashlight_deinit (FlashlightContext_t *flashlight)
{
	int intensity = 0;

	printf ("flashlight_deinit()\n");

	if (flashlight == NULL) {
		printf ("flashlight_deinit: flashlight context is not valid\n");
		return ENOCONTEXT;
	}

	if (flashlight->fd != -1) {
		/* check if flashlight isn't enabled before closing device */
		if (flashlight_get_intensity (flashlight, &intensity) == -1)
			return EGENERROR;

		if (intensity > 0) {
			if (flashlight_set_intensity (flashlight, 0) == -1)
				return EGENERROR;
		}

		if (flashlight_close(flashlight))
			return EGENERROR;
	}

	/* free allocated memory */
	free (flashlight);

	return ENOERROR;
}
