/*
  This source is part of the FindMine program.
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <iostream>

#include <Lum/OS/Probe.h>

#include <Lum/Base/L10N.h>
#include <Lum/Base/Object.h>
#include <Lum/Base/String.h>
#include <Lum/Base/Util.h>

#include <Lum/Def/Menu.h>

#include <Lum/Dlg/About.h>
#include <Lum/Dlg/ActionDialog.h>
#include <Lum/Dlg/Help.h>
#include <Lum/Dlg/Msg.h>

#include <Lum/Model/Action.h>
#include <Lum/Model/Boolean.h>
#include <Lum/Model/Number.h>
#include <Lum/Model/String.h>
#include <Lum/Model/Table.h>

#include <Lum/Boolean.h>
#include <Lum/Dialog.h>
#include <Lum/Grid.h>
#include <Lum/Label.h>
#include <Lum/Slider.h>
#include <Lum/String.h>
#include <Lum/Table.h>
#include <Lum/Text.h>
#include <Lum/TextValue.h>
#include <Lum/Toggle.h>
#include <Lum/Panel.h>
#include <Lum/Space.h>
#include <Lum/StatusLine.h>
#include <Lum/View.h>

#include "config.h"
#include "Configuration.h"
#include "Control.h"

static Lum::Def::AppInfo info;

typedef Lum::Model::StdRefTable<Hiscore::Entry,std::vector<Hiscore::Entry> > HiscoreEntriesModel;
typedef Lum::Base::Reference<HiscoreEntriesModel>                            HiscoreEntriesModelRef;

class HiscoreEntriesModelPainter : public Lum::TableCellPainter
{
public:
  void Draw(Lum::OS::DrawInfo* draw,
            int x, int y, size_t width, size_t height) const
  {
    const Hiscore::Entry entry=dynamic_cast<const HiscoreEntriesModel*>(GetModel())->GetEntry(GetRow());

    //
    // Name of User
    //

    DrawStringLeftAligned(draw,
                          x,y,width,height,
                          entry.name);


    //
    //
    // Time in seconds

    std::wstring time=Lum::Base::NumberToWString(entry.time);

    if (entry.time==1) {
      time+=L" second";
    }
    else {
      time+=L" seconds";
    }

    DrawStringRightAligned(draw,
                           x,y,width,height,
                           time);
  }
};

class ScoreView : public Lum::Dlg::ActionDialog
{
private:
  HiscoreEntriesModelRef list;

public:
  ScoreView(Sweeper* sweeper)
  {
    Hiscore *score;

    score=GetConfig(sweeper->GetAreaWidth(),
                    sweeper->GetAreaHeight(),
                    sweeper->GetMines());

    list=new HiscoreEntriesModel(score->list);
    list->SetEmptyText(L"(No hiscores)");
  }

  Lum::Object* GetContent()
  {
    Lum::Panel            *panel;
    Lum::Table            *table;
    Lum::Model::HeaderRef headerModel;

    panel=Lum::VPanel::Create(true,true);

    headerModel=new Lum::Model::HeaderImpl();
    headerModel->AddColumn(L"Score",Lum::Base::Size::stdCharWidth,26,true);

    table=new Lum::Table();
    table->SetFlex(true,true);
    table->SetShowHeader(false);
    table->GetTableView()->SetAutoFitColumns(true);
    table->GetTableView()->SetAutoHSize(true);
    table->GetTableView()->SetAutoVSize(true);
    table->SetModel(list);
    table->SetHeaderModel(headerModel);
    table->SetPainter(new HiscoreEntriesModelPainter());
    panel->Add(table);

    return panel;
  }

  void GetActions(std::vector<Lum::Dlg::ActionInfo>& actions)
  {
    Lum::Dlg::ActionDialog::CreateActionInfosClose(actions,GetClosedAction());
  }
};

class ScoreEdit : public Lum::Dlg::ActionDialog
{
private:
  Lum::Model::ActionRef closeAction;
  Lum::Model::StringRef name;
  size_t                pos;
  Hiscore               *score;

public:
  ScoreEdit(Sweeper* sweeper, size_t pos)
  : closeAction(new Lum::Model::Action()),
    name(new Lum::Model::String()),
    pos(pos)
  {
    score=GetConfig(sweeper->GetAreaWidth(),
                    sweeper->GetAreaHeight(),
                    sweeper->GetMines());

    Observe(closeAction);

    name->Set(score->list[pos].name);
    Observe(name);
  }

  Lum::Object* GetContent()
  {
    Lum::Grid      *grid;
    Lum::Panel     *panel;
    Lum::Text      *text;

    panel=Lum::VPanel::Create(true,true);

    grid=new Lum::Grid();
    grid->SetSpace(true,true);

    for (size_t x=0; x<10; x++) {
      if (x==pos) {
        text=new Lum::Text();
        text->SetFlex(true,true);
        text->SetText(Lum::Base::NumberToWString((unsigned long)x+1)+L":");
        grid->SetObject(0,x,text);

        grid->SetObject(1,x,Lum::String::Create(name,10,true,true));

        text=new Lum::Text();
        text->SetAlignment(Lum::Text::left);
        text->SetFlex(true,true);
        text->SetText(Lum::Base::NumberToWString((unsigned long)score->list[x].time)+L" second(s)");
        grid->SetObject(2,x,text);
      }
      else if (x<score->list.size() && (x+1==pos|| x+2==pos || x==pos+1 || x==pos+2)) {
        text=new Lum::Text();
        text->SetFlex(true,true);
        text->SetText(Lum::Base::NumberToWString((unsigned long)x+1)+L":");
        grid->SetObject(0,x,text);

        text=new Lum::Text();
        text->SetFlex(true,true);
        text->SetAlignment(Lum::Text::left);
        text->SetText(score->list[x].name);
        grid->SetObject(1,x,text);

        text=new Lum::Text();
        text->SetAlignment(Lum::Text::left);
        text->SetFlex(true,true);
        text->SetText(Lum::Base::NumberToWString((unsigned long)score->list[x].time)+L" second(s)");
        grid->SetObject(2,x,text);
      }
    }

    panel->Add(grid);

    return panel;
  }

  void GetActions(std::vector<Lum::Dlg::ActionInfo>& actions)
  {
    Lum::Dlg::ActionDialog::CreateActionInfosClose(actions,GetClosedAction());
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==closeAction &&  closeAction->IsFinished()) {
      Exit();
    }
    else if (model==name) {
      score->list[pos].name=name->Get();
      SetUserName(name->Get());
    }
    else {
      Dialog::Resync(model,msg);
    }
  }
};

typedef Lum::Model::StdRefTable<Theme,std::vector<Theme> > ThemesModel;
typedef Lum::Base::Reference<ThemesModel>                  ThemesModelRef;

class ThemesComparator : public ThemesModel::Comparator
{
public:
  bool operator()(const Theme& a,
                  const Theme& b,
                  size_t column,
                  bool ascending) const
  {
    assert(column==2);

    if (ascending) {
      return a.boxSize<b.boxSize;
    }
    else {
      return a.boxSize>b.boxSize;
    }
  }
};

class ThemesModelPainter : public Lum::TableCellPainter
{
public:
  void Draw(Lum::OS::DrawInfo* draw,
            int x, int y, size_t width, size_t height) const
  {
    const Theme theme=dynamic_cast<const ThemesModel*>(GetModel())->GetEntry(GetRow());

    //
    // Name of theme
    //

    DrawStringLeftAligned(draw,
                          x,y,width,height,
                          theme.name);

    //
    //
    // Size of theme

    std::wstring size=Lum::Base::NumberToWString(theme.boxSize)+L" pixel";

    DrawStringRightAligned(draw,
                           x,y,width,height,
                           size);
  }
};

class SelectTheme : public Lum::Dlg::ActionDialog
{
private:
  Lum::Model::ActionRef              okAction;
  ThemesModelRef                     list;
  Lum::Model::SingleLineSelectionRef selection;
  std::wstring                       result;

public:
  SelectTheme()
  : okAction(new Lum::Model::Action),
    list(new ThemesModel(themes,new ThemesComparator())),
    selection(new Lum::Model::SingleLineSelection),
    result(L"")
  {
    list->Sort(2);
    list->SetEmptyText(L"(No themes found)");

    okAction->Disable();

    Observe(okAction);
    Observe(selection);
  }

  Lum::Object* GetContent()
  {
    Lum::Panel            *panel;
    Lum::Table            *table;
    Lum::Model::HeaderRef headerModel;

    panel=Lum::VPanel::Create(true,true);

    headerModel=new Lum::Model::HeaderImpl();
    headerModel->AddColumn(L"Theme",Lum::Base::Size::stdCharWidth,20,true);

    table=new Lum::Table();
    table->SetFlex(true,true);
    table->SetShowHeader(false);
    table->GetTableView()->SetAutoFitColumns(true);
    table->SetModel(list);
    table->SetHeaderModel(headerModel);
    table->SetSelection(selection);
    table->SetDefaultAction(okAction);
    table->SetPainter(new ThemesModelPainter());
    table->GetTableView()->SetAutoFitColumns(true);
    table->GetTableView()->SetAutoHSize(true);
    table->GetTableView()->SetAutoVSize(true);
    panel->Add(table);

    return panel;
  }

  void GetActions(std::vector<Lum::Dlg::ActionInfo>& actions)
  {
    Lum::Dlg::ActionDialog::CreateActionInfosOkCancel(actions,okAction,GetClosedAction());
  }

  void Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==okAction && okAction->IsFinished()) {
      if (selection->HasSelection()) {
        result=themes[selection->GetLine()-1].name;
      }

      Exit();
    }
    else if (model==selection) {
      if (selection->HasSelection()) {
        okAction->Enable();
      }
      else {
        okAction->Disable();
      }
    }

    Dialog::Resync(model,msg);
  }

  std::wstring GetResult() const
  {
    return result;
  }
};

class Settings : public Lum::Dlg::ActionDialog
{
private:
  Lum::Model::ActionRef  smallAction;
  Lum::Model::ActionRef  mediumAction;
  Lum::Model::ActionRef  bigAction;
  Lum::Model::ActionRef  themeAction;

public:
  Lum::Model::ULongRef   horiz;
  Lum::Model::ULongRef   vert;
  Lum::Model::ULongRef   mines;
  Lum::Model::BooleanRef oneButtonMode;
  Lum::Model::StringRef  themeName;

public:
  Settings(Sweeper* sweeper)
  : smallAction(new Lum::Model::Action()),
    mediumAction(new Lum::Model::Action()),
    bigAction(new Lum::Model::Action()),
    themeAction(new Lum::Model::Action()),
    horiz(new Lum::Model::ULong(sweeper->GetAreaWidth(),1,30)),
    vert(new Lum::Model::ULong(sweeper->GetAreaHeight(),1,30)),
    mines(new Lum::Model::ULong(sweeper->GetMinesPercent(),0,100)),
    oneButtonMode(new Lum::Model::Boolean(GetOneButtonMode())),
    themeName(new Lum::Model::String())
  {
    Observe(smallAction);
    Observe(mediumAction);
    Observe(bigAction);
    Observe(themeAction);

    themeName->Set(::themeName);
  }

  Lum::Object* GetContent()
  {
    Lum::Label        *label;
    Lum::Panel        *panel,*hPanel,*vPanel;
    Lum::Slider       *slider;
    Lum::TextValue    *textValue;
    /*Lum::Def::Boolean oneButtonModeDef(Lum::Def::Desc(L"One button mode"),
                                       oneButtonMode);*/

    panel=Lum::HPanel::Create(true,false);

    label=Lum::Label::Create(true,false);

    slider=new Lum::HSlider();
    slider->SetFlex(true,false);
    slider->SetScale(true);
    slider->SetModel(horiz);
    label->AddLabel(L"Horizontal:",slider);

    slider=new Lum::HSlider();
    slider->SetFlex(true,false);
    slider->SetScale(true);
    slider->SetModel(vert);
    label->AddLabel(L"Vertical:",slider);

    slider=new Lum::HSlider();
    slider->SetFlex(true,false);
    slider->SetModel(mines);
    label->AddLabel(L"Mines (in %):",slider);

    if (!Lum::OS::display->GetTheme()->RequestFingerFriendlyControls()) {
      label->AddLabel(L"One button mode:",Lum::Boolean::Create(oneButtonMode));
    }


/*
    label->AddLabel(L"One button mode:",
                    Lum::OS::display->GetBehaviour()->GetBooleanControl(oneButtonModeDef));*/

    hPanel=Lum::HPanel::Create(true,false);

    textValue=new Lum::TextValue();
    textValue->SetFlex(true,false);
    textValue->SetModel(themeName);
    hPanel->Add(textValue);

    hPanel->Add(Lum::Button::Create(L"...",themeAction,false,true));

    label->AddLabel(L"Theme:",hPanel);
    panel->Add(label);

    panel->AddSpace();

    vPanel=Lum::VPanel::Create(false,true);

    vPanel->Add(Lum::Button::Create(L"_Small",smallAction,true));
    vPanel->Add(Lum::Button::Create(L"_Medium",mediumAction,true));
    vPanel->Add(Lum::Button::Create(L"_Big",bigAction,true));
    vPanel->AddSpace(true);

    panel->Add(vPanel);

    return panel;
  }

  void GetActions(std::vector<Lum::Dlg::ActionInfo>& actions)
  {
    Lum::Dlg::ActionDialog::CreateActionInfosClose(actions,GetClosedAction());
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==smallAction && smallAction->IsFinished()) {
      horiz->Set(8);
      vert->Set(8);
      mines->Set(Lum::Base::RoundDiv(10*100,8*8));
    }
    else if (model==mediumAction && mediumAction->IsFinished()) {
      horiz->Set(16);
      vert->Set(16);
      mines->Set(Lum::Base::RoundDiv(40*100,16*16));
    }
    else if (model==bigAction && bigAction->IsFinished()) {
      horiz->Set(30);
      vert->Set(16);
      mines->Set(Lum::Base::RoundDiv(99*100,30*16));
    }
    else if (model==themeAction && themeAction->IsFinished()) {
      SelectTheme dialog;

      dialog.SetParent(this);
      if (dialog.Open()) {
        dialog.SetExitAction(dialog.GetClosedAction());
        dialog.EventLoop();
        dialog.Close();
      }

      if (!dialog.GetResult().empty()) {
        ::themeName=dialog.GetResult();
        themeName->Set(dialog.GetResult());
      }
    }
    else {
      Dialog::Resync(model,msg);
    }
  }
};

class MyWindow : public Lum::Dialog
{
private:
  Sweeper                *sweeper;
  Lum::Model::ActionRef  newGameAction;
  Lum::Model::BooleanRef zoom;
  Lum::Model::ActionRef  hiscoreAction;
  Lum::Model::ActionRef  settingsAction;
  Lum::Model::ActionRef  helpAction;
  Lum::Model::ActionRef  aboutAction;
  Lum::Model::ActionRef  timerAction;

  Lum::Text              *marks;
  Lum::Text              *time;

  size_t                 unzoomX;
  size_t                 unzoomY;

public:

  MyWindow()
  : sweeper(NULL),
    newGameAction(new Lum::Model::Action),
    zoom(new Lum::Model::Boolean(false)),
    hiscoreAction(new Lum::Model::Action),
    settingsAction(new Lum::Model::Action),
    helpAction(new Lum::Model::Action),
    aboutAction(new Lum::Model::Action),
    timerAction(new Lum::Model::Action)
  {
    Observe(newGameAction);
    Observe(zoom);
    Observe(hiscoreAction);
    Observe(settingsAction);
    Observe(helpAction);
    Observe(aboutAction);
    Observe(timerAction);

    Observe(GetOpenedAction());
  }

  void PreInit()
  {
    Lum::Panel       *vPanel,*hPanel;
    size_t           digitWidth,widestDigit;
    std::wstring     timeFormat,marksFormat;
    Lum::OS::FontRef font;

    vPanel=Lum::VPanel::Create(true,true);

    hPanel=Lum::HPanel::Create(true,false);

    hPanel->Add(Lum::Toggle::Create(L"Zoom out",zoom));
    hPanel->Add(Lum::Button::Create(L"_New",newGameAction));
    hPanel->Add(new Lum::HSpace(Lum::Space::sizeNormal,true));

    font=Lum::OS::display->GetFont(Lum::OS::Display::fontTypeFixed,210);

    digitWidth=0;
    widestDigit=0;
    for (size_t d=0; d<=9; d++) {
      size_t width;

      width=font->StringWidth(Lum::Base::NumberToWString(d));

      if (width>digitWidth) {
        digitWidth=width;
        widestDigit=d;
      }
    }

    marksFormat=L"-";
    for (size_t x=1; x<=3; x++) {
      marksFormat.append(Lum::Base::NumberToWString(widestDigit));
    }

    marks=new Lum::Text();
    marks->SetFont(font);
    marks->SetStyle(Lum::OS::Font::bold);
    marks->SetAlignment(Lum::Text::right);
    marks->SetMinWidth(Lum::Base::Size::pixel,font->StringWidth(marksFormat));
    marks->SetText(L"0");

    hPanel->Add(marks);

    hPanel->Add(new Lum::HSpace(Lum::Space::sizeNormal,true));

    for (size_t x=1; x<=4; x++) {
      timeFormat.append(Lum::Base::NumberToWString(widestDigit));
    }

    time=new Lum::Text();
    time->SetFont(font);
    time->SetStyle(Lum::OS::Font::bold);
    time->SetAlignment(Lum::Text::centered);
    time->SetMinWidth(Lum::Base::Size::pixel,font->StringWidth(timeFormat));
    time->SetText(L"0000");

    hPanel->Add(time);

    vPanel->Add(hPanel);
    vPanel->AddSpace();

    sweeper=new Sweeper();
    sweeper->SetFlex(true,true);
    Observe(sweeper->GetStatusModel());
    Observe(sweeper->GetMarksModel());

    vPanel->Add(Lum::View::Create(sweeper,true,true));

    SetMain(vPanel);

    Lum::Def::Menu *menu=Lum::Def::Menu::Create();

    menu
      ->GroupProject()
        ->Action(Lum::Def::Action(Lum::Def::Desc(L"_New Game")
                                  .SetShortcut(Lum::OS::qualifierControl,L"n"),
                                  newGameAction))
        ->Separator()
        ->Action(Lum::Def::Action(Lum::Def::Desc(L"_Hiscore"),hiscoreAction)
                 .SetOpensDialog())
        ->Separator()
        ->ActionQuit(GetClosedAction())
      ->End()
      ->GroupEdit()
        ->ActionSettings(settingsAction)
      ->End()
      ->GroupHelp()
        ->ActionHelp(helpAction)
        ->ActionAbout(aboutAction)
      ->End();

    SetMenu(menu);

    Dialog::PreInit();
  }

  size_t GetHiscorePos() const
  {
    Hiscore *score;
    size_t  pos;

    score=GetConfig(sweeper->GetAreaWidth(),
                    sweeper->GetAreaHeight(),
                    sweeper->GetMines());

    pos=0;
    while (pos<score->list.size() && sweeper->GetElapsedTime()>=score->list[pos].time) {
      pos++;
    }

    return pos;
  }

  void ShowSettings()
  {
    Settings settings(sweeper);

    settings.SetParent(this);
    if (settings.Open()) {
      settings.SetExitAction(settings.GetClosedAction());
      settings.EventLoop();
      settings.Close();

      if (sweeper->SetSize(settings.horiz->Get(),
                           settings.vert->Get(),
                           Lum::Base::RoundDiv(settings.horiz->Get()*
                                               settings.vert->Get()*
                                               settings.mines->Get(),
                                               100))) {
        SetCurrentGame(sweeper->GetAreaWidth(),
                       sweeper->GetAreaHeight(),
                       sweeper->GetMines());
        SetOneButtonMode(settings.oneButtonMode->Get());

        sweeper->SetOneButtonMode(settings.oneButtonMode->Get());
        sweeper->SetThemeName(themeName);
        sweeper->Run();
      }

    }
  }

  void ShowHiscore()
  {
    ScoreView score(sweeper);

    score.SetTitle(L"Hiscore");
    score.SetParent(this);
    if (score.Open()) {
      score.SetExitAction(score.GetClosedAction());
      score.EventLoop();
      score.Close();
    }
  }

  void ShowHiscoreEdit(size_t pos)
  {
    Hiscore::Entry                        entry;
    ScoreEdit                             *scoreDlg;
    Hiscore*                              score;
    std::vector<Hiscore::Entry>::iterator iter;

    score=GetConfig(sweeper->GetAreaWidth(),
                    sweeper->GetAreaHeight(),
                    sweeper->GetMines());

    entry.time=sweeper->GetElapsedTime();
    entry.name=GetUserName();

    iter=score->list.begin();
    iter+=pos;

    score->list.insert(iter,entry);

    scoreDlg=new ScoreEdit(sweeper,pos);
    scoreDlg->SetTitle(L"Hiscore");
    scoreDlg->SetParent(this);

    if (scoreDlg->Open()) {
      scoreDlg->SetExitAction(scoreDlg->GetClosedAction());
      scoreDlg->EventLoop();
      scoreDlg->Close();
    }

    delete scoreDlg;
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() && GetOpenedAction()->IsFinished()) {
      LoadThemes();

      if (!LoadConfig()) {
        std::cout << "Did not find a configuration, use default..." << std::endl;
      }

      size_t width,height,mines;

      GetCurrentGame(width,height,mines);
      sweeper->SetThemeName(themeName);
      sweeper->SetSize(width,height,mines);
      sweeper->SetOneButtonMode(GetOneButtonMode());
    }
    if (model==newGameAction && newGameAction->IsFinished()) {
      sweeper->SetThemeName(themeName);
      zoom->Set(false);
      sweeper->Run();
    }
    else if (model==zoom && sweeper!=NULL) {
      if (zoom->Get()) {
        sweeper->SetThemeName(zoomThemeName);
      }
      else {
        sweeper->SetThemeName(themeName);
      }
    }
    else if (model==hiscoreAction && hiscoreAction->IsFinished()) {
      ShowHiscore();
    }
    else if (model==settingsAction && settingsAction->IsFinished()) {
      ShowSettings();
    }
    else if (model==aboutAction && aboutAction->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }
    else if (model==helpAction && helpAction->IsFinished()) {
      Lum::Dlg::Help::Show(this,
                           L"This application has a two button mode, for devices\n"
                           L"that have a mouse with at least two buttons and a one\n"
                           L"button mode in case your mouse only has one button or\n"
                           L"you are using a touch screen (in this case you just\n"
                           L"touch instead of click) or similar. In case your device\n"
                           L"has two buttons, you can toggle between both modes using\n"
                           L"a setting in the settings dialog.\n"
                           L"\n"
                           L"In one button mode you mark squares as containing a\n"
                           L"bomb with a simple click, and unmark them with another\n"
                           L"single click. You can open a square by doing a double\n"
                           L"click. Make sure, that you do not click too fast so that\n"
                           L"single clicks are interpreted as double clicks.\n"
                           L"\n"
                           L"In two button mode, you just open a square by pressing\n"
                           L"the left button on it. You mark a square as containing\n"
                           L"a bomb by pressing the right mouse button on it and\n"
                           L"unmark it clicking with the right button on it again\n"
                           L"\n"
                           L"The game is won, if you have opened all squares that\n"
                           L"do not contain a bomb! There is no need to mark all\n"
                           L"squares as containg a bombto finish!\n"
                           L"\n"
                           L"Have fun :-)");
    }
    else if (sweeper!=NULL && model==sweeper->GetStatusModel()) {
      if (sweeper->GetStatus()==Sweeper::won) {
        sweeper->SetStatusToWaiting();
        size_t pos;

        pos=GetHiscorePos();
        if (pos<10) {
          ShowHiscoreEdit(pos);
        }
        else {
          if (Lum::Dlg::Msg::Ask(this,
                                 L"You have won!",
                                 L"You have won.\n"
                                 L"\n"
                                 L"...but sadly you did not make it into the hiscore.",
                                 L"Play one more!*|No, enough!^")==0) {
            sweeper->Run();
          }
        }
      }
      else if (sweeper->GetStatus()==Sweeper::lost) {
        sweeper->SetStatusToWaiting();
        if (Lum::Dlg::Msg::Ask(this,
                               L"Sorry, you have lost.",
                               L"Sorry, you have lost. Please try again.",
                               L"Play one more!*|No, enough!^")==0) {
          sweeper->Run();
        }
      }
      else if (sweeper->GetStatus()==Sweeper::playing) {
        Lum::OS::display->AddTimer(1,0,timerAction);
      }
    }
    else if (model==timerAction && timerAction->IsFinished()) {
      if (sweeper->GetStatus()==Sweeper::playing) {
        time_t       time=sweeper->GetElapsedTime();
        std::wstring value;

        if (time>9999) {
          time=9999;
        }

        value=Lum::Base::NumberToWString(time);

        while (value.length()<4) {
          value.insert(0,L"0");
        }

        this->time->SetText(value);
      }
      Lum::OS::display->AddTimer(1,0,timerAction);
    }
    else if (sweeper!=NULL && model==sweeper->GetMarksModel()) {
      this->marks->SetText(Lum::Base::NumberToWString(sweeper->GetMarksModel()->Get()));
    }
    else if (model==GetWindow()->GetMapedAction() &&
             GetWindow()->GetMapedAction()->IsFinished()) {
      Lum::OS::display->AddTimer(1,0,timerAction);
    }
    else if (model==GetWindow()->GetUnmapedAction() &&
             GetWindow()->GetMapedAction()->IsFinished()) {
      Lum::OS::display->RemoveTimer(timerAction);
    }

    Dialog::Resync(model,msg);
  }
};

int main(int argc, char* argv[])
{
  info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
  info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
  info.SetDescription(L"Mark all the hidden mines...");
  info.SetAuthor(L"Tim Teulings");
  info.SetContact(L"Tim Teulings <tim@teulings.org>");
  info.SetCopyright(L"(c) 2004, Tim Teulings");
  info.SetLicense(L"GNU Public License");

  if (Lum::OS::prober->Open(L"FindMine",argc,argv)) {
    MyWindow* window;

    window=new MyWindow;

    if (window->Open()) {
      window->SetExitAction(window->GetClosedAction());
      window->EventLoop();
      window->Close();
    }

    delete window;

    Lum::OS::display->Close();
  }

  SaveConfig();

  FreeConfig();
}
