/*
  This source is part of the FindMine program.
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "Configuration.h"

#include <cassert>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <fstream>

#include <dirent.h>

#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>

#include <Lum/Config/Config.h>

#include <Lum/Images/Loader.h>

#include <Lum/OS/Theme.h>

static std::list<Hiscore*> hiscores;
static std::wstring        userName(L"???");
static size_t              areaWidth=16,areaHeight=16,areaBombs=40;
static bool                oneButtonMode=false;
std::vector<Theme>         themes;
std::wstring               themeName;
std::wstring               zoomThemeName;

void SetUserName(const std::wstring& name)
{
  userName=name;
}

void SetCurrentGame(size_t width, size_t height, size_t bombs)
{
  areaWidth=width;
  areaHeight=height;
  areaBombs=bombs;
}

std::wstring GetUserName()
{
  return userName;
}

void GetCurrentGame(size_t& width, size_t& height, size_t& bombs)
{
  width=areaWidth;
  height=areaHeight;
  bombs=areaBombs;
}

void SetOneButtonMode(bool use)
{
  oneButtonMode=use;
}

bool GetOneButtonMode()
{
  return oneButtonMode;
}

bool LoadConfig()
{
  Lum::Config::Node      *top;
  Lum::Config::ErrorList errors;
  Lum::Base::Path        path(Lum::Base::Path::GetApplicationConfigPath());
  const char*            tmp;

  tmp=getenv("LOGNAME");

  if (tmp!=NULL && tmp[0]!='\0') {
    userName=Lum::Base::StringToWString(tmp);
  }

  oneButtonMode=Lum::OS::display->GetTheme()->RequestFingerFriendlyControls();

  size_t boxSize=0;

  // Find the biggest theme available, prefer external themes
  for (std::vector<Theme>::const_iterator theme=themes.begin();
       theme!=themes.end();
       ++theme)  {
    if (theme->boxSize>=boxSize) {
      if (theme->name.find(L"Internal")==std::wstring::npos) {
        boxSize=theme->boxSize;
        themeName=theme->name;
      }

      if (theme->name.find(L"Internal")!=std::wstring::npos) {
        boxSize=theme->boxSize;
        themeName=theme->name;
      }
    }
  }

  size_t zoomBoxSize=boxSize;
  zoomThemeName=themeName;

  // Find the biggest theme available, prefer external themes
  for (std::vector<Theme>::const_iterator theme=themes.begin();
       theme!=themes.end();
       ++theme)  {
    if (theme->boxSize<=zoomBoxSize) {
      if (theme->name.find(L"Internal")==std::wstring::npos) {
        zoomBoxSize=theme->boxSize;
        zoomThemeName=theme->name;
      }

      if (theme->name.find(L"Internal")!=std::wstring::npos) {
        zoomBoxSize=theme->boxSize;
        zoomThemeName=theme->name;
      }
    }
  }

  std::cout << "Theme: " << Lum::Base::WStringToString(themeName) << std::endl;

  // TODO: Find matching theme

  top=Lum::Config::LoadConfigFromXMLFile(path.GetPath(),errors);

  if (top==NULL) {
    return false;
  }

  if (top->GetName()!=L"FindMine") {
    std::cerr << "'" << Lum::Base::WStringToString(path.GetPath()) << "' is a valid config file!" << std::endl;
    delete top;
    return false;
  }

  bool              obm;
  size_t            w,h,b;
  std::wstring      user,tn;

  if (top->GetAttribute(L"width",w) &&
      top->GetAttribute(L"height",h) &&
      top->GetAttribute(L"bombs",b)) {
    areaWidth=w;
    areaHeight=h;
    areaBombs=b;
  }

  if (top->GetAttribute(L"user",user)) {
    SetUserName(user);
  }

  if (top->GetAttribute(L"oneButtonMode",obm)) {
    SetOneButtonMode(obm);
  }

  top->GetAttribute(L"theme",themeName);

  for (Lum::Config::Node::NodeList::const_iterator iter=top->GetChildren().begin();
       iter!=top->GetChildren().end();
       ++iter) {
    Lum::Config::Node *node=*iter;

    if (node->GetName()==L"Hiscore") {

      if (node->GetAttribute(L"width",w) &&
          node->GetAttribute(L"height",h) &&
          node->GetAttribute(L"bombs",b)) {
        Hiscore *score=new Hiscore();

        score->width=w;
        score->height=h;
        score->bombs=b;

        hiscores.push_back(score);

        for (Lum::Config::Node::NodeList::const_iterator iter=node->GetChildren().begin();
             iter!=node->GetChildren().end(); ++iter) {
          Lum::Config::Node *sub=*iter;

          if (sub->GetName()==L"Entry") {
            std::wstring name;
            size_t       time;

            if (sub->GetAttribute(L"name",name) && sub->GetAttribute(L"time",time)) {
              Hiscore::Entry entry;

              entry.name=name;
              entry.time=time;

              score->list.push_back(entry);
            }
          }
        }
      }
    }
  }

  delete top;

  return true;
}

bool SaveConfig()
{
  std::list<Hiscore*>::const_iterator iter;
  Lum::Config::Node                   *top;
  Lum::Base::Path                     path(Lum::Base::Path::GetApplicationConfigPath());
  std::wstring                        config;
  bool                                res;

  top=new Lum::Config::Node();
  top->SetName(L"FindMine");
  top->SetAttribute(L"width",areaWidth);
  top->SetAttribute(L"height",areaHeight);
  top->SetAttribute(L"bombs",areaBombs);
  top->SetAttribute(L"user",userName);
  top->SetAttribute(L"oneButtonMode",oneButtonMode);
  top->SetAttribute(L"theme",themeName);

  iter=hiscores.begin();
  while (iter!=hiscores.end()) {
    Lum::Config::Node *node,*sub;

    if ((*iter)->list.size()>10) {
      (*iter)->list.resize(10);
    }

    node=new Lum::Config::Node();
    node->SetName(L"Hiscore");
    node->SetAttribute(L"width",(*iter)->width);
    node->SetAttribute(L"height",(*iter)->height);
    node->SetAttribute(L"bombs",(*iter)->bombs);

    for (size_t x=0; x<(*iter)->list.size(); x++) {
      sub=new Lum::Config::Node();
      sub->SetName(L"Entry");
      sub->SetAttribute(L"name",(*iter)->list[x].name);
      sub->SetAttribute(L"time",(*iter)->list[x].time);

      node->Add(sub);
    }

    top->Add(node);

    ++iter;
  }

  Lum::Base::Status status;

  status=path.CreateDirRecursive();

  if (!status) {
    std::cerr << "Cannot create config directory '" << Lum::Base::WStringToString(path.GetDir()) << "': " << Lum::Base::WStringToString(status.GetDescription()) << std::endl;
    return false;
  }

  res=Lum::Config::SaveConfigToXMLFile(path.GetPath(),top);

  delete top;

  return res;
}

void FreeConfig()
{
  std::list<Hiscore*>::iterator iter;

  iter=hiscores.begin();
  while (iter!=hiscores.end()) {
    delete (*iter);

    ++iter;
  }

  hiscores.clear();
}

Hiscore* GetConfig(size_t width, size_t height, size_t bombs)
{
  std::list<Hiscore*>::const_iterator iter;
  Hiscore                             *hiscore;

  iter=hiscores.begin();
  while (iter!=hiscores.end()) {
    if ((*iter)->width==width && (*iter)->height==height && (*iter)->bombs==bombs) {
      return (*iter);
    }

    ++iter;
  }

  hiscore=new Hiscore;
  hiscore->width=width;
  hiscore->height=height;
  hiscore->bombs=bombs;

  hiscores.push_back(hiscore);

  return hiscore;
}

static const wchar_t* imageNames[] = {
  L"hidden.png",
  L"bomb.png",
  L"marked.png"
};

bool LoadThemes()
{
    Theme theme;

    theme.name=L"Internal small";
    theme.boxSize=16;
    themes.push_back(theme);

    theme.name=L"Internal Normal";
    theme.boxSize=24;
    themes.push_back(theme);

    theme.name=L"Internal Big";
    theme.boxSize=32;
    themes.push_back(theme);

    theme.name=L"Internal Huge";
    theme.boxSize=48;
    themes.push_back(theme);

  // Find sub directories in the global or local theme directory that contain
  // at least one of the required images

  if (Lum::OS::display->GetType()==Lum::OS::Display::typeGraphical) {
    DIR             *dir;
    struct dirent   *dirEnt;
    Lum::Base::Path path;
    Lum::Base::Path appPath;
    Lum::Base::Path locPath;

    appPath.SetNativeDir(Lum::Base::StringToWString(APP_DATADIR));
    appPath.AppendDir(L"themes");
    path=appPath;

    dir=opendir(Lum::Base::WStringToString(appPath.GetPath()).c_str());
    if (dir==NULL) {
      locPath.SetNativeDir(L"themes");
      path=locPath;

      dir=opendir(Lum::Base::WStringToString(locPath.GetPath()).c_str());
      if (dir==NULL) {
        std::cerr << "Cannot open theme directory as directory '" << Lum::Base::WStringToString(appPath.GetPath()) << "' or '" << Lum::Base::WStringToString(locPath.GetPath()) << "'" << std::endl;
        return false;
      }
    }

    while ((dirEnt=readdir(dir))!=NULL) {
      std::wstring    dirname=Lum::Base::StringToWString(dirEnt->d_name);
      Lum::Base::Path dirPath(path);

      dirPath.AppendDir(dirname);

      if (strcmp(dirEnt->d_name,".")!=0 &&
          strcmp(dirEnt->d_name,"..")!=0 &&
          dirEnt->d_name[0]!='.' &&
          dirPath.IsDir()) {
        Theme  theme;
        size_t imageCount=0;

        theme.name=Lum::Base::StringToWString(dirEnt->d_name);
        theme.boxSize=0;

        for (size_t i=0; i<countImage; i++) {
          Lum::Base::Path path;

          dirPath.SetBaseName(imageNames[i]);

          theme.images[i]=Lum::Images::Factory::factory->CreateImage();

          if (Lum::Images::loader->Load(dirPath.GetPath(),theme.images[i])) {
            imageCount++;
            theme.boxSize=std::max(theme.boxSize,theme.images[i]->GetWidth());
            theme.boxSize=std::max(theme.boxSize,theme.images[i]->GetHeight());
          }
          else {
            std::cerr << "Cannot load image '" << Lum::Base::WStringToString(dirPath.GetPath()) << "'" << std::endl;
            theme.images[i]=NULL;
          }
        }

        if (imageCount!=countImage) {
          std::cerr << "Not all theming images found => Skipping theme" << std::endl;
          continue;
        }

        themes.push_back(theme);
      }
    }
  }

  return true;
}

