#ifndef CONTROL_H
#define CONTROL_H

/*
  This source is part of the FindMine program.
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/Base/DateTime.h>
#include <Lum/Base/Size.h>

#include <Lum/Model/Integer.h>

#include <Lum/OS/Bitmap.h>
#include <Lum/OS/Display.h>
#include <Lum/OS/Fill.h>

#include <Lum/Object.h>

#include "Configuration.h"

class Sweeper : public Lum::Scrollable
{
public:
  static const int areaWidth=100;
  static const int areaHeight=100;

public:
  enum Status {
    waiting,
    playing,
    won,
    lost
  };

private:
  struct Cell
  {
    bool   active; //< The cell is part of the game area and thus visible
    bool   bomb;   //< Has a bomb
    bool   open;   //< The cell has been uncovered
    bool   marked; //< The cell is hidden but marked to have a bomb
    size_t count;  //< Number of bombs in neigbourhood
  };

private:
  size_t               sWidth;
  size_t               sHeight;
  size_t               sMaxWidth;
  size_t               sMaxHeight;
  Lum::OS::FontRef     font;
  Lum::OS::ColorRef    emptyColor;
  Lum::OS::ColorRef    cellColor;
  Lum::OS::ColorRef    lightColor;
  Lum::OS::ColorRef    darkColor;
  Lum::OS::ColorRef    textColor;
  Lum::OS::ColorRef    bombColor;
  Lum::OS::ColorRef    markColor;
  Lum::Model::ULongRef status;
  Lum::Model::LongRef  marks;
  size_t               bombs;
  Cell                 area[areaWidth][areaHeight];
  Lum::Base::Timer     timer;
  int                  mbdx; //< Position of mouse on button down
  int                  mbdy; //< Position of mouse on button down
  bool                 oneButtonMode;
  Lum::OS::Bitmap      *game;
  size_t               themeIndex;

private:
  // Base class
  void UpdateDimensions();

  // Drawing
  size_t GetBoxSize() const;
  Lum::Images::Image* GetImage(Image image) const;

  void DrawEmptyCellBackground(Lum::OS::DrawInfo* draw,
                               int x, int y,
                               size_t w, size_t h);
  void DrawHiddenCellBackground(Lum::OS::DrawInfo* draw,
                                int x, int y,
                                size_t w, size_t h);
  void DrawHiddenCellInternal(Lum::OS::DrawInfo* draw,
                              int x, int y,
                              size_t w, size_t h);
  void DrawCell(size_t row, size_t column);
  void RedrawArea();

  // Game logic
  bool GetNeighbour(size_t n,
                    size_t column, size_t row,
                    size_t& x, size_t& y);
  void RecalcArea();
  void OpenField(size_t x, size_t y);
  void ToggleMark(size_t x, size_t y);

public:
  Sweeper();
  ~Sweeper();

  bool RequiresKnobs() const;

  void SetThemeName(const std::wstring& themeName);

  void CalcSize();

  void Draw(int x, int y, size_t w, size_t h);

  bool HandleMouseEvent(const Lum::OS::MouseEvent& event);

  bool SetSize(size_t width, size_t height, size_t bombs);
  void SetOneButtonMode(bool set);

  size_t GetAreaWidth() const;
  size_t GetAreaHeight() const;
  size_t GetMines() const;
  size_t GetMinesPercent() const;

  Lum::Model::ULong* GetStatusModel() const;
  Lum::Model::Long* GetMarksModel() const;

  void Run();

  Status GetStatus() const;
  time_t GetElapsedTime() const;
  void   SetStatusToWaiting();

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg);
};

typedef Sweeper* SweeperPtr;

#endif
