#include <QApplication>
#include <QObject>
#include <QVBoxLayout>
#include <QPropertyAnimation>
#include <QPalette>
#include <QMessageBox>

#include "OverlayWidget.h"

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>

#include "Viewfinder.h"
#include "ThumbnailView.h"
#include "CameraThread.h"
#include "ScrollArea.h"

#include "AdjustmentWidget.h"
#include "SplashDialog.h"
#include "ExtendedSettings.h"

CameraThread *cameraThread;

// Handle Ctrl-C with a clean quit
void sigint(int) {
    cameraThread->stop();
}
 
int main(int argc, char **argv) {
    QApplication app(argc, argv);

    // We're going to be passing around Frames using Qt Signals, so we
    // need to first register the type with Qt.
    qRegisterMetaType<FCam::Frame>("FCam::Frame");

    // Make a thread that controls the camera and maintains its state
    cameraThread = new CameraThread();
   
    QMainWindow mainWindow;
    VScrollArea *scrollArea = new VScrollArea(&mainWindow);
    scrollArea->setGeometry(0, 0, 800, 480);
    Viewfinder *viewfinder = new Viewfinder(cameraThread);
    ThumbnailView *review = new ThumbnailView();
    
    scrollArea->addWidget(new ExtendedSettings());
    scrollArea->addWidget(viewfinder);
    scrollArea->addWidget(review);
    scrollArea->jumpTo(1);
    // Hook up camera thread image signal to the review widget
    QObject::connect(cameraThread, SIGNAL(newImage(ImageItem *)),
                     review, SLOT(newImage(ImageItem *)) );

    // When the shutter of focus button is pressed, the view should
    // slide back to the viewfinder. The cameraThread doesn't know
    // about the scrollArea, so we send the signal via a signal mapper
    // which provides it.
    QSignalMapper mapper;
    mapper.setMapping(cameraThread, viewfinder);
    QObject::connect(cameraThread, SIGNAL(focusPressed()),
                     &mapper, SLOT(map()));
    QObject::connect(cameraThread, SIGNAL(shutterPressed()),
                     &mapper, SLOT(map()));
    QObject::connect(&mapper, SIGNAL(mapped(QWidget *)),
                     scrollArea, SLOT(slideTo(QWidget *)));

    // Once the camera thread stops, quit the app
    QObject::connect(cameraThread, SIGNAL(finished()), 
                     &IOThread::instance(), SLOT(stop()));
    QObject::connect(&IOThread::instance(), SIGNAL(finished()),
                     &app, SLOT(quit()));

    new SplashDialog(&mainWindow);
    
    //review->launchShareDialog();
    // Launch the camera control thread
    cameraThread->start();

    // Make Ctrl-C call app->exit(SIGINT)
    signal(SIGINT, sigint);

    mainWindow.showFullScreen();
    int rval = app.exec();

    printf("About to delete camera thread\n");

    delete cameraThread;
    printf("Camera thread deleted\n");
    return rval;
}


