#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <sys/fcntl.h>
#include <poll.h>
#include <unistd.h>
#include <sstream>

#include "FCam/Event.h" 
#include "FCam/Time.h" 

#include "ButtonListener.h"
#include "../Debug.h"




namespace FCam { 
    namespace N900 {

        void *button_listener_thread_(void *arg) {
            ButtonListener *b = (ButtonListener *)arg;
            b->run();    
            pthread_exit(NULL);
            return NULL;
        }
        
        ButtonListener *ButtonListener::instance() {
            // Return a pointer to a static function variable. I feel
            // dirty doing this, but it guarantees that 
            // 1) The buttonlistener is created on first use
            // 2) The destructor gets called on program exit

            // See http://www.research.ibm.com/designpatterns/pubs/ph-jun96.txt
            // for an interesting discussion of ways to make sure singletons get deleted.

            static ButtonListener _instance;
            return &_instance;
        }
        
        ButtonListener::ButtonListener() : stop(false) {
            pthread_attr_t attr;
            struct sched_param param;
            
            // make the thread
            
            param.sched_priority = sched_get_priority_max(SCHED_OTHER);            
            
            pthread_attr_init(&attr);
            
            if ((errno =
                 -(pthread_attr_setschedparam(&attr, &param) ||
                   pthread_attr_setschedpolicy(&attr, SCHED_OTHER) ||
                   pthread_attr_setinheritsched(&attr, PTHREAD_EXPLICIT_SCHED) ||
                   pthread_create(&thread, &attr, button_listener_thread_, this)))) {
                error(Event::InternalError, "Error creating button listener thread");
            }

        }

        ButtonListener::~ButtonListener() {
            stop = true;
            pthread_join(thread, NULL);
        }

        void ButtonListener::run() {
            // TODO: The below is a horrible hack to prevent the built-in
            // camera program from stealing shutter press events. I really
            // wish there was a better way to do this.
            
            // kill the built-in camera program if it was launched at boot
            if (!fork()) {
                dprintf(2, "Killing camera-ui using dsmetool\n");
                // suppress output
                close(STDOUT_FILENO);
                close(STDERR_FILENO);
                execl("/usr/sbin/dsmetool",
                      "/usr/sbin/dsmetool",
                      "-k", "/usr/bin/camera-ui",
                      (char *)NULL);
                exit(0);
            }
            
            if (!fork()) {
                // Give the dsmetool a chance to do it's thing                
                usleep(100000);
                dprintf(2, "Killing camera-ui using killall\n");
                // suppress output
                close(STDOUT_FILENO);
                close(STDERR_FILENO);
                execl("/usr/bin/killall", "/usr/bin/killall", "camera-ui", (char *)NULL);
                exit(0);
            }
            
            const int BUTTONS = 4;

            const char *fnames[BUTTONS] = {"/sys/devices/platform/gpio-switch/cam_shutter/state",
                                           "/sys/devices/platform/gpio-switch/cam_focus/state",
                                           "/sys/devices/platform/gpio-switch/cam_launch/state",
                                           "/sys/devices/platform/gpio-switch/slide/state"};
                                               
            char buf[81];

            bool state[BUTTONS];

            int event[BUTTONS*2] = {Event::N900LensOpened,
                                    Event::N900LensClosed,
                                    Event::FocusPressed,
                                    Event::FocusReleased,
                                    Event::ShutterPressed,
                                    Event::ShutterReleased,
                                    Event::N900SlideOpened,
                                    Event::N900SlideClosed};

            std::string descriptions[BUTTONS*2] = {"Lens cover opened",
                                                   "Lens cover closed",
                                                   "Focus button pressed",
                                                   "Focus button released",
                                                   "Shutter button pressed",
                                                   "Shutter button released",
                                                   "Keyboard slide opened",
                                                   "Keyboard slide closed"};

            // open all the devices
            int rval;
            struct pollfd fds[BUTTONS];
            for (int i = 0; i < BUTTONS; i++) {
                fds[i].fd = open(fnames[i], O_RDONLY);
                fds[i].events = POLLPRI;
                fds[i].revents = 0;
            }

            // read the initial state
            for (int i = 0; i < BUTTONS; i++) {
                rval = read(fds[i].fd, &buf, 80);
                buf[rval] = 0;
                switch (buf[0]) {
                case 'c': // closed
                case 'i': // inactive
                    state[i] = false;
                    break;
                case 'o': // open
                case 'a': // active
                    state[i] = true;
                    break;
                default:                    
                    error(Event::InternalError, "ButtonListener: Unknown state: %s", buf);
                }
            }            

            while (!stop) {               
                // wait for a change
                rval = poll(fds, BUTTONS, 1000);
                if (rval == -1) {
                    // this fails once on load, not sure why
                    dprintf(2, "ButtonListener: poll failed");
                    //error(Event::InternalError, "ButtonListener: poll failed");
                    continue;
                }
                if (rval == 0) continue; // timeout

                for (int i = 0; i < BUTTONS; i++) {
                    if (fds[i].revents & POLLPRI) {
                        close(fds[i].fd);
                        fds[i].fd = open(fnames[i], O_RDONLY, 0);
                        rval = read(fds[i].fd, &buf, 80);
                        buf[rval] = 0;
                        switch (buf[0]) {
                        case 'c': // closed
                        case 'i': // inactive
                            if (state[i] != false) {
                                state[i] = false;
                                postEvent(event[i*2+1], 0, descriptions[i*2+1]);
                            }
                            break;
                        case 'o': // open
                        case 'a': // active
                            if (state[i] != true) {
                                state[i] = true;
                                postEvent(event[i*2], 0, descriptions[i*2]);
                            }
                            break;
                        default:
                            error(Event::InternalError, "ButtonListener: Unknown state: %s", buf);
                        }
                    }
                }                
            }

            dprintf(2, "Button listener shutting down\n");

            for (int i = 0; i < BUTTONS; i++) {
                close(fds[i].fd);
            }
            
            // restart the built-in camera program
            if (!fork()) {
                // on success, this function never returns
                execl("/usr/sbin/dsmetool", 
                      "/usr/sbin/dsmetool", 
                      "-U", "user", 
                      "-o", "/usr/bin/camera-ui", 
                      (char *)NULL);
                // it returned, so there was a failure
                error(Event::InternalError, "Error restarting camera-ui");
                exit(0);
            }
        }
    }
}


