/****************************************************************************
**
** Copyright (C) 2010  Aimone Davide realsportcars@hotmail.com
**
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see <http://www.gnu.org/licenses/>.
**
****************************************************************************/

#include "fastsmswidget.h"
#include "ui_fastsmswidget.h"
#include "busManagerServ.h"
#include "kb_qwerty.h"
#include "dialogabout.h"
#include "dialogsettings.h"
#include <QCloseEvent>
#include <QMenuBar>

FastSMSWidget::FastSMSWidget(QWidget *parent) :
        QWidget(parent),
        ui(new Ui::FastSMSWidget)
{
    //Setup for QSettings
    QCoreApplication::setOrganizationName(Core::SETTING_ORG);
    QCoreApplication::setApplicationName(Core::SETTING_APPNAME);

    //    //Test purpose: reset all settings
    //    QSettings settings;
    //    settings.remove("");

    ui->setupUi(this);
    m_layoutManager = qobject_cast<LayoutManager *> (ui->centralwidget);
    m_textProcessor = m_layoutManager->getTextProcessor();
    m_bDictionaryUpdated = false;
    m_dictionary = NULL;


    qDebug() << "Load settings...";
    //Setup and Load Settings
    loadSettings(true);
    //
    qDebug() << "Settings loaded";

    //Menu
    QMenuBar *menu = new QMenuBar(this);
    QMenu *mainMenu = new QMenu("Menu",menu);

    QAction *aOption = new QAction("Options",mainMenu);
    connect(aOption,SIGNAL(triggered()),this, SLOT(openOptionDialog()));
    mainMenu->addAction(aOption);

    QAction *aNewWord = new QAction("New Word",mainMenu);
    connect(aNewWord,SIGNAL(triggered()),this, SLOT(insertNewWord()));
    mainMenu->addAction(aNewWord);

    QAction *aDonate = new QAction("Donate",mainMenu);
    connect(aDonate,SIGNAL(triggered()),this, SLOT(donate()));
    mainMenu->addAction(aDonate);

    QAction *aAbout = new QAction("About",mainMenu);
    mainMenu->addAction(aAbout);
    connect(aAbout,SIGNAL(triggered()),this, SLOT(openAboutDialog()));

    QAction *aQuit = new QAction("Quit",mainMenu);
    connect(aQuit,SIGNAL(triggered()),this, SLOT(close()));
    mainMenu->addAction(aQuit);

    menu->addMenu(mainMenu);
    //

}

FastSMSWidget::~FastSMSWidget()
{
    delete ui;
}


/** If is vkb prevent from real close, so dictionary haven't to be reloaded each time,
    then check if dictionary need save **/
void FastSMSWidget::closeEvent(QCloseEvent *event)
{
    qDebug() << "closeEvent:" << event->type();

    if(!m_bStandalone)
    {
        event->ignore();
        qDebug() << "About to close, message is " << m_textProcessor->getText();

        qDebug() << "Curr orientation:"<<m_sCurrOrientation<<"-"<<Core::ORIENTATION_PORTRAIT;

#if defined(Q_WS_MAEMO_5)
        //If not landscape, need to return to landscape view
        if(m_sCurrOrientation.compare(Core::ORIENTATION_LANDSCAPE) != 0)
        {
            setAttribute(Qt::WA_Maemo5LandscapeOrientation, true);

            hide();
            showMaximized();
        }
#endif

        m_textProcessor->commit();
        hide();
    }

    //TODO: spostare in un thread
    if(m_bDictionaryUpdated)
    {
        m_dictionary->saveDictionaryToFile();
    }
}

void FastSMSWidget::setStandalone(bool bStandalone)
{
    m_bStandalone = bStandalone;

    if(m_bStandalone)
    {
        setWindowFlags(Qt::Window | Qt::WindowSystemMenuHint);
    }
    else
    {
        setWindowFlags(Qt::Dialog | Qt::FramelessWindowHint);

        QDBusConnection systemBus = QDBusConnection::systemBus();
        systemBus.connect("com.nokia.mce",
                          "/com/nokia/mce/signal",
                          "com.nokia.mce.signal",
                          "sig_device_orientation_ind",
                          this, SLOT(orientationChanged(QString)));
    }

    m_textProcessor->setStandalone(bStandalone);
}

/** Workaround for enabling autorotation when used as keyboard **/
void FastSMSWidget::orientationChanged(QString sOrientation)
{
    if(m_bStandalone)
        return;

    if(isVisible() && m_sCurrOrientation.compare(sOrientation) != 0)
    {
        qDebug() << "Orientation changed:" << sOrientation;

        if(sOrientation.compare(Core::ORIENTATION_PORTRAIT) == 0 ||
           sOrientation.compare(Core::ORIENTATION_LANDSCAPE) == 0)
        {
            m_sCurrOrientation = sOrientation;
            hide();
            showMaximized();
        }
        else
            qWarning() << "Unknown orientation:" << sOrientation;
    }
}

/** Menu Management */
void FastSMSWidget::openAboutDialog()
{
    DialogAbout *about = new DialogAbout(this);
    about->setAttribute(Qt::WA_DeleteOnClose);
    about->show();
}

void FastSMSWidget::donate()
{
    QDesktopServices::openUrl(QUrl("https://www.paypal.com/cgi-bin/webscr?cmd=_s-xclick&hosted_button_id=V353RBQ6TATN6"));
}

void FastSMSWidget::openOptionDialog()
{
    DialogSettings *dialSettings = new DialogSettings(this);
    dialSettings->setAttribute(Qt::WA_DeleteOnClose);
    connect(dialSettings,SIGNAL(settingsChanged(bool)),this,SLOT(loadSettings(bool)));
    dialSettings->show();
}

void FastSMSWidget::loadSettings(bool bLoadDictionary)
{
    QSettings settings;

    //Language
    if(bLoadDictionary)
    {
        QString sLang = settings.value(Core::SETTING_LANGUAGE,"").toString();

        //First start
        if(sLang.isEmpty())
        {
            openOptionDialog();
            return;
        }

        if(m_bDictionaryUpdated)
            m_dictionary->saveDictionaryToFile();

        if(m_dictionary != NULL)
            delete m_dictionary;

        m_dictionary = new dictionary(new QString(sLang),this);
        m_textProcessor->setDictionary(m_dictionary);
        settings.setValue(Core::SETTING_LANGUAGE,sLang);
    }

    //Orientation
#if defined(Q_WS_MAEMO_5)
    QString sCurrOrient = settings.value(Core::SETTING_ORIENTATION).toString();
    if(sCurrOrient.compare(Core::ORIENTATION_LANDSCAPE) == 0)
        setAttribute(Qt::WA_Maemo5LandscapeOrientation, true);
    else if(sCurrOrient.compare(Core::ORIENTATION_PORTRAIT) == 0)
        setAttribute(Qt::WA_Maemo5PortraitOrientation, true);
    else
        setAttribute(Qt::WA_Maemo5AutoOrientation, true);
#endif

    //Font
    int nFontSize = settings.value(Core::SETTING_FONTSIZE).toInt();
    m_textProcessor->setSmsFontSize(nFontSize * 9 + 18); //we will have 18,27 or 36
}

bool FastSMSWidget::addRecipient(const QString &phone, const QString &name)
{
    return m_textProcessor->m_recipient->addRecipient(phone,name);
}

void FastSMSWidget::insertNewWord()
{
    m_dialNewWord = new QDialog(this);
    m_dialNewWord->setAttribute(Qt::WA_DeleteOnClose);
    m_inputNewWord = new QLineEdit(m_dialNewWord);
    QPushButton *btOk = new QPushButton("Ok",m_dialNewWord);
    connect(btOk,SIGNAL(clicked()),this,SLOT(newWordInserted()));
    QGridLayout *insWordLayout = new QGridLayout();
    insWordLayout->addWidget(m_inputNewWord,0,0);
    kb_Qwerty *kb = new kb_Qwerty(m_dialNewWord,true);
    connect(kb,SIGNAL(input(QString,Core::Keyboards)),this,SLOT(composite(QString,Core::Keyboards)));
    insWordLayout->addWidget(kb,1,0);
    insWordLayout->addWidget(btOk,2,0);
    insWordLayout->setContentsMargins(0,0,0,0);
    m_inputNewWord->setFocus();

    m_dialNewWord->setWindowTitle("Type new word");
    m_dialNewWord->setLayout(insWordLayout);
    m_dialNewWord->showMaximized();
}

void FastSMSWidget::composite(QString sChar, Core::Keyboards)
{
    if(sChar.compare(Keyboard::DELETE) == 0)
        m_inputNewWord->backspace();
    else
        m_inputNewWord->insert(sChar);
}

void FastSMSWidget::newWordInserted()
{
    QString *newWord = new QString(m_inputNewWord->text().toLower());
    if(!newWord->isEmpty())
    {
        qDebug() << "New word is:" << newWord->toAscii();
        m_bDictionaryUpdated = m_dictionary->addWord(newWord);
        m_textProcessor->addNewWord(newWord);
    }

    m_dialNewWord->close();
}

void FastSMSWidget::enableQwerty()
{
    m_layoutManager->enableQwerty();
}

/** Method called by external app. Behaviour must be like virtual keyboard**/
void FastSMSWidget::surroundRequest(const QString &text)
{
#if defined(Q_WS_MAEMO_5)
    setAttribute(Qt::WA_Maemo5AutoOrientation, true);
#endif

    if(m_bStandalone)
        setStandalone(false);

    m_sCurrOrientation = getDeviceOrientation();
    m_textProcessor->setText(text);
    showMaximized();
    raise();
}

void FastSMSWidget::standalone()
{
    if(!m_bStandalone)
        setStandalone(true);

    showMaximized();
    raise();
}

/** For MAEMO5 ask to DBus current orientation, empty string otherwise **/
QString FastSMSWidget::getDeviceOrientation()
{
    QString sOrientation;

#if defined(Q_WS_MAEMO_5)
    QDBusInterface iface("com.nokia.mce","/com/nokia/mce/request","com.nokia.mce.request",QDBusConnection::systemBus());
    QDBusPendingReply<QString, QString, QString, int, int, int> reply;
    reply = iface.call("get_device_orientation");
    reply.waitForFinished();
    sOrientation = reply.argumentAt(0).toString();
    if(reply.isError())
        qWarning() << "Can't retrive start orientation:" << reply.error();
    else
        qDebug() << "Orientation from DBus:" << m_sCurrOrientation;

    if(sOrientation.compare("unknown") == 0)
        sOrientation = Core::ORIENTATION_LANDSCAPE;

#else
    sOrientation = "";
#endif

    return sOrientation;
}












