/****************************************************************************
**
** Copyright (C) 2010  Aimone Davide realsportcars@hotmail.com
**
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see <http://www.gnu.org/licenses/>.
**
****************************************************************************/

#include "fastsms.h"
#include "ui_fastsms.h"
#include "button.h"
#include "dictionary.h"
#include "busManagerServ.h"
#include <QMessage>
#include <QMessageService>
#include <QMessageAddress>
#include <QMenuBar>
#include <QMainWindow>
//#include <QContactManager>

#define STATE_SPACE 0
#define STATE_SYM 1
#define STATE_WORD 2
#define STATE_UNKNOWN 3
#define STATE_DEL 3
#define STATE_MOVECUR_BUTTON 3
#define STATE_CURMOVED 5
#define STATE_AFTERCURMOVED 6
#define STATE_SEND 8

#define SETTING_LANGUAGE "language"
#define SETTING_ORIENTATION "orientation"
#define SETTING_PHONEBOOK "phonebook"

#define SETTING_FONTSIZE "fontsize"
#define FONTSIZE_DEFAULT 0
#define FONTSIZE_BIG 1
#define FONTSIZE_BIGGER 2

#define ORIENTATION_AUTO "auto"
#define ORIENTATION_LANDSCAPE "landscape"
#define ORIENTATION_PORTRAIT "portrait"

#define PHONEBOOK_HORIZONTALCALL 0
#define PHONEBOOK_MYCONTACTS 1

#define BUTTON_CURNEXT "btMoveCurNext"
#define BUTTON_CURPREV "btMoveCurPrev"

#define ACTION_INS 0
#define ACTION_DEL 1
#define ACTION_SEND 1

#define DATA_PATH "/data/"

#define CAPITALIZATION_CHAR ".!?" //After those character, uppercase is needed
#define IN_STYLE_CAPITALIZE 0
#define IN_STYLE_UPPER 1
#define IN_STYLE_LOWER 2
#define IN_STYLE_DIGIT 3
#define IN_STYLE_FW_ON 4
#define IN_STYLE_FW_OFF 5

#define SINGLE_CHAR_TIMEOUT 750

//using namespace QtMobility;
QTM_USE_NAMESPACE

        Fastsms::Fastsms(QWidget *parent) :
        QWidget(parent),
        ui(new Ui::Fastsms)
{
    ui->setupUi(this);
    setWindowTitle("FastSMS");

    connect(ui->btOne, SIGNAL(clicked()), this, SLOT(digitClicked()));
    connect(ui->btTwo, SIGNAL(clicked()), this, SLOT(digitClicked()));
    connect(ui->btThree, SIGNAL(clicked()), this, SLOT(digitClicked()));
    connect(ui->btFour, SIGNAL(clicked()), this, SLOT(digitClicked()));
    connect(ui->btFive, SIGNAL(clicked()), this, SLOT(digitClicked()));
    connect(ui->btSix, SIGNAL(clicked()), this, SLOT(digitClicked()));
    connect(ui->btSeven, SIGNAL(clicked()), this, SLOT(digitClicked()));
    connect(ui->btEight, SIGNAL(clicked()), this, SLOT(digitClicked()));
    connect(ui->btNine, SIGNAL(clicked()), this, SLOT(digitClicked()));
    connect(ui->btZero, SIGNAL(clicked()), this, SLOT(digitClicked()));
    connect(ui->btTo, SIGNAL(clicked()), this, SLOT(openPhoneBook()));
    connect(ui->btSend, SIGNAL(clicked()), this, SLOT(sendSMS()));
    connect(ui->btMoveCurNext, SIGNAL(clicked()), this, SLOT(moveCursorBt()));
    connect(ui->btMoveCurPrev, SIGNAL(clicked()), this, SLOT(moveCursorBt()));
    connect(ui->btFunct, SIGNAL(clicked()), this, SLOT(deleteChar()));
    connect(ui->btDelChar, SIGNAL(clicked()), this, SLOT(deleteChar()));
    connect(ui->btNext, SIGNAL(clicked()), this, SLOT(nextWord()));
    connect(ui->btPrev, SIGNAL(clicked()), this, SLOT(prevWord()));

    m_phoneNumber = ui->sPhoneNumber;
    m_smsEdit = ui->sSmsEdit;

    //Add menu for functions button
    QMenu *funcMenu = new QMenu("functions",this);

    QAction *aFastWrite = new QAction("Disable FastWrite",funcMenu);
    aFastWrite->setData(IN_STYLE_FW_OFF);
    connect(aFastWrite,SIGNAL(triggered()),this, SLOT(chageInputStyle()));
    funcMenu->addAction(aFastWrite);

    funcMenu->addSeparator();

    QAction *aISCapitalized = new QAction("Abc",funcMenu);
    aISCapitalized->setData(IN_STYLE_CAPITALIZE);
    connect(aISCapitalized,SIGNAL(triggered()),this, SLOT(chageInputStyle()));
    funcMenu->addAction(aISCapitalized);

    QAction *aISUpperCase = new QAction("ABC",funcMenu);
    aISUpperCase->setData(IN_STYLE_UPPER);
    connect(aISUpperCase,SIGNAL(triggered()),this, SLOT(chageInputStyle()));
    funcMenu->addAction(aISUpperCase);

    QAction *aISLowerCase = new QAction("abc",funcMenu);
    aISLowerCase->setData(IN_STYLE_LOWER);
    connect(aISLowerCase,SIGNAL(triggered()),this, SLOT(chageInputStyle()));
    funcMenu->addAction(aISLowerCase);

    QAction *aISDigit = new QAction("123",funcMenu);
    aISDigit->setData(IN_STYLE_DIGIT);
    connect(aISDigit,SIGNAL(triggered()),this, SLOT(chageInputStyle()));
    funcMenu->addAction(aISDigit);

    ui->btFunct->setMenu(funcMenu);
    //

    //load configuration
    m_dict = NULL;
    m_settings = new QSettings("it.rsc.fastsms","fastsms",this);
    loadDict();

    //init variables
    m_nWordIndex=0;
    m_bNeedSave = false;
    m_sCompleteValue = QString();
    m_cursorSmsText = m_smsEdit->textCursor(); // retrieve the visible cursor
    m_nMsgLength = 0;
    m_nBeginWord = 0;
    m_nEndWord = 0;
    m_nCharCount = 160;
    m_nSMSCount = 1;
    //init variables for capitalization
    m_bMustCapitalize = true;
    m_sCapitalization = CAPITALIZATION_CHAR;
    m_nInputSyle = IN_STYLE_CAPITALIZE;
    m_bFastInput = true;
    m_bNewChar = true;
    //init variables for standard input style
    m_bTimerNewChar = true;
    timerNewChar = new QTimer(this);
    connect(timerNewChar,SIGNAL(timeout()),this,SLOT(alarmNewChar()));
    timerNewChar->setSingleShot(true);
    timerNewChar->setInterval(SINGLE_CHAR_TIMEOUT);

    //Menu
    QMenuBar *menu = new QMenuBar(this);
    QMenu *mainMenu = new QMenu("File",menu);

    QAction *aOption = new QAction("Options",mainMenu);
    connect(aOption,SIGNAL(triggered()),this, SLOT(openOptionDialog()));
    mainMenu->addAction(aOption);

    QAction *aNewWord = new QAction("New Word",mainMenu);
    connect(aNewWord,SIGNAL(triggered()),this, SLOT(insertNewWord()));
    mainMenu->addAction(aNewWord);

    QAction *aDonate = new QAction("Donate",mainMenu);
    connect(aDonate,SIGNAL(triggered()),this, SLOT(donate()));
    mainMenu->addAction(aDonate);

    QAction *aAbout = new QAction("About",mainMenu);
    mainMenu->addAction(aAbout);
    connect(aAbout,SIGNAL(triggered()),this, SLOT(openAboutDialog()));

    QAction *aQuit = new QAction("Quit",mainMenu);
    connect(aQuit,SIGNAL(triggered()),this, SLOT(close()));
    mainMenu->addAction(aQuit);

    menu->addMenu(mainMenu);

    //manage autorotate from start
    this->setFocus();

    //manage cursor
    connect(m_smsEdit,SIGNAL(textChanged()),this,SLOT(smsChanged()));
    connect(m_smsEdit,SIGNAL(selectionChanged()),this,SLOT(cursorPosChanged()));
}

/** check if configuration and dict file exists. If not prompt to option dialog, else load it**/
void Fastsms::loadDict()
{
    bool bOpenOpt = true;

    QString sLang = m_settings->value(SETTING_LANGUAGE,"").toString();

    if(!sLang.isEmpty())
    {
        QString sPath = QApplication::applicationDirPath() + DATA_PATH + sLang;
        QFile *dictFile = new QFile(sPath);

        if(dictFile->exists())
            bOpenOpt = false;

        delete dictFile;

#if defined(Q_WS_MAEMO_5)
        QString sCurrOrient = m_settings->value(SETTING_ORIENTATION).toString();
        if(sCurrOrient.compare(ORIENTATION_LANDSCAPE) == 0)
            setAttribute(Qt::WA_Maemo5LandscapeOrientation, true);
        else if(sCurrOrient.compare(ORIENTATION_PORTRAIT) == 0)
            setAttribute(Qt::WA_Maemo5PortraitOrientation, true);
        else
            setAttribute(Qt::WA_Maemo5AutoOrientation, true);
#endif
    }

    if(bOpenOpt)
        openOptionDialog();
    else
        m_dict = new dictionary(new QString(m_settings->value(SETTING_LANGUAGE).toString()),this);
}

/** Open and populate option dialog **/
void Fastsms::openOptionDialog()
{
    m_dialOptions = new QDialog(this);
    m_dialOptions->setAttribute(Qt::WA_DeleteOnClose);
    m_uiOptions.setupUi(m_dialOptions);

    connect(m_uiOptions.btSave,SIGNAL(clicked()),this,SLOT(closeOptions()));

    QDir dictDir(QApplication::applicationDirPath() + "/data");
    QStringList dictList = dictDir.entryList(QStringList("*.idc"));

    QString sCurrLang = m_settings->value(SETTING_LANGUAGE).toString();
    for(int i=0; i<dictList.size(); i++)
    {
        if(!dictList.at(i).contains("user"))
        {
            m_uiOptions.cbLanguage->addItem(dictList.at(i));

            if(dictList.at(i).compare(sCurrLang) == 0)
                m_uiOptions.cbLanguage->setCurrentIndex(i);
        }
    }

    m_uiOptions.cOrientationAuto->setChecked(true);

    QString sCurrOrient = m_settings->value(SETTING_ORIENTATION).toString();
    if(sCurrOrient.compare(ORIENTATION_LANDSCAPE) == 0)
        m_uiOptions.cOrientationLandscape->setChecked(true);
    else if(sCurrOrient.compare(ORIENTATION_PORTRAIT) == 0)
        m_uiOptions.cOrientationPortrait->setChecked(true);

    int nPhoneBook = m_settings->value(SETTING_PHONEBOOK).toInt();
    m_uiOptions.cbPhonebook->setCurrentIndex(nPhoneBook);

    m_uiOptions.cbFontSize->setCurrentIndex(m_settings->value(SETTING_FONTSIZE).toInt());

    m_dialOptions->show();

}

/** Open and populate option dialog **/
void Fastsms::openAboutDialog()
{
    QDialog * dlAbout = new QDialog(this);
    dlAbout->setAttribute(Qt::WA_DeleteOnClose);

    dlAbout->setWindowTitle("About");
    QGridLayout *layout = new QGridLayout();
    QString sAbout = QString("Created by: Aimone Davide \n\n For requests,suggestions,bugs write to: \n fastsms@hotmail.it \n");
    QLabel *msg = new QLabel(sAbout);
    layout->addWidget(msg,0,0);
    dlAbout->setLayout(layout);

    dlAbout->show();
}

/** SLOT: close option dialog and save options in settings file **/
void Fastsms::closeOptions()
{
    QString sOldLang = m_settings->value(SETTING_LANGUAGE).toString();

    QString sDictFile = m_uiOptions.cbLanguage->currentText();
    m_settings->setValue(SETTING_LANGUAGE,sDictFile);
    m_settings->setValue(SETTING_PHONEBOOK,m_uiOptions.cbPhonebook->currentIndex());

    if(m_uiOptions.cOrientationAuto->isChecked())
    {
        m_settings->setValue(SETTING_ORIENTATION,ORIENTATION_AUTO);
#if defined(Q_WS_MAEMO_5)
        setAttribute(Qt::WA_Maemo5AutoOrientation, true);
#endif
    }
    else if(m_uiOptions.cOrientationLandscape->isChecked())
    {
        m_settings->setValue(SETTING_ORIENTATION,ORIENTATION_LANDSCAPE);
#if defined(Q_WS_MAEMO_5)
        setAttribute(Qt::WA_Maemo5LandscapeOrientation, true);
#endif
    }
    else if(m_uiOptions.cOrientationPortrait->isChecked())
    {
        m_settings->setValue(SETTING_ORIENTATION,ORIENTATION_PORTRAIT);
#if defined(Q_WS_MAEMO_5)
        setAttribute(Qt::WA_Maemo5PortraitOrientation, true);
#endif
    }

    if(sOldLang.compare(sDictFile) != 0)
    {
        qDebug() << "Language changed!";
        if(m_bNeedSave)
            m_dict->saveDictionaryToFile();

        delete m_dict;
        m_dict = new dictionary(new QString(m_settings->value(SETTING_LANGUAGE).toString()),this);
    }

    //Font size
    m_settings->setValue(SETTING_FONTSIZE,m_uiOptions.cbFontSize->currentIndex());
    int nFontSize = m_settings->value(SETTING_FONTSIZE).toInt();
    QFont font = m_smsEdit->font();
    //we will have 18,27 or 36
    font.setPointSize(nFontSize * 9 + 18);
    qDebug() << font.pointSize();
    m_smsEdit->setFont(font);

    qDebug() << "Settings:" << m_settings->value(SETTING_LANGUAGE).toString() << "," << m_settings->value(SETTING_ORIENTATION).toString();

    m_dialOptions->close();
}

void Fastsms::moveCursorBt()
{
    bool bMoveNext = false;
    m_nState = STATE_MOVECUR_BUTTON;

    //Button *clickedButton = qobject_cast<Button *>(sender());
    QPushButton *clickedButton = qobject_cast<QPushButton *>(sender());
    QString btPressed = clickedButton->objectName();
    if(btPressed.compare(BUTTON_CURNEXT) == 0)
        bMoveNext = true;

    if(m_cursorSmsText.hasSelection() ||
       (bMoveNext && m_cursorSmsText.atEnd()) ||
       (!bMoveNext && m_cursorSmsText.atStart()))
    {
        m_smsEdit->disconnect();

        if(bMoveNext)
            m_nBeginWord = m_nEndWord;
        else
            m_nEndWord = m_nBeginWord;

        m_lWord.clear();
        m_sCompleteValue.clear();
        m_nWordIndex=0;

        m_cursorSmsText.setPosition(m_nEndWord, QTextCursor::MoveAnchor);
        m_smsEdit->setTextCursor(m_cursorSmsText);
        connect(m_smsEdit,SIGNAL(textChanged()),this,SLOT(smsChanged()));
        connect(m_smsEdit,SIGNAL(selectionChanged()),this,SLOT(cursorPosChanged()));
    }
    else
    {
        m_smsEdit->disconnect();

        m_lWord.clear();
        m_sCompleteValue.clear();
        m_nWordIndex=0;

        if(bMoveNext)
        {
            m_cursorSmsText.movePosition(QTextCursor::NextCharacter);
            m_nBeginWord = m_cursorSmsText.position();
            m_nEndWord = m_nBeginWord;
        }
        else
        {
            m_cursorSmsText.movePosition(QTextCursor::PreviousCharacter);
            m_nBeginWord = m_cursorSmsText.position();
            m_nEndWord = m_nBeginWord;
        }

        m_cursorSmsText.setPosition(m_cursorSmsText.position()-1,QTextCursor::MoveAnchor);
        m_cursorSmsText.setPosition(m_cursorSmsText.position()+2,QTextCursor::KeepAnchor);
        QString sSel = m_cursorSmsText.selectedText();

        m_cursorSmsText.setPosition(m_cursorSmsText.position()-1,QTextCursor::MoveAnchor);
        m_smsEdit->setTextCursor(m_cursorSmsText);

        if((sSel.isEmpty()) || (!sSel.contains(" ") && !m_dict->isSymbol(sSel.at(0)) && !m_dict->isSymbol(sSel.at(1))))
        {
            cursorPosChanged();
        }

        connect(m_smsEdit,SIGNAL(textChanged()),this,SLOT(smsChanged()));
        connect(m_smsEdit,SIGNAL(selectionChanged()),this,SLOT(cursorPosChanged()));
    }
}

void Fastsms::cursorPosChanged()
{
    if(m_nState != STATE_AFTERCURMOVED)
    {
        m_smsEdit->disconnect();

        qDebug() << "cursor moved";
        m_cursorSmsText = m_smsEdit->textCursor(); // retrieve the visible cursor
        m_cursorSmsText.select(QTextCursor::WordUnderCursor);
        m_nBeginWord = m_cursorSmsText.anchor();
        m_nEndWord = m_cursorSmsText.position();
        m_smsEdit->setTextCursor(m_cursorSmsText);

        if(m_cursorSmsText.hasSelection())
        {
            qDebug() << "Selected text from Cursor moved" << m_cursorSmsText.selectedText().toAscii();

            QString *sWord = new QString(m_cursorSmsText.selectedText());
            m_sCompleteValue = *m_dict->calculateKey(sWord);
            m_lWord = m_dict->getWordList(&m_sCompleteValue);
            //m_nWordIndex = 0;
            m_nWordIndex = m_lWord.indexOf(*sWord);
            if(m_nWordIndex < 0)
                m_nWordIndex = 0;
            m_nState = STATE_AFTERCURMOVED;

            delete sWord;
        }
        else
        {
            qDebug() << "Nothing selected";
            m_sCompleteValue.clear();
            m_nWordIndex = 0;
            m_lWord.clear();
        }

        connect(m_smsEdit,SIGNAL(textChanged()),this,SLOT(smsChanged()));
        connect(m_smsEdit,SIGNAL(selectionChanged()),this,SLOT(cursorPosChanged()));
    }
    else
    {
        m_nState = STATE_CURMOVED;
    }
}

void Fastsms::smsChanged()
{
    m_smsEdit->disconnect();

    m_cursorSmsText.setPosition(m_nBeginWord, QTextCursor::MoveAnchor);
    m_cursorSmsText.setPosition(m_nEndWord, QTextCursor::KeepAnchor);

    m_smsEdit->setTextCursor(m_cursorSmsText);

    connect(m_smsEdit,SIGNAL(textChanged()),this,SLOT(smsChanged()));
    connect(m_smsEdit,SIGNAL(selectionChanged()),this,SLOT(cursorPosChanged()));
}

/**Create button */
Button *Fastsms::createButton(const QString &text, const char *member)
{
    Button *button = new Button(text,this);
    connect(button, SIGNAL(clicked()), this, member);
    button->setFocusPolicy(Qt::NoFocus);
    return button;
}


void Fastsms::digitClicked()
{

    //Button *clickedButton = qobject_cast<Button *>(sender());
    QPushButton *clickedButton = qobject_cast<QPushButton *>(sender());
    QString lastPress = clickedButton->text().at(0);

    if(m_phoneNumber->hasFocus())
    {
        m_phoneNumber->insert(lastPress.toAscii());
    }
    else
    {
        m_smsEdit->setFocus();

        //FastWrite input style selected
        if(m_bFastInput && m_nInputSyle != IN_STYLE_DIGIT)
        {
            if(lastPress.compare("0") == 0)
            {
                m_nState = STATE_SPACE;
                m_nMsgLength++;
                m_sCompleteValue = lastPress;
                m_nWordIndex=0;
                m_lWord.clear();
                m_lWord = m_dict->getWordList(&m_sCompleteValue);

                m_nEndWord ++;
                m_nBeginWord = m_nEndWord-1;
                QString sChoosenWord = m_lWord.at(0);
                addText(sChoosenWord);
                updateCharCounter(ACTION_INS);
                m_sCompleteValue.clear();

            }
            else
            {
                if(m_nState == STATE_SPACE)
                {
                    m_cursorSmsText.clearSelection();
                    m_nBeginWord = m_nEndWord;
                }

                m_sCompleteValue.append(lastPress);

                if(lastPress.compare("1") == 0)
                    m_nState = STATE_SYM;

                if(m_dict->existsKey(&m_sCompleteValue))
                {
                    m_nState = STATE_WORD;
                    m_nMsgLength++;
                    m_nWordIndex=0;
                    m_lWord = m_dict->getWordList(&m_sCompleteValue);

                    m_nEndWord++;
                    QString sChoosenWord = m_lWord.at(0);
                    updateText(sChoosenWord);
                    updateCharCounter(ACTION_INS);
                }
                else
                {
                    if(lastPress.compare("1") == 0)
                    {
                        m_nState = STATE_SYM;
                        m_nMsgLength++;
                        m_sCompleteValue.clear();
                        m_sCompleteValue = lastPress;

                        m_nWordIndex=0;
                        m_lWord = m_dict->getWordList(&m_sCompleteValue);

                        m_nBeginWord = m_nEndWord;
                        m_nEndWord ++;
                        QString sChoosenWord = m_lWord.at(0);
                        addText(sChoosenWord);
                        updateCharCounter(ACTION_INS);
                    }
                    else
                    {
                        if(m_sCompleteValue.at(m_sCompleteValue.length()-2) == QChar('1'))//if(m_nState == STATE_SYM)
                        {
                            m_nState = STATE_WORD;
                            m_nMsgLength++;
                            m_sCompleteValue.clear();
                            m_sCompleteValue = lastPress;

                            m_nWordIndex=0;
                            m_lWord = m_dict->getWordList(&m_sCompleteValue);

                            m_nBeginWord = m_nEndWord;
                            m_nEndWord ++;
                            QString sChoosenWord = m_lWord.at(0);
                            addText(sChoosenWord);
                            updateCharCounter(ACTION_INS);
                        }

                        else
                        {
                            //m_nState = STATE_UNKNOWN;
                            m_sCompleteValue = m_sCompleteValue.left(m_sCompleteValue.length()-1);
                            //TODO: no knowing words
                        }
                    }
                }
            }
        }//End FastWrite routine
        else if(m_nInputSyle == IN_STYLE_DIGIT)//Only-Digit input selected
        {
            m_nState = STATE_SPACE;
            m_nMsgLength++;
            m_sCompleteValue = lastPress;
            m_nWordIndex=0;
            m_lWord.clear();
            m_lWord = m_dict->getWordList(&m_sCompleteValue);

            m_nEndWord ++;
            m_nBeginWord = m_nEndWord-1;
            addText(lastPress);
            updateCharCounter(ACTION_INS);
            m_sCompleteValue.clear();
        }
        else if(!m_bFastInput) //standard-input selected
        {

            if(!m_bTimerNewChar && lastPress.compare(m_sLastPress) == 0)
            {
                m_bNewChar = false;

                //Manage capitalization
                int nPos = m_cursorSmsText.position();
                nPos--;

                if(m_bWasCapitalized)
                    m_bMustCapitalize = true;
                else
                    m_bMustCapitalize = false;
                //

                nextWord();
                timerNewChar->start();
            }
            else
            {
                m_bNewChar = true;
                m_bWasCapitalized = false;
                m_nState = STATE_SYM;
                m_nMsgLength++;
                m_sCompleteValue.clear();
                m_sCompleteValue = lastPress;
                m_nWordIndex=0;
                m_lWord = m_dict->getWordList(&m_sCompleteValue,true);

                m_nBeginWord = m_nEndWord;
                m_nEndWord ++;
                QString sChoosenWord = m_lWord.at(0);
                addText(sChoosenWord);
                updateCharCounter(ACTION_INS);

                //reset timer
                m_bTimerNewChar = false;
                timerNewChar->start();
            }
        }
        m_sLastPress = lastPress;
    }
}

void Fastsms::nextWord()
{
    if(m_lWord.count() > 0)
    {
        m_nWordIndex++;

        if(m_nWordIndex >= m_lWord.count())
            m_nWordIndex = 0;

        updateText(m_lWord.at(m_nWordIndex));
    }
}

void Fastsms::prevWord()
{
    if(m_lWord.count() > 0)
    {
        m_nWordIndex--;

        if(m_nWordIndex < 0)
            m_nWordIndex = m_lWord.count() - 1;

        updateText(m_lWord.at(m_nWordIndex));
    }
}

void Fastsms::insertNewWord()
{
    m_dialNewWord = new QDialog(this);
    m_dialNewWord->setAttribute(Qt::WA_DeleteOnClose);
    inputNewWord = new QLineEdit(m_dialNewWord);
    Button *ok = createButton(tr("Ok"),SLOT(newWordInserted()));
    QGridLayout *insWordLayout = new QGridLayout();
    insWordLayout->addWidget(inputNewWord,0,0);
    insWordLayout->addWidget(ok,1,0);
    inputNewWord->setFocus();

    m_dialNewWord->setWindowTitle("Type new word");
    m_dialNewWord->setLayout(insWordLayout);
    m_dialNewWord->show();
}

void Fastsms::newWordInserted()
{
    QString *newWord = new QString(inputNewWord->text().toLower());
    if(!newWord->isEmpty())
    {
        qDebug() << "New word is:" << newWord->toAscii();
        m_bNeedSave = m_dict->addWord(newWord);
        m_nState = STATE_WORD;

        m_nBeginWord = m_cursorSmsText.anchor();
        m_nEndWord=m_nBeginWord + newWord->length();

        m_sCompleteValue.clear();
        m_sCompleteValue.append(m_dict->calculateKey(newWord));
        m_lWord = m_dict->getWordList(&m_sCompleteValue);
        m_nWordIndex=m_lWord.indexOf(*newWord);

        updateText(*newWord);

        for(int i=0; i<newWord->length();i++)
        {
            updateCharCounter(ACTION_INS);
        }
    }

    m_dialNewWord->close();

    this->setFocus();
}

/** Update sms and character counter */
void Fastsms::updateCharCounter(int action)
{
    if(action==ACTION_SEND)
    {
        m_nCharCount=160;
        m_nSMSCount=1;
    }
    else if(action==ACTION_INS)
    {
        m_nCharCount--;

        if(m_nCharCount == -1)
        {
            m_nSMSCount++;

            if(m_nSMSCount==2)
            {
                m_nCharCount=145;
            }
            else if(m_nSMSCount >= 3)
            {
                m_nCharCount=152;
            }
        }
    }
    else if(m_nCharCount > -1)
    {
        m_nCharCount++;

        if(m_nCharCount == 161)
        {
            m_nCharCount = 160;
        }
        else if(m_nCharCount == 153 && m_nSMSCount >= 3)
        {
            m_nCharCount = 0;
            m_nSMSCount--;
        }
        else if(m_nCharCount == 146 && m_nSMSCount == 2)
        {
            m_nCharCount = 0;
            m_nSMSCount--;
        }
    }

    ui->lblCharCount->setText(QString::number(m_nCharCount,10));
    ui->lblMessageCount->setText(QString::number(m_nSMSCount,10));
}

/** Delete cursors selection then insert sWord **/
void Fastsms::updateText(QString sWord)
{
    disconnect(m_smsEdit,SIGNAL(selectionChanged()),0,0);

    m_cursorSmsText.insertText(capitalizeIfNeeded(sWord));
    connect(m_smsEdit,SIGNAL(selectionChanged()),this,SLOT(cursorPosChanged()));

    qDebug() << "Write sms:" << m_smsEdit->toPlainText();
}

/** Add sWord in cursor current position, after deselect all **/
void Fastsms::addText(QString sWord)
{
    disconnect(m_smsEdit,SIGNAL(selectionChanged()),0,0);

    m_cursorSmsText.clearSelection();
    m_cursorSmsText.insertText(capitalizeIfNeeded(sWord));

    connect(m_smsEdit,SIGNAL(selectionChanged()),this,SLOT(cursorPosChanged()));
}

QString Fastsms::capitalizeIfNeeded(QString sWord)
{
    if(m_nInputSyle == IN_STYLE_CAPITALIZE)
    {
        if(m_sCapitalization.contains(sWord))
            m_bMustCapitalize = true;
        else if(m_dict->isSymbol(sWord[0]))
            m_bMustCapitalize = false;
        else if(sWord==" ")
            m_bMustCapitalize = false;


        if(!m_bFastInput && m_bNewChar)
        {
            int nPos = m_cursorSmsText.position();
            if(nPos > 0)
            {
                nPos--;

                if(m_sCapitalization.contains(m_smsEdit->toPlainText().at(nPos)))
                {
                    sWord[0]=sWord[0].toUpper();
                    m_bWasCapitalized = true;
                }
            }
            else
            {
                sWord[0]=sWord[0].toUpper();
                m_bWasCapitalized = true;
            }
        }
        else if(m_bMustCapitalize)
        {
            sWord[0]=sWord[0].toUpper();
        }


    }
    else if(m_nInputSyle == IN_STYLE_UPPER)
        sWord = sWord.toUpper();
    else if(m_nInputSyle == IN_STYLE_LOWER)
        sWord = sWord.toLower();



    return sWord;
}

void Fastsms::deleteChar()
{
    if(m_phoneNumber->hasFocus())
    {
        m_phoneNumber->backspace();
    }
    else
    {
        m_nState = STATE_DEL;
        updateCharCounter(ACTION_DEL);

        m_smsEdit->disconnect();
        m_cursorSmsText.clearSelection();
        m_cursorSmsText.deletePreviousChar();

        m_sCompleteValue.clear();

        cursorPosChanged();

        connect(m_smsEdit,SIGNAL(textChanged()),this,SLOT(smsChanged()));
        connect(m_smsEdit,SIGNAL(selectionChanged()),this,SLOT(cursorPosChanged()));
    }
}

void Fastsms::openPhoneBook()
{
    bool ok = false;
    int Pb = m_settings->value(SETTING_PHONEBOOK).toInt();



    if(Pb == PHONEBOOK_MYCONTACTS)
    {
        QDBusMessage msg = QDBusMessage::createMethodCall(
                "com.crochik.MyContacts", // --dest
                "/", // destination object path
                "com.crochik.MyContacts",
                "select_phone"
                );

        msg << "Opening Mycontacts";
        QList<QVariant> args;
        args.append("FastSMS");
        msg.setArguments(args);
        msg = QDBusConnection::sessionBus().call(msg);
        qDebug() << msg;
        ok = msg.type() != QDBusMessage::ErrorMessage;
    }

    else if((Pb) == PHONEBOOK_HORIZONTALCALL)
    {
        QDBusMessage msg = QDBusMessage::createMethodCall(
                "com.eitam.horizontalcall", // --dest
                "/com/eitam/horizontalcall", // destination object path
                "com.eitam.horizontalcall", // message name (w/o method)
                "top_application" // method
                );

        msg = QDBusConnection::sessionBus().call(msg);
        qDebug() << msg;
        ok = msg.type() != QDBusMessage::ErrorMessage;
    }

    if(!ok)
    {
        QString sPhoneBook;
        if((Pb) == PHONEBOOK_HORIZONTALCALL)
            sPhoneBook = "Horizontal-Call";
        else if (Pb == PHONEBOOK_MYCONTACTS)
            sPhoneBook = "My-Contacts";
        QDialog *dlCont = new QDialog(this);
        dlCont->setAttribute(Qt::WA_DeleteOnClose);

        dlCont->setWindowTitle("Can't open selected phonebook");
        QGridLayout *layout = new QGridLayout();
        QString sInfo = QString("\nInstall " + sPhoneBook + "\n"
                                "from Application Manager \n"
                                "or change it in Option Menu.\n");
        QLabel *msg = new QLabel(sInfo);
        layout->addWidget(msg,0,0);
        dlCont->setLayout(layout);
        dlCont->show();
    }
}

void Fastsms::sendSMS()
{
    QMessageService service;
    QMessageManager manager;

    //Find right account for SMS: default one doesnt work well
    QMessageAccountIdList lstMsgAccountId = manager.queryAccounts();

    if(!m_smsEdit->toPlainText().isEmpty() && !m_phoneNumber->text().isEmpty())
    {
        if(!lstMsgAccountId.isEmpty())
        {
            qDebug() << "SMS text" << m_smsEdit->toPlainText();
            for(int i = 0; i < lstMsgAccountId.count(); ++i)
            {
                QMessageAccount account(lstMsgAccountId[i]);
                bool isSmsAccount = (account.messageTypes() & QMessage::Sms) > 0;

                if(isSmsAccount)
                {
                    QMessage myMessage;
                    myMessage.setType(QMessage::Sms);
                    myMessage.setParentAccountId(lstMsgAccountId[i]);
                    myMessage.setTo(QMessageAddress(QMessageAddress::Phone, m_phoneNumber->text()));
                    myMessage.setBody(m_smsEdit->toPlainText());

                    qDebug() << "state:" << service.send(myMessage);
                    break;
                }
            }
        }
    }

    m_smsEdit->disconnect();

    m_nState = STATE_SEND;
    //m_cursorSmsText.select(QTextCursor::Document);
    m_smsEdit->clear();
    m_sCompleteValue.clear();
    m_phoneNumber->clear();
    m_lWord.clear();
    m_nWordIndex=0;

    //update char and sms count
    updateCharCounter(ACTION_SEND);
    //

    connect(m_smsEdit,SIGNAL(textChanged()),this,SLOT(smsChanged()));
    connect(m_smsEdit,SIGNAL(selectionChanged()),this,SLOT(cursorPosChanged()));
}

void Fastsms::saveDictionaryOnExit()
{
    if(m_bNeedSave)
    {
        m_dict->saveDictionaryToFile();
    }
}

bool Fastsms::addRecipient(const QString &phone, const QString &name)
{
    m_phoneNumber->clear();
    m_phoneNumber->setText(phone);

    this->activateWindow();
    this->raise();

    return true;
}

bool Fastsms::registerService()
{
    QDBusConnection connection = QDBusConnection::sessionBus();
    if ( !connection.registerService("it.rsc.fastsms") ) {
        qDebug() << "error registering service.";
        return false;
    }
    if ( !connection.registerObject("/", this) ) {
        qDebug() << "error registering object";
        return false;
    }

    return true;
}

void Fastsms::chageInputStyle()
{
    QAction *action = qobject_cast<QAction *>(sender());

    if(action->data() == IN_STYLE_FW_OFF)
    {
        action->setText("Enable FastWrite");
        action->setData(IN_STYLE_FW_ON);
        m_bFastInput = false;
        ui->lbl_fastWrite->setText("FastWrite OFF");
    }
    else if(action->data() == IN_STYLE_FW_ON)
    {
        action->setText("Disable FastWrite");
        action->setData(IN_STYLE_FW_OFF);
        m_bFastInput = true;
        ui->lbl_fastWrite->setText("FastWrite ON");
    }
    else
    {
        m_nInputSyle = action->data().toInt();
        ui->lbl_format->setText(action->text());
    }
}

void Fastsms::alarmNewChar()
{
    m_bTimerNewChar = true;
}

void Fastsms::donate()
{
    QDesktopServices::openUrl(QUrl("https://www.paypal.com/cgi-bin/webscr?cmd=_s-xclick&hosted_button_id=V353RBQ6TATN6"));
}

void Fastsms::doNothing()
{


}

//void Fastsms::changeEvent(QEvent *e)
//{
//    QWidget::changeEvent(e);
//    switch (e->type())
//    {
//    case QEvent::LanguageChange:
//        //ui->retranslateUi(this);
//        break;
//    default:
//        break;
//    }
//}

Fastsms::~Fastsms()
{
    delete ui;
}
