/*
	This file is part of Faster Application Manager.

	Faster Application Manager is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Faster Application Manager is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Faster Application Manager.  If not, see <http://www.gnu.org/licenses/>.

	(C) Heikki Holstila 2010
*/

#include "packageview.h"
#include "ui_packageview.h"
#include "package.h"
#include "filterselect.h"
#include "confirmdialog.h"
#include "dimmer.h"
#include "packageselector.h"
#include "help.h"
#include "aaptinterface.h"
#include "logview.h"
#include "sortselector.h"
#include "settings.h"
#include "searchoptions.h"

PackageListWidgetItem::PackageListWidgetItem(Package* p_, QString name_) : QListWidgetItem(name_)
{
	iPackage = p_;
}


void ListItemDelegate::paint(QPainter *painter, const QStyleOptionViewItem &option, const QModelIndex &index) const
{
	//QTime t;
	//t.start();

	QString name = index.data(UserRoleName).toString();
	QString version = index.data(UserRoleVersion).toString();
	QString desc = index.data(UserRoleDescShort).toString();
	Package::operation marked = (Package::operation) index.data(UserRoleMarked).toInt();
	bool installed = index.data(UserRoleInstalled).toBool();
	bool upgradeable = index.data(UserRoleUpgradeable).toBool();
	int statfilter = index.data(UserRoleCurrentStatFilter).toInt();
	int catfilter = index.data(UserRoleCurrentCatFilter).toInt();
	QString upg_version = index.data(UserRoleAvailVersion).toString();

	painter->save();
	QRect r = option.rect;

	QLinearGradient gradientBase(r.topLeft(), r.bottomLeft());
	QColor base = option.palette.color(QPalette::Window);
	QColor base2 = base;
	int r1=base.red()+15;
	int g1=base.green()+15;
	int b1=base.blue()+15;
	if( r1>255 ) r1=255;
	if( g1>255 ) g1=255;
	if( b1>255 ) b1=255;
	int r2=base2.red()-20;
	int g2=base2.green()-20;
	int b2=base2.blue()-20;
	if( r2<0 ) r2=0;
	if( g2<0 ) g2=0;
	if( b2<0 ) b2=0;
	base.setRgb( r1, g1, b1 );
	base2.setRgb( r2, g2, b2 );
	gradientBase.setColorAt(0, base);
	gradientBase.setColorAt(1, base2);

	painter->fillRect(r, gradientBase);
	painter->drawLine(QPoint(r.left(),r.bottom()), QPoint(r.right(),r.bottom()));

	QPixmap icon = index.data(Qt::DecorationRole).value<QPixmap>();
	if( icon.isNull() ) {
		// use default icon
		painter->drawPixmap( r.left(), r.top()+4, 48, 48, iDefaultIcon );
	} else {
		painter->drawPixmap( r.left(), r.top()+4, 48, 48, icon );
	}

	QPixmap statusicon;
	if( marked == Package::PkgOpNone )
	{
		if( installed && upgradeable )
			statusicon = iIconPkgNoOpInstalledUpgradeable;
		else if( installed )
			statusicon = iIconPkgNoOpInstalled;
		else if( !installed )
			statusicon = iIconPkgNoOpNotInstalled;
	} else if( marked == Package::PkgOpInstallUpgrade ) {
		if( upgradeable )
			statusicon = iIconPkgUpgrade;
		else
			statusicon = iIconPkgInstall;
	} else if( marked == Package::PkgOpRemove ) {
		statusicon = iIconPkgRemove;
	}

	QString showVer = "";
	if( upgradeable && (statfilter==Package::PkgStatUpgradeable ||
						(statfilter==Package::PkgStatUnknown && marked==Package::PkgOpInstallUpgrade) ||
						(catfilter==PackageView::CatFilterAllMarked && marked==Package::PkgOpInstallUpgrade) ))
	{
		showVer = upg_version;
	} else {
		showVer = version;
	}

	int ver_w = 0;
	if( QApplication::desktop()->width() > QApplication::desktop()->height() )
	{
		r = option.rect;
		r.setRight( r.right()-statusicon.width()-4 );
		painter->drawText(r, Qt::AlignTop|Qt::AlignRight, showVer, &r);
		ver_w = r.width();
	}

	r = option.rect;
	r.setRight( r.right()-statusicon.width()-4-ver_w );  //does not work as it should?
	QFont f = painter->font();
	f.setBold(true);
	painter->setFont(f);
	painter->drawText(r.left()+iDefaultIcon.width()+2, r.top(), r.width(), r.height(), Qt::AlignTop|Qt::AlignLeft, name, &r);
	f.setBold(false);
	painter->setFont(f);

	f.setPointSize( f.pointSize()-4 );
	painter->setFont(f);
	r = option.rect;
	painter->drawText(r.left()+iDefaultIcon.width()+2, r.top(), r.width(), r.height(), Qt::AlignBottom|Qt::AlignLeft, desc, &r);

	r = option.rect;
	painter->drawPixmap(r.right()-statusicon.width()-2, r.top()+4, 24, 24, statusicon);

	painter->restore();

	//if( t.elapsed()>=100 )
	//qDebug() << name << t.elapsed();
}

void ListItemDelegate::loadIcons()
{
	iDefaultIcon = QPixmap(":/icons/icons/appdefault.png");
	iIconPkgInstall = QPixmap(":/icons/icons/pkg_install.png");
	iIconPkgUpgrade = QPixmap(":/icons/icons/pkg_upgrade.png");
	iIconPkgRemove = QPixmap(":/icons/icons/pkg_remove.png");
	iIconPkgNoOpInstalled = QPixmap(":/icons/icons/pkg_nop_installed.png");
	iIconPkgNoOpNotInstalled = QPixmap(":/icons/icons/pkg_nop_notinstalled.png");
	iIconPkgNoOpInstalledUpgradeable = QPixmap(":/icons/icons/pkg_nop_instupgr.png");
}


QSize ListItemDelegate::sizeHint(const QStyleOptionViewItem&, const QModelIndex&) const
{
	return QSize(400, 58);
}


PackageView::PackageView(QWidget *parent) : QMainWindow(parent), ui(new Ui::PackageView)
{
	iMainWindow = dynamic_cast<MainWindow*>(parent);
	ui->setupUi(this);
#ifdef Q_WS_MAEMO_5
	this->setAttribute(Qt::WA_Maemo5StackedWindow);
	this->setWindowFlags(Qt::Window);
	this->setAttribute(Qt::WA_Maemo5AutoOrientation);
#endif
	iSettings = 0;
	iAptInterface = 0;

	connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));

	iCatFilterLabels
			<< tr("All marked packages")	// a special case
			<< tr("All user categories")
			<< tr("\tDesktop")
			<< tr("\tEducation")
			<< tr("\tGames")
			<< tr("\tGraphics")
			<< tr("\tInternet & Networking")
			<< tr("\tLocation & Navigation")
			<< tr("\tMultimedia")
			<< tr("\tOffice")
			<< tr("\tOther")
			<< tr("\tProgramming")
			<< tr("\tScience")
			<< tr("\tSystem")
			<< tr("\tUtilities")
			<< tr("All packages (ADVANCED)")
			<< tr("Blacklisted packages");	// a special case

	iCatFilterStrings
			<< ""
			<< "user/"
			<< "user/desktop"
			<< "user/education"
			<< "user/games"
			<< "user/graphics"
			<< "user/network"
			<< "user/navigation"
			<< "user/multimedia"
			<< "user/office"
			<< "user/other"
			<< "user/development"
			<< "user/science"
			<< "user/system"
			<< "user/utilities"
			<< ""
			<< "";

	iDefaultCatFilter = CatFilterAllUser;
	iSelectedCatFilter = iDefaultCatFilter;

	iStatFilterLabels
			<< tr("All")
			<< tr("Not installed")
			<< tr("Upgradeable")
			<< tr("Installed");

	iSelectedStatFilter = Package::PkgStatUnknown;
	iSortOrder = SortAlpha;

	iDimmer = new dimmer(this);

	ui->searchBar->hide();

	iListCoverLabel = new QLabel(ui->listWidget);
	iListCoverLabel->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
	iListCoverLabel->setAlignment(Qt::AlignCenter);

	iKeyFilter = new KeyEventGrabber(this);
	ui->listWidget->installEventFilter(iKeyFilter);

	iSortNoticeShown = false;

	iSearchPkgName = false;
	iSearchDisplayName = true;
	iSearchDescShort = false;
	iSearchDescLong = false;

	// fine-tune kinetic scrolling parameters
	QAbstractKineticScroller* listscroller = ui->listWidget->property("kineticScroller").value<QAbstractKineticScroller*>();
	if( listscroller )
	{
		//qDebug() << listscroller->dragInertia() << listscroller->decelerationFactor()
		//		<< listscroller->minimumVelocity() << listscroller->maximumVelocity();
		listscroller->setDecelerationFactor(0.75); // default is 0.85
		listscroller->setDragInertia(0.60);	// default is 0.85
		listscroller->setMaximumVelocity(1800); // default is 3500
	}
}

PackageView::~PackageView()
{
	delete iListCoverLabel;
	delete iKeyFilter;
	delete iDimmer;
    delete ui;
}

void PackageView::orientationChanged()
{
	ui->listWidget->scroll(1,1);	// this causes all items to be repainted
	iListCoverLabel->setGeometry( ui->listWidget->rect() );
}

void PackageView::resizeEvent(QResizeEvent* event)
{
	iListCoverLabel->setGeometry( ui->listWidget->rect() );
	QWidget::resizeEvent(event);
}

bool PackageView::doFilterCategory(Package* pkg)
{
	if( pkg->section()=="user/hidden" )
		return false;

	if( pkg->isBlacklisted() )
	{
		if( iSelectedCatFilter == CatFilterBlacklisted )
			return true;
		if( pkg->isInstalled() && (iSelectedStatFilter==Package::PkgStatInstalled || iSelectedStatFilter==Package::PkgStatUnknown) )
			return true;
		if( iSelectedCatFilter != CatFilterBlacklisted )
			return false;
	} else if( !pkg->isBlacklisted() && iSelectedCatFilter == CatFilterBlacklisted )
		return false;

	if( iSelectedCatFilter==CatFilterAllMarked ) {
		if( pkg->isMarkedForOperation() )
			return true;
		else
			return false;
	}
	if( pkg->section().startsWith( iCatFilterStrings.at(iSelectedCatFilter) ) )
		return true;

	return false;
}

QString PackageView::generateSortString(Package* pkg)
{
	QString sortstr;

	if( iSortOrder==SortAlpha ) {
		sortstr = pkg->displayName();
	}
	else if( iSortOrder==SortDateDesc ) {
		Package* upg = 0;
		if( pkg->isUpgradeable() )
			upg = pkg->availablePackage();

		if( (!upg && pkg->date().isValid()) || iSelectedStatFilter==Package::PkgStatInstalled )
			sortstr = pkg->date().toString("yyyy-MM-dd hh:mm:ss");
		else if( upg && upg->date().isValid() ) {
			sortstr = upg->date().toString("yyyy-MM-dd hh:mm:ss");
		} else {
			sortstr = "";
			iPackagesEmptySort++;
		}
	}
	else if( iSortOrder==SortSizeDesc ) {
		if( pkg->isInstalled() )
			sortstr = QString("%1").arg(pkg->installedSize()*1024, 12 );
		else
			sortstr = QString("%1").arg(pkg->size(), 12 );
	}

	//qDebug() << sortstr;

	return sortstr;
}

void PackageView::openWin()
{
	ui->listWidget->clear();
	ui->listWidget->setSortingEnabled(true);
	iPackagesEmptySort = 0;

	if( iSortOrder==SortDateDesc || iSortOrder==SortSizeDesc )
		ui->listWidget->sortItems(Qt::DescendingOrder);
	else
		ui->listWidget->sortItems(Qt::AscendingOrder);

	delete ui->listWidget->itemDelegate();
	ListItemDelegate* delegate = new ListItemDelegate(ui->listWidget);
	delegate->loadIcons();
	ui->listWidget->setItemDelegate( delegate );

	if( !ui->searchBar->isVisible() )
	{
		if( iSelectedStatFilter == Package::PkgStatNotInstalled || iSelectedStatFilter == Package::PkgStatUnknown ||
			iSelectedCatFilter == CatFilterAllMarked || iSelectedCatFilter == CatFilterBlacklisted )
		{
			QHashIterator<QString, Package*> i( *iAptInterface->packagesAvailable() );
			while (i.hasNext())
			{
				i.next();
				Package* inst = iAptInterface->packagesInstalled()->value(i.value()->name(),0);
				if( doFilterCategory(i.value()) && !inst )
					addListItem(i.value(), generateSortString(i.value()));
			}
		}
		if( iSelectedStatFilter == Package::PkgStatInstalled || iSelectedStatFilter == Package::PkgStatUpgradeable ||
			iSelectedStatFilter == Package::PkgStatUnknown || iSelectedCatFilter == CatFilterAllMarked ||
			iSelectedCatFilter == CatFilterBlacklisted )
		{
			QHashIterator<QString, Package*> i( *iAptInterface->packagesInstalled() );
			while (i.hasNext())
			{
				i.next();
				if( iSelectedStatFilter == Package::PkgStatUpgradeable ) {
					if( i.value()->isUpgradeable() && doFilterCategory(i.value()) )
						addListItem(i.value(), generateSortString(i.value()));
				} else {
					if( doFilterCategory(i.value()) )
						addListItem(i.value(), generateSortString(i.value()));
				}
			}
		}
	} else {
		for( int j=0; j<iSearchResults.count(); j++ )
		{
			addListItem(iSearchResults.at(j), generateSortString( iSearchResults.at(j) ));
		}
	}

	iListCoverLabel->hide();
	iListCoverLabel->setAutoFillBackground(false);

	updateLabel();

	if( iSelectedStatFilter==Package::PkgStatUpgradeable && ui->listWidget->count()>0 &&
		iSelectedCatFilter != CatFilterAllMarked && iSelectedCatFilter != CatFilterBlacklisted )
	{
		ui->actionUpgrade_all->setVisible(true);
	} else {
		ui->actionUpgrade_all->setVisible(false);
	}

	if( iSelectedCatFilter == CatFilterBlacklisted && ui->listWidget->count()>0 ) {
		ui->actionRestore_all->setVisible(true);
	} else {
		ui->actionRestore_all->setVisible(false);
	}

	show();

	if( !ui->searchBar->isVisible() ) {
		ui->listWidget->setFocusPolicy(Qt::StrongFocus);
		ui->listWidget->setFocus();
	} else {
		ui->listWidget->setFocusPolicy(Qt::NoFocus);
		ui->lineEdit->setFocus();
	}

	if( ui->listWidget->count() == 0 )
	{
		iListCoverLabel->setGeometry( ui->listWidget->rect() );
		iListCoverLabel->setText("No packages");
		iListCoverLabel->show();
	}

	if( ui->listWidget->count()>1 && iPackagesEmptySort == ui->listWidget->count() && !iSortNoticeShown ) {
		ConfirmDialog d(false, this);
		QString msg = "No shown packages currently have the required information for sorting by this criterion. The list is unsorted.";
		if( iSortOrder == SortDateDesc && !iSettings->qsettings()->value("fetch_dates",false).toBool() )
			msg += " You can enable date fetching in the options menu.";
		d.setText("Notice", msg);
		d.exec();
		iSortNoticeShown = true;
	}
}

void PackageView::enableMenu()
{
	ui->menuMenu->setEnabled(true);
}

void PackageView::disableMenu()
{
	ui->menuMenu->setEnabled(false);
}

void PackageView::addListItem(Package* pkg_, QString listname_)
{
	PackageListWidgetItem* p = new PackageListWidgetItem( pkg_, listname_ );

	if( pkg_ != 0 )
	{
		QString name = pkg_->name();
		if( pkg_->maemoDisplayName()!="" )
			name = pkg_->maemoDisplayName();
		p->setData(UserRoleName, name);
	} else {
		p->setData(UserRoleName, listname_);
	}

	if( pkg_ != 0 )
	{
		p->setData(UserRoleDescShort, pkg_->descShort());
		p->setData(UserRoleVersion, pkg_->version());
		p->setData(UserRoleMarked, (int)pkg_->markedOperation());
		p->setData(UserRoleInstalled, pkg_->isInstalled());
		p->setData(UserRoleUpgradeable, pkg_->isUpgradeable());
		p->setData(UserRoleAvailVersion, pkg_->upgradeableVersion());
		p->setData(UserRoleCurrentStatFilter, (int)iSelectedStatFilter);
		p->setData(UserRoleCurrentCatFilter, iSelectedCatFilter);

		//qDebug()<<pkg_->name();

		pkg_->convertIcon();
		p->setData(Qt::DecorationRole, *pkg_->icon());
	}
	ui->listWidget->addItem( p );
}

void PackageView::closeEvent(QCloseEvent *event)
{
	if( !iAptInterface ) {
		resetWindow();
		event->accept();
	}

	if( iDimmer->busy() )
	{
		iAptInterface->cancel();
		event->ignore();
		return;
	}

#ifdef Q_WS_MAEMO_5
	if( iAptInterface->numSelectedPackages() == 0 )
	{		
		resetWindow();
		event->accept();
	} else {
		QString c;
		c.setNum( iAptInterface->numSelectedPackages() );
		ConfirmDialog d(true, this);
		d.setText("Returning to main menu", QString("Clear %1 package selection(s) and lose all the pending changes?").arg(iAptInterface->numSelectedPackages()));
		if( d.exec() ) {
			resetWindow();
			event->accept();
		} else {
			event->ignore();
		}
	}
#else	// for simulator
	resetWindow();
	event->accept();
#endif
}

void PackageView::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void PackageView::on_btn_Commit_clicked()
{
	QStringList pkgnames;

	QHashIterator<QString, Package*> i( *iAptInterface->packagesAvailable() );
	while (i.hasNext())
	{
		i.next();

		if( i.value()->markedOperation() == Package::PkgOpInstallUpgrade )
			pkgnames << i.value()->name();
		if( i.value()->markedOperation() == Package::PkgOpRemove ) {
			qDebug() << "warning: trying to add package marked from the wrong list";
			//pkgnames << i.value()->name() + "-";
		}
	}

	QHashIterator<QString, Package*> r( *iAptInterface->packagesInstalled() );
	while (r.hasNext())
	{
		r.next();

		if( r.value()->markedOperation() == Package::PkgOpInstallUpgrade )
			pkgnames << r.value()->name();
		if( r.value()->markedOperation() == Package::PkgOpRemove )
			pkgnames << r.value()->name() + "-";
	}

	iMainWindow->busyDialog(true, "Operation in progress", "Reading dependencies");

	iAptInterface->setProcessPackages(pkgnames);
	iAptInterface->addQueuedOperation(AAptInterface::ModeAptGetSimulate);
	iAptInterface->run(iDimmer);
}

void PackageView::on_actionClear_selections_triggered()
{
	QString c;
	c.setNum( iAptInterface->numSelectedPackages() );
	ConfirmDialog d(true, this);
	d.setText(tr("Confirmation"), tr("Clear ") + c + tr(" package selection(s) and lose all the pending changes?"));
	if( d.exec() )
	{
		clearSelections();
		openWin();
	}
}

void PackageView::clearSelections()
{
	QHashIterator<QString, Package*> i( *iAptInterface->packagesInstalled() );
	while (i.hasNext())
	{
		i.next();
		i.value()->setMarkedForOperation(Package::PkgOpNone);
	}
	QHashIterator<QString, Package*> a( *iAptInterface->packagesAvailable() );
	while (a.hasNext())
	{
		a.next();
		a.value()->setMarkedForOperation(Package::PkgOpNone);
	}
	iAptInterface->setNumSelectedPackages(0);
}

void PackageView::on_listWidget_itemClicked(QListWidgetItem* item)
{
	Package* pkg = dynamic_cast<PackageListWidgetItem*>(item)->package();
	if( !pkg )
		return;

	bool bl = pkg->isBlacklisted();

	PackageSelector s(pkg, iAptInterface, this);
	s.exec();

	pkg->setMarkedForOperation( s.selectedOperation() );
	item->setData(UserRoleMarked, (int)s.selectedOperation());
	updateLabel();

	if( pkg->isBlacklisted() != bl ) {
		openWin();
	}
}

void PackageView::updateLabel()
{
	QString s;
	s.setNum( iAptInterface->numSelectedPackages() );
	QString s2;
	s2.setNum( ui->listWidget->count() );
	QString statlabel = iStatFilterLabels.at(iSelectedStatFilter);
	if( iSelectedCatFilter == CatFilterAllMarked || iSelectedCatFilter == CatFilterBlacklisted )
		statlabel = "All";
	ui->label->setText("<font size=\"-2\"><b>" + s + "</b> package(s) marked<br>"
					   + "Showing: <b>" + statlabel + "</b><br>"
					   + "Filter: " + iCatFilterLabels.at(iSelectedCatFilter) + " - " + s2 + " package(s)</font>");

	if( iAptInterface->numSelectedPackages()==0 ) {
		ui->btn_Commit->setEnabled(false);
		ui->actionClear_selections->setVisible(false);
		ui->actionSave_selections->setVisible(false);
	} else {
		ui->btn_Commit->setEnabled(true);
		ui->actionClear_selections->setVisible(true);
		ui->actionSave_selections->setVisible(true);
	}
}

void PackageView::on_btn_CategoryFilter_clicked()
{
	FilterSelect f("Category filter", this);
	f.setList( iCatFilterLabels, iSelectedCatFilter );

	bool s = f.exec();

	if( s )
		iSelectedCatFilter = f.selection();

	if( iSelectedCatFilter == CatFilterAllMarked || iSelectedCatFilter == CatFilterBlacklisted ) {
		ui->btn_StatusFilter->setEnabled(false);
	} else {
		ui->btn_StatusFilter->setEnabled(true);
	}

	if( s ) {
		iListCoverLabel->setText("Loading...");
		iListCoverLabel->setAutoFillBackground(true);
		iListCoverLabel->show();
		QApplication::processEvents();
		iSortNoticeShown = false;

		openWin();
	}
}

void PackageView::setStatFilter(Package::packageStatus f_)
{
	iSelectedStatFilter = f_;
}

void PackageView::on_btn_StatusFilter_clicked()
{
	FilterSelect f("Status filter", this);
	f.setList( iStatFilterLabels, iSelectedStatFilter );

	int oldfilter = iSelectedStatFilter;

	bool s = f.exec();

	if( s ) {
		iSelectedStatFilter = (Package::packageStatus)f.selection();

		iListCoverLabel->setText("Loading...");
		iListCoverLabel->setAutoFillBackground(true);
		iListCoverLabel->show();
		QApplication::processEvents();
		iSortNoticeShown = false;

		openWin();

		if( oldfilter==Package::PkgStatInstalled && iSelectedStatFilter!=Package::PkgStatInstalled &&
			iAptInterface->needListOrDateRefresh() )
		{
			iMainWindow->setNextOperation(MainWindow::OpOpenPkgView);
			iMainWindow->busyDialog(true,"Operation in progress","Reading the rest of the package lists");

			if( iAptInterface->needRepoRefresh() )
				iAptInterface->addQueuedOperation(AAptInterface::ModeAptGetUpdate);

			iAptInterface->addQueuedOperation(AAptInterface::ModeReadPackages);
			iAptInterface->addQueuedOperation(AAptInterface::ModeFetchDates);
			iAptInterface->run(iDimmer);
		}
	}
}

void PackageView::resetWindow()
{
	iMainWindow->resetIdlingTime();

	iAptInterface->writeBlacklist();

	ui->btn_StatusFilter->setEnabled(true);
	//iSelectedCatFilter = iDefaultCatFilter;
	//iSortOrder = SortAlpha;
	iSortNoticeShown = false;

	clearSelections();
	clearSearch();
}

void PackageView::on_actionHelp_triggered()
{
	Help h(this);
	h.exec();
}

void PackageView::on_btn_searchClose_clicked()
{
	clearSearch();

	iListCoverLabel->setText("Loading...");
	iListCoverLabel->setAutoFillBackground(true);
	iListCoverLabel->show();
	QApplication::processEvents();

	openWin();
}

void PackageView::clearSearch()
{
	ui->lineEdit->clear();
	ui->searchBar->hide();
	ui->toolBar->show();
	ui->listWidget->setFocusPolicy(Qt::StrongFocus);

	iSearchResults.clear();
}

void PackageView::on_actionSearch_triggered()
{
	if( ui->searchBar->isVisible() )
		return;

	ui->listWidget->setFocusPolicy(Qt::NoFocus);
	ui->searchLabel->setText( QString("%1 results").arg(ui->listWidget->count()) );
	ui->toolBar->hide();
	ui->searchBar->show();
	ui->lineEdit->setFocus();
	iPrevSearchText = "";
}

void PackageView::on_lineEdit_textEdited(QString text)
{
	if( !ui->searchBar->isVisible() )
		return;

	if( text=="" ) {
		on_btn_searchClose_clicked();
		return;
	}

	if( iPrevSearchText.length() > text.length() )
	{
		iListCoverLabel->setText("Loading...");
		iListCoverLabel->setAutoFillBackground(true);
		iListCoverLabel->show();
		QApplication::processEvents();

		ui->searchBar->hide();
		openWin();
		ui->searchBar->show();
		ui->toolBar->hide(); // ensure it stays hidden
		ui->lineEdit->setFocus();
	}

	iPrevSearchText = text;

	QList<Package*> packages;
	iSearchResults.clear();

	for( int i=0; i<(ui->listWidget->count()); i++ ) {
		packages.append( dynamic_cast<PackageListWidgetItem*>( ui->listWidget->item(i) )->package() );
	}

	if( text.startsWith(":") ) {
		for( int i=0; i<packages.count(); i++ ) {
			if( packages.at(i) ) {
				if( (iSearchPkgName && packages.at(i)->name().startsWith(text.mid(1), Qt::CaseInsensitive)) ||
					(iSearchDisplayName && packages.at(i)->displayName().startsWith(text.mid(1), Qt::CaseInsensitive)) ||
					(iSearchDescShort && packages.at(i)->descShort().startsWith(text.mid(1), Qt::CaseInsensitive)) ||
					(iSearchDescLong && packages.at(i)->descLong().startsWith(text.mid(1), Qt::CaseInsensitive)) )
				{
					iSearchResults.append( packages.at(i) );
				}
			}
		}
	} else {
		for( int i=0; i<packages.count(); i++ ) {
			if( packages.at(i) ) {
				if( (iSearchPkgName && packages.at(i)->name().contains(text, Qt::CaseInsensitive)) ||
					(iSearchDisplayName && packages.at(i)->displayName().contains(text, Qt::CaseInsensitive)) ||
					(iSearchDescShort && packages.at(i)->descShort().contains(text, Qt::CaseInsensitive)) ||
					(iSearchDescLong && packages.at(i)->descLong().contains(text, Qt::CaseInsensitive)) )
				{
					iSearchResults.append( packages.at(i) );
				}
			}
		}
	}

	ui->searchLabel->setText( QString("%1 results").arg( iSearchResults.count()) );

	openWin();
}

void PackageView::setSearchText(QString text)
{
	ui->lineEdit->setText(text);
	on_lineEdit_textEdited(text);
}

KeyEventGrabber::KeyEventGrabber(QObject* parent) : QObject(parent)
{
}

bool KeyEventGrabber::eventFilter(QObject *obj, QEvent *event)
{
	if( event->type() == QEvent::KeyPress ) {
		QString text = dynamic_cast<QKeyEvent*>(event)->text();
		int key = dynamic_cast<QKeyEvent*>(event)->key();
		if( (text.trimmed() != "" || text==" ") && key!=Qt::Key_Backspace ) {
			dynamic_cast<PackageView*>(this->parent())->on_actionSearch_triggered();
			dynamic_cast<PackageView*>(this->parent())->setSearchText( text );
			return true;
		}
	}
	return QObject::eventFilter(obj, event);
}

void PackageView::on_pushButton_searchOptions_clicked()
{
	SearchOptions s(this);
	s.setSelections(iSearchPkgName, iSearchDisplayName, iSearchDescShort, iSearchDescLong);
	if( s.exec() )
	{
		iSearchPkgName = s.searchPkgName();
		iSearchDisplayName = s.searchDisplayName();
		iSearchDescShort = s.searchDescShort();
		iSearchDescLong = s.searchDescLong();

		on_lineEdit_textEdited( ui->lineEdit->text() );
	}
}

void PackageView::setSearchOptions(bool pkgname, bool dispname, bool dshort, bool dlong)
{
	iSearchPkgName = pkgname;
	iSearchDisplayName = dispname;
	iSearchDescShort = dshort;
	iSearchDescLong = dlong;
}

void PackageView::on_actionUpgrade_all_triggered()
{
	for( int i=0; i<ui->listWidget->count(); i++ )
	{
		Package* pkg = dynamic_cast<PackageListWidgetItem*>(ui->listWidget->item(i))->package();
		if( pkg && pkg->isUpgradeable() ) {
			pkg->setMarkedForOperation( Package::PkgOpInstallUpgrade );
			ui->listWidget->item(i)->setData(UserRoleMarked, (int)Package::PkgOpInstallUpgrade);
		}
	}
	updateLabel();
}

void PackageView::on_actionView_log_triggered()
{
	QByteArray log = iAptInterface->readLogFile();
	LogView l(log, this);
	l.exec();
}

void PackageView::on_btn_Sort_clicked()
{
	SortSelector s(iSortOrder, this);
	if( s.exec() ) {
		iSortOrder = s.selectedOperation();
		iSortNoticeShown = false;
		openWin();
	}
}

void PackageView::on_actionLoad_selections_triggered()
{
	if( iAptInterface->numSelectedPackages() > 0 ) {
		ConfirmDialog d(true, this);
		d.setText("Confirmation", "Proceed loading selections? All current selections will be cleared.");
		if( !d.exec() )
			return;
	}

	clearSelections();

	QStringList unknownList;
	QStringList wrongverList;
	int success=0;
	int errors=0;

	QFile f("/root/.fapman/selections.list");
	if( f.open(QIODevice::ReadOnly | QIODevice::Text ) )
	{
		while(!f.atEnd()) {
			QString line = f.readLine().trimmed();
			QStringList parts = line.split(' ');
			if( parts.count()==3 ) {
				Package* pkgAv = iAptInterface->packagesAvailable()->value(parts.at(0),0);
				Package* pkgIn = iAptInterface->packagesInstalled()->value(parts.at(0),0);
				if( parts.at(2)=="install" ) {
					if( !pkgAv && !pkgIn ) {
						unknownList << parts.at(0);
					}
					else if( pkgIn ) {
						wrongverList << parts.at(0);
					}
					else if( pkgAv ) {
						if( pkgAv->version() == parts.at(1) && !pkgAv->isInstalled() ) {
							pkgAv->setMarkedForOperation(Package::PkgOpInstallUpgrade);
							success++;
						} else {
							wrongverList << parts.at(0);
						}
					}

				} else if( parts.at(2)=="upgrade" ) {
					if( !pkgAv && !pkgIn ) {
						unknownList << parts.at(0);
					}
					else if( (pkgAv && !pkgIn) || (pkgIn && !pkgAv) ) {
						wrongverList << parts.at(0);
					}
					else if( pkgIn && pkgAv ) {
						if( pkgIn->version() == parts.at(1) && pkgIn->isInstalled() && pkgIn->isUpgradeable() ) {
							pkgIn->setMarkedForOperation(Package::PkgOpInstallUpgrade);
							success++;
						} else {
							wrongverList << parts.at(0);
						}
					}

				} else if( parts.at(2)=="remove" ) {
					if( !pkgAv && !pkgIn ) {
						unknownList << parts.at(0);
					}
					else if( pkgAv && !pkgIn ) {
						wrongverList << parts.at(0);
					}
					else if( pkgIn ) {
						if( pkgIn->version() == parts.at(1) && pkgIn->isInstalled() ) {
							pkgIn->setMarkedForOperation(Package::PkgOpRemove);
							success++;
						} else {
							wrongverList << parts.at(0);
						}
					}
				} else {
					errors++;
				}
			} else if( line!="" ){
				errors++;
			}
		}
		f.close();
	}

	ConfirmDialog d(false, this);
	QString msg;
	msg += QString("<b>%1 successful</b><br>").arg(success);
	if( wrongverList.count() > 0 ) {
		msg += QString("%1 wrong versions/changed statuses (not selected):<br>").arg(wrongverList.count());
		msg += "<font size=\"-1\">";
		for( int i=0; i<wrongverList.count(); i++ ) {
			msg += wrongverList.at(i) + " ";
		}
		msg += "</font><br>";
	}
	if( unknownList.count() > 0 ) {
		msg += QString("%1 unknown packages:<br>").arg(unknownList.count());
		msg += "<font size=\"-1\">";
		for( int i=0; i<unknownList.count(); i++ ) {
			msg += unknownList.at(i) + " ";
		}
		msg += "</font><br>";
	}
	if( errors>0 || (wrongverList.count()==0 && unknownList.count()==0) ) {
		msg += QString("%1 errors<br>").arg(errors);
	}
	if( success==0 && wrongverList.count()==0 && unknownList.count()==0 )
		msg = "No stored selections";
	QString title;
	if( success > 0 )
		title = "Selections loaded";
	else
		title = "No selections loaded";
	d.setText(title, msg);
	d.exec();

	if( success > 0 ) {
		ui->btn_StatusFilter->setEnabled(false);
		iSelectedCatFilter = CatFilterAllMarked;
	}
	openWin();
}

void PackageView::on_actionSave_selections_triggered()
{
	if( iAptInterface->numSelectedPackages() == 0 )
		return;

	QFile f("/root/.fapman/selections.list");
	bool fail = false;
	int count = 0;
	if( f.open(QIODevice::WriteOnly | QIODevice::Text) )
	{
		QTextStream out(&f);

		QHashIterator<QString, Package*> i( *iAptInterface->packagesAvailable() );
		while (i.hasNext())
		{
			i.next();

			if( i.value()->markedOperation() == Package::PkgOpInstallUpgrade ) {
				out << i.value()->name() << " " << i.value()->version() << " install\n";
				count++;
			}
			if( i.value()->markedOperation() == Package::PkgOpRemove )
				qDebug() << "Warning: package is marked for removal in the wrong list";
		}

		QHashIterator<QString, Package*> r( *iAptInterface->packagesInstalled() );
		while (r.hasNext())
		{
			r.next();

			if( r.value()->markedOperation() == Package::PkgOpInstallUpgrade ) {
				out << r.value()->name() << " " << r.value()->version() << " upgrade\n";
				count++;
			}
			if( r.value()->markedOperation() == Package::PkgOpRemove ) {
				out << r.value()->name() << " " << r.value()->version() << " remove\n";
				count++;
			}
		}

		f.close();
	} else {
		fail = true;
	}

	ConfirmDialog d(false, this);
	if( fail )
		d.setText( "Error", "Failed to write package selections" );
	else
		d.setText( "Selections stored", QString("Stored %1 selections").arg(count) );
	d.exec();
}

void PackageView::on_actionRestore_all_triggered()
{
	ConfirmDialog d(true, this);
	d.setText("Confirmation","Restore all shown blacklisted packages?");
	if( !d.exec() )
		return;

	for( int i=0; i<ui->listWidget->count(); i++ )
	{
		Package* pkg = dynamic_cast<PackageListWidgetItem*>(ui->listWidget->item(i))->package();
		if( pkg && pkg->isBlacklisted() ) {
			BlacklistSelect::blackList old = pkg->blacklisted();
			pkg->setBlacklisted(BlacklistSelect::BlacklistNone);
			iAptInterface->removeFromBlacklist(pkg, old);
		}
	}
	iAptInterface->writeBlacklist();
	openWin();
}
