/*
	This file is part of Faster Application Manager.

	Faster Application Manager is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Faster Application Manager is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Faster Application Manager.  If not, see <http://www.gnu.org/licenses/>.

	(C) Heikki Holstila 2010
*/

#include <QtGui>
#include <iostream>
#include "package.h"
#include "repository.h"
#include "aaptinterface.h"
#include "apt-src/debversion.h"

Package::Package(QString name_, AAptInterface *apt_)
{
	iAptInterface = apt_;
	iName = name_;
	iIsInstalled = false;
	iMarkedForOperation = false;
	iSize = 0;
	iInstalledSize = 0;
	iMarkedOperation = PkgOpNone;
	iPkgStatus = PkgStatUnknown;
	iIcon = 0;
/*	//should be empty anyway
	iIconData = "";
	iFileNameFull = "";
	iRepoFileName = "";
	iDateString = "";
	iDescriptionShort = "";
	iDescriptionLong = "";
	iSection = "";
	iMaemoDisplayName = "";
	iVersion = "";
*/
}

Package::~Package()
{
	if( iIcon != 0 )
		delete iIcon;
}

QString Package::displayName()
{
	QString name = iName;
	if( iMaemoDisplayName!="" )
		name = iMaemoDisplayName;
	QString n( name.at(0) );
	n = n.toUpper();
	name.replace(0,1,n);
	return name;
}

void Package::setMarkedForOperation(operation op_)
{
	if( iMarkedOperation != op_ ) {
		if( op_==PkgOpNone ) {
			if( iAptInterface )
				iAptInterface->setNumSelectedPackages( iAptInterface->numSelectedPackages()-1 );
		} else if( iMarkedOperation==PkgOpNone ) {
			if( iAptInterface )
				iAptInterface->setNumSelectedPackages( iAptInterface->numSelectedPackages()+1 );
		}
	}

	iMarkedOperation = op_;

	if( op_ == PkgOpNone )
		iMarkedForOperation = false;
	else
		iMarkedForOperation = true;
}

void Package::convertIcon()
{
	if( iIconData.length() > 0 && iIcon == 0 ) {
		iIcon = new QPixmap();
		iIconData = QByteArray::fromBase64( iIconData );
		if( !iIcon->loadFromData( iIconData ) ) {
			qDebug() << "Warning: Package" << iName << "has invalid icon data";
		}
	}

}

bool Package::isUpgradeable()
{
	if( iIsInstalled )
	{
		QString newer = upgradeableVersion();
		//qDebug() << newer << iVersion << versionCompare(newer,iVersion);
		return versionCompare(newer,iVersion);
	}

	return false;
}

QString Package::upgradeableVersion()
{
	QString ver="";

	if( isInstalled() ) {
		Package* newer = iAptInterface->packagesAvailable()->value(iName,0);
		if( newer )
			ver = newer->version();
	}
	return ver;
}

Package* Package::availablePackage()
{
	if( !isUpgradeable() )
		return 0;

	Package* newer = iAptInterface->packagesAvailable()->value(iName,0);

	return newer;
}

void Package::updateStatus()
{
	if( iIsInstalled ) {
		iPkgStatus = PkgStatInstalled;
		if( isUpgradeable() )
			iPkgStatus = PkgStatUpgradeable;
	} else {
		iPkgStatus = PkgStatNotInstalled;
	}
}

Package::packageStatus Package::status()
{
	updateStatus(); // not optimal, but it's here just in case
	return iPkgStatus;
}

bool Package::hasIconData()
{
	if( iIconData.length()>0 )
		return true;
	else
		return false;
}

bool Package::versionCompare(QString isNewer, QString compare)
{
	//int res=0;
	int res = debVS.CmpVersion(isNewer.toStdString(), compare.toStdString());
	//qDebug() << isNewer << compare << res;

	if( res > 0 )
		return true;

	return false;
}

/*
// does not work right
bool Package::versionCompare(QString isNewer, QString compare)
{
	//qDebug() << isNewer << compare;

	if( isNewer == compare )
		return false;
	if( isNewer == "" && compare == "" )
		return false;
	if( isNewer == "" )
		return false;
	if( compare == "" )
		return true;

	QList<int> parts1base;
	QList<int> parts1extra;
	QList<int> parts2base;
	QList<int> parts2extra;

	QString addstr = "";
	bool extra = false;
	for(int i=0; i<isNewer.length(); i++)
	{
		QChar c = isNewer.at(i);
		if( c.isDigit() )
			addstr += c;
		else {
			if( addstr.length()>0 ) {
				if( !extra )
					parts1base << addstr.toInt();
				else
					parts1extra << addstr.toInt();
			}
			if( c != '.' )
				extra = true;
			addstr = "";
		}
	}
	if( addstr.length()>0 ) {
		if( !extra )
			parts1base << addstr.toInt();
		else
			parts1extra << addstr.toInt();
	}

	addstr = "";
	extra = false;
	for(int i=0; i<compare.length(); i++)
	{
		QChar c = compare.at(i);
		if( c.isDigit() )
			addstr += c;
		else {
			if( addstr.length()>0 ) {
				if( !extra )
					parts2base << addstr.toInt();
				else
					parts2extra << addstr.toInt();
			}
			if( c != '.' )
				extra = true;
			addstr = "";
		}
	}
	if( addstr.length()>0 ) {
		if( !extra )
			parts2base << addstr.toInt();
		else
			parts2extra << addstr.toInt();
	}

	//qDebug() << parts1base << parts1extra << parts2base << parts2extra;

	if( parts1base.count()>0 || parts2base.count()>0 )
	{
		int c = parts1base.count();
		if( parts2base.count() < c )
			c = parts2base.count();

		for(int i=0; i<c; i++)
		{
			if( parts1base.at(i) > parts2base.at(i) )
				return true;
			else if( parts1base.at(i) < parts2base.at(i) )
				return false;
		}
		if( parts1base.count() > parts2base.count() )
			return true;
		if( parts1base.count() < parts2base.count() )
			return false;
	}

	if( parts1extra.count()>0 || parts2extra.count()>0 )
	{
		int c = parts1extra.count();
		if( parts2extra.count() < c )
			c = parts2extra.count();

		for(int i=0; i<c; i++)
		{
			if( parts1extra.at(i) > parts2extra.at(i) )
				return true;
			else if( parts1extra.at(i) < parts2extra.at(i) )
				return false;
		}
		if( parts1extra.count() > parts2extra.count() )
			return true;
		if( parts1extra.count() < parts2extra.count() )
			return false;
	}

	return false;
}
*/
