/*
	This file is part of Faster Application Manager.

	Faster Application Manager is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Faster Application Manager is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Faster Application Manager.  If not, see <http://www.gnu.org/licenses/>.

	(C) Heikki Holstila 2010
*/

#include <QtCore>
#include <QtGui>
#include <iostream>

#ifdef Q_WS_MAEMO_5
#include <QtMaemo5>
#endif

#ifdef MYDEF_GTK_EXISTS
#include <gdk/gdk.h>
#endif

#include "mainwindow.h"
#include "version.h"
#include "ui_mainwindow.h"
#include "aaptinterface.h"
#include "packageview.h"
#include "confirmdialog.h"
#include "dimmer.h"
#include "repoview.h"
#include "help.h"
#include "settings.h"
#include "logview.h"


MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    ui->setupUi(this);

	iAptInterface = new AAptInterface(this);
	iWinPackageView = new PackageView(this);
	iWinPackageView->setAptInterface(iAptInterface);
	iWinRepoView = new RepoView(this);
	iWinRepoView->setAptInterface(iAptInterface);
	iSettings = new Settings(this);
	iAptInterface->setSettings(iSettings);

#ifdef Q_WS_MAEMO_5
	this->setAttribute(Qt::WA_Maemo5StackedWindow);
	if( !iSettings->qsettings()->value("disable_autorotation",false).toBool() ) {
		this->setAttribute(Qt::WA_Maemo5AutoOrientation);
	} else {
		this->setAttribute(Qt::WA_Maemo5LandscapeOrientation);
	}
#endif

	//iOperation = OpNone;
	iDimmer = new dimmer(this);

	//iAptInterface->setUpdateDimmer( iWinPackageView->dim() );

	iReposAutoUpdating = false;
	iUpgradeAutoUpdate = true;
	iNextOperation = OpNone;

	// clear log on startup
	QFile logfile(KLogFileName);
	logfile.remove();
}

MainWindow::~MainWindow()
{
	// save "need repo refresh" status
	iSettings->qsettings()->setValue("need_repo_refresh", iAptInterface->needRepoRefresh());

	delete iWinPackageView;
	delete iWinRepoView;
	delete iAptInterface;
	delete iDimmer;
	delete iSettings;
    delete ui;
}

void MainWindow::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void MainWindow::on_btnRepos_clicked()
{
	iWinRepoView->openWin();
}

void MainWindow::on_btnUpdate_clicked()
{	
	// update catalogs

	busyDialog(true, tr("Operation in progress"), tr("Updating catalogs"));
	iAptInterface->addQueuedOperation(AAptInterface::ModeAptGetUpdate);
	//iAptInterface->addQueuedOperation(AAptInterface::ModeFetchDates);	// experimental
	iAptInterface->run(iDimmer);
}

void MainWindow::on_btnListInstalled_clicked()
{
	//uninstall

	iWinPackageView->setStatFilter( Package::PkgStatInstalled );

	// don't require this for package removal, even though other operations can be performed
	// through the same view
	//if( iAptInterface->needRepoRefresh() )
	//	iAptInterface->addQueuedOperation(AAptInterface::ModeAptGetUpdate);

	busyDialog(true, tr("Operation in progress"), tr("Reading package lists"));

	iNextOperation = OpOpenPkgView;
	iAptInterface->addQueuedOperation(AAptInterface::ModeReadPackages);
	iAptInterface->run(iDimmer);
}

void MainWindow::on_btnListInstallable_clicked()
{
	//install

	iWinPackageView->setStatFilter( Package::PkgStatNotInstalled );

	if( iAptInterface->needRepoRefresh() )
		iAptInterface->addQueuedOperation(AAptInterface::ModeAptGetUpdate);

	busyDialog(true, tr("Operation in progress"), tr("Reading package lists"));

	iNextOperation = OpOpenPkgView;
	iAptInterface->addQueuedOperation(AAptInterface::ModeReadPackages);
	iAptInterface->run(iDimmer);
}

void MainWindow::on_btnUpgrade_clicked()
{
	// upgrade

	iWinPackageView->setStatFilter( Package::PkgStatUpgradeable );

	if( iAptInterface->needRepoRefresh() )
		iAptInterface->addQueuedOperation(AAptInterface::ModeAptGetUpdate);

	busyDialog(true, tr("Operation in progress"), tr("Reading package lists"));

	iNextOperation = OpOpenPkgView;
	iAptInterface->addQueuedOperation(AAptInterface::ModeReadPackages);
	iAptInterface->run(iDimmer);
}

void MainWindow::operationQueueFinished(QList<AAptInterface::interfaceMode> lastModes, bool success, QString title, QStringList msgs)
{
	if( lastModes.contains(AAptInterface::ModeAptGetSimulate) && success ) {
		iNextOperation = OpPromptSimulated;
	}
	if( lastModes.contains(AAptInterface::ModeAptGetInstall) && success ) {
		iWinPackageView->clearSelections();
		busyDialog(false);
		iWinPackageView->close();
		iNextOperation = OpNone;

		GdkEventIconThemeReload();
	}

	if( iNextOperation == OpOpenPkgView && success )
	{
		iWinPackageView->openWin();
		busyDialog(false);
		iNextOperation = OpNone;
	} else if( iNextOperation == OpPromptSimulated && success )
	{
		QStringList inst,remv,instver,remvver;
		QStringList all = iAptInterface->processPackages();
		QStringList vers = iAptInterface->processPackageVersions();
		for(int zyx=0; zyx<all.count(); zyx++)
		{
			if( all.at(zyx).endsWith('-') )
			{
				remv.append( all.at(zyx).left( all.at(zyx).size()-1 ) );
				if( vers.count()>zyx )
					remvver.append( vers.at(zyx) );
			} else {
				inst.append( all.at(zyx) );
				if( vers.count()>zyx )
					instver.append( vers.at(zyx) );
			}
		}

		QString pkglist;
		pkglist = QString("<b>The following operations will be performed:</b><br>" \
				  "%1 to install/upgrade, %2 to remove<br>").arg(inst.count()).arg(remv.count());
		bool mismatch = false;
		int total_inst_size = 0;

		if( remv.count()>0 ) {
			pkglist += "<br><b><u>REMOVE:</u></b><br><font size=\"-1\">";
			for( int i=0; i<remv.count(); i++ ) {
				pkglist += "<b>" + remv.at(i) + "</b>";
				Package* pkg = iAptInterface->packagesInstalled()->value(remv.at(i),0);
				if( !pkg ) {
					qDebug() << "Warning: unknown package" << remv.at(i);
					pkglist += "<font color=\"red\">***UNKNOWN***</font>";
					mismatch = true;
				}
				if( remvver.count()>i ) {
					pkglist += " " + remvver.at(i);
					if( pkg && remvver.at(i) != pkg->version() ) {
						qDebug() << "Version mismatch, database version is" << pkg->version() << "but installing" << remvver.at(i);
						mismatch = true;
						pkglist += " <font color=\"red\">***TRYING TO INSTALL " + pkg->version() + "***</font> ";
					}
				}
				if( pkg && pkg->installedSize()>0 )
					pkglist += QString(" (%L1 kB)").arg(pkg->installedSize());
				pkglist += "<br>";
			}
		}
		pkglist += "</font>";

		if( inst.count()>0 ) {
			pkglist += "<br><b><u>INSTALL/UPGRADE:</u></b><br><font size=\"-1\">";
			for( int i=0; i<inst.count(); i++ ) {
				pkglist += "<b>" + inst.at(i) + "</b>";
				Package* pkg = iAptInterface->packagesAvailable()->value(inst.at(i),0);
				if( !pkg ) {
					qDebug() << "Warning: unknown package" << inst.at(i);
					pkglist += "<font color=\"red\">***NEW/UNKNOWN***</font>";
					mismatch = true;
				}
				if( instver.count()>i ) {
					pkglist += " " + instver.at(i);
					if( pkg && instver.at(i) != pkg->version() ) {
						qDebug() << "Version mismatch, database version is" << pkg->version() << "but installing" << instver.at(i);
						mismatch = true;
						pkglist += " <font color=\"red\">***TRYING TO INSTALL " + pkg->version() + "***</font> ";
					}
				}
				if( pkg && pkg->size()>0 ) {
					pkglist += QString(" (%L1 kB)").arg(pkg->size()/1024);
					total_inst_size += pkg->size()/1024;
				}
				pkglist += "<br>";
			}
		}
		pkglist += "</font>";

		if( mismatch ) {
			ConfirmDialog m(false, this);
			m.setText("Warning", "There is a version mismatch between your original package selections and some of the packages being installed " \
					  "from the repositories. This could be due to your application catalogs being out of date.");
			m.exec();
		}

		busyDialog(false);
		ConfirmDialog d(true, this);
		d.setText("Confirmation",pkglist);
		if( d.exec() ) {
			iAptInterface->addQueuedOperation(AAptInterface::ModeAptGetInstall);

			if( iSettings->qsettings()->value("enable_autoclean",false).toBool() && inst.count()>0 )
				iAptInterface->addQueuedOperation(AAptInterface::ModeAptGetClean);

			QString busytext;
			if( remv.count() > 0 )
				busytext += QString("Remove %1 package(s)<br>").arg(remv.count());
			if( inst.count() > 0 )
				busytext += QString("Install %1 package(s)<br>").arg(inst.count());
			busytext += "Preparing...";
			busyDialog(true, "Operation in progress", busytext);

			// "run" really does nothing here since the previous item should still be running
			if( iWinPackageView->isVisible() ) {
				iAptInterface->run(iWinPackageView->mydimmer());
			} else {
				iAptInterface->run(iDimmer);
			}
		}
		iNextOperation = OpNone;
		return;
	} else {
		busyDialog(false);
		iNextOperation = OpNone;
		ConfirmDialog d(false, this);
		QString text = msgs.join("<br>");
		d.setText(title, text);
		d.exec();
	}

}

void MainWindow::busyDialog(bool show_, QString title, QString text)
{
	if( show_ ) {
		iDimmer->setProgress(-1);
		ui->menuMenu->setEnabled(false);
		ui->centralWidget->setEnabled(false);
		iWinPackageView->disableMenu();
		iDimmer->resizeEvent(0);
		iDimmer->dim(title, text);
		iWinPackageView->mydimmer()->resizeEvent(0);
		iWinPackageView->mydimmer()->dim(title, text);
	} else {
		iDimmer->undim();
		iWinPackageView->mydimmer()->undim();
		ui->menuMenu->setEnabled(true);
		ui->centralWidget->setEnabled(true);
		iWinPackageView->enableMenu();
	}
}

void MainWindow::on_actionAbout_triggered()
{
	ConfirmDialog d(false, this);
	d.setText("About","Faster Application Manager<br>"
			  "<font size=\"-1\">Version " + PROGRAM_VERSION + "</font><br><br>"
			  "(C) Heikki Holstila 2010");
	d.exec();
}

void MainWindow::on_actionClean_triggered()
{
	//if( iOperation != OpNone ) return;
	//iOperation = OpClean;
	iAptInterface->addQueuedOperation(AAptInterface::ModeAptGetClean);
	iAptInterface->run(iDimmer);
}

void MainWindow::closeEvent(QCloseEvent *event)
{
	if( iDimmer->busy() ) {
		iAptInterface->cancel();
		event->ignore();
	} else {
		event->accept();
	}
}

void MainWindow::on_actionView_log_triggered()
{
	QByteArray log = iAptInterface->readLogFile();
	LogView l(log, this);
	l.exec();
}

void MainWindow::on_actionOptions_triggered()
{
	iSettings->openWin();
}

void MainWindow::GdkEventIconThemeReload()
{
	// This feature does not work when compiled on the Qt SDK in Ubuntu. (?)
	// Only works when compiled in scratchbox or by the maemo.org autobuilder. (?)

#ifdef MYDEF_GTK_EXISTS
	//std::cout << "Sending GDK client icon theme reload event" << std::endl;

	GdkEventClient ev;
	ev.type = GDK_CLIENT_EVENT;
	ev.window = NULL;
	ev.send_event = TRUE;
	ev.message_type = gdk_atom_intern_static_string("_GTK_LOAD_ICONTHEMES");
	ev.data_format = 32;
	gdk_event_send_clientmessage_toall((GdkEvent*)&ev);
#endif
}
