/*
	This file is part of Faster Application Manager.

	Faster Application Manager is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Faster Application Manager is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Faster Application Manager.  If not, see <http://www.gnu.org/licenses/>.

	(C) Heikki Holstila 2010
*/

#include <QtCore>
#include <QtGui>
#include <iostream>

#ifdef Q_WS_MAEMO_5
#include <QtMaemo5>
#endif

#include "mainwindow.h"
#include "version.h"
#include "ui_mainwindow.h"
#include "aptinterface.h"
#include "packageview.h"
#include "confirmdialog.h"
#include "dimmer.h"
#include "repoview.h"

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    ui->setupUi(this);
#ifdef Q_WS_MAEMO_5
	this->setAttribute(Qt::WA_Maemo5StackedWindow);
	this->setAttribute(Qt::WA_Maemo5AutoOrientation);
#endif

	iAptInterface = new AptInterface(this);
	iWinPackageView = new PackageView(this);
	iWinPackageView->setAptInterface(iAptInterface);
	iAptInterface->setPackageView(iWinPackageView);
	iWinRepoView = new RepoView(this);
	iWinRepoView->setAptInterface(iAptInterface);

	iOperation = OpNone;
	iDimmer = new dimmer(this);

	iReposAutoUpdating = false;
}

MainWindow::~MainWindow()
{
	delete iWinPackageView;
	delete iWinRepoView;
	delete iAptInterface;
    delete ui;
}

void MainWindow::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void MainWindow::on_btnRepos_clicked()
{
	iWinRepoView->openWin();
}

void MainWindow::on_btnUpdate_clicked()
{
	busyDialog(true, tr("Operation in progress"), tr("Updating catalogs"));
	if( !iAptInterface->updateCatalogs() )
	{
		ConfirmDialog d(false, this);
		d.setText("Error","<b>Unable to perform operation</b><br>Can't write repository lists");
		d.exec();
	}
}

void MainWindow::on_btnListInstalled_clicked()
{
	if( iOperation != OpNone ) return;
	iOperation = OpUninstall;

	busyDialog(true, tr("Operation in progress"), tr("Reading package lists"));
	if( !iAptInterface->updatePackageList() )
	{
		ConfirmDialog d(false, this);
		d.setText("Error","<b>Unable to perform operation</b><br>Can't write repository lists");
		d.exec();
	}
}

void MainWindow::on_btnListInstallable_clicked()
{
	if( iOperation != OpNone ) return;
	iOperation = OpInstall;

	if( iAptInterface->needRepoRefresh() ) {
		iReposAutoUpdating = true;
		on_btnUpdate_clicked();
		return;
	}

	busyDialog(true, tr("Operation in progress"), tr("Reading package lists"));
	if( !iAptInterface->updatePackageList() )
	{
		ConfirmDialog d(false, this);
		d.setText("Error","<b>Unable to perform operation</b><br>Can't write repository lists");
		d.exec();
	}
}

void MainWindow::on_btnUpgrade_clicked()
{
	if( iOperation != OpNone ) return;
	iOperation = OpUpgrade;

	if( iAptInterface->needRepoRefresh() ) {
		iReposAutoUpdating = true;
		on_btnUpdate_clicked();
		return;
	}

	busyDialog(true, tr("Operation in progress"), tr("Reading package lists"));
	if( !iAptInterface->updatePackageList() )
	{
		ConfirmDialog d(false, this);
		d.setText("Error","<b>Unable to perform operation</b><br>Can't write repository lists");
		d.exec();
	}
}

void MainWindow::operationFinished(AptInterface::aptMode lastMode_, bool success_, QString msg_)
{
	if( iOperation == OpInstall || iOperation == OpUninstall || iOperation == OpUpgrade )
	{
		if( lastMode_ == AptInterface::ModeReadInfo && success_ )
		{
			iWinPackageView->openWin(iOperation);
			busyDialog(false);
			return;
		}
		if( (lastMode_ == AptInterface::ModeAptGetSimulateInstall || lastMode_ == AptInterface::ModeAptGetSimulateRemove )
			&& success_ )
		{
			QStringList inst,remv;
			iAptInterface->getSimulatedPackageLists(inst,remv);

			QString pkglist;
			pkglist = QString("<b>The following operations will be performed:</b><br>" \
					  "%0 to install/upgrade, %1 to remove<br>").arg(inst.count()).arg(remv.count());

			if( inst.count()>0 ) {
				pkglist += "<br><b><u>INSTALL/UPDATE:</u></b><br>";
				for( int i=0; i<inst.count(); i++ )
					pkglist += inst.at(i) + "<br>";
			}

			if( remv.count()>0 ) {
				pkglist += "<br><b><u>REMOVE:</u></b><br>";
				for( int i=0; i<remv.count(); i++ )
					pkglist += remv.at(i) + "<br>";
			}

			busyDialog(false);

			ConfirmDialog d(true, this);
			d.setText("Confirmation",pkglist);
			if( d.exec() ) {
				//std::cout << "*** Proceeding with install/remove operation" << std::endl;
				QStringList names;
				iAptInterface->getOriginalPackageList(names);
				if( lastMode_ == AptInterface::ModeAptGetSimulateInstall )
					iAptInterface->installPackages(names);
				if( lastMode_ == AptInterface::ModeAptGetSimulateRemove )
					iAptInterface->removePackages(names);
			} else {
				//std::cout << "*** Installation/removal cancelled" << std::endl;
			}

			return;
		}

		if( (lastMode_ == AptInterface::ModeAptGetInstall || lastMode_ == AptInterface::ModeAptGetRemove )
			&& success_ )
		{
			iOperation = OpNone;
			busyDialog(false);
			iWinPackageView->hide();
			ConfirmDialog d(false, this);
			if( lastMode_ == AptInterface::ModeAptGetInstall )
				d.setText("Operation finished","Package(s) successfully installed");
			else if( lastMode_ == AptInterface::ModeAptGetRemove )
				d.setText("Operation finished","Package(s) successfully removed");
			d.exec();
			return;
		}

	}

	if( lastMode_ == AptInterface::ModeAptGetClean && success_ )
	{
		iOperation = OpNone;
		busyDialog(false);
		ConfirmDialog d(false, this);
		d.setText("Operation finished","Package cache cleaned");
		d.exec();
		return;
	}

	if( lastMode_ == AptInterface::ModeAptGetUpdate && success_ && iReposAutoUpdating
		&& ( iOperation==OpInstall || iOperation==OpUpgrade ) )
	{
		iReposAutoUpdating = false;
		if( iOperation==OpInstall ) {
			iOperation = OpNone;
			on_btnListInstallable_clicked();
			return;
		}
		if( iOperation==OpUpgrade ) {
			iOperation = OpNone;
			on_btnUpgrade_clicked();
			return;
		}
	} else if( lastMode_ == AptInterface::ModeAptGetUpdate ) {
		iOperation = OpNone;
		busyDialog(false);
		ConfirmDialog d(false, this);
		d.setText("Operation finished","Repositories/catalogs updated");
		d.exec();
		return;
	}

	busyDialog(false);
	if( !success_ ) {
		ConfirmDialog d(false, this);
		d.setText("Error",msg_);
		d.exec();
	}
	iOperation = OpNone;
	//std::cout << "finished" << std::endl;
}

void MainWindow::busyDialog(bool visible_, QString title, QString text)
{
	if( visible_ ) {
		ui->menuMenu->setEnabled(false);
		iWinPackageView->disableMenu();
		iDimmer->resizeEvent(0);
		iDimmer->dim(title, text);
		iWinPackageView->dim()->resizeEvent(0);
		iWinPackageView->dim()->dim(title, text);
	} else {
		iDimmer->undim();
		iWinPackageView->dim()->undim();
		ui->menuMenu->setEnabled(true);
		iWinPackageView->enableMenu();
	}
}

void MainWindow::on_actionAbout_triggered()
{
	ConfirmDialog d(false, this);
	d.setText("About","Faster Application Manager<br>"
			  "<font size=\"-1\">Version " + PROGRAM_VERSION + "</font><br><br>"
			  "(C) Heikki Holstila 2010");
	d.exec();
}

void MainWindow::on_actionClean_triggered()
{
	if( iOperation != OpNone ) return;
	iOperation = OpClean;
	iAptInterface->aptGetClean();
}

void MainWindow::closeEvent(QCloseEvent *event)
{
	if( iDimmer->busy() ) {
		event->ignore();
	} else {
		event->accept();
	}
}
