/*
	This file is part of Faster Application Manager.

	Faster Application Manager is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Faster Application Manager is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Faster Application Manager.  If not, see <http://www.gnu.org/licenses/>.

	(C) Heikki Holstila 2010
*/

#include <QtCore>
#include <iostream>
#include "aptinterface.h"
#include "mainwindow.h"
#include "packageview.h"
#include "package.h"

AptInterface::AptInterface(MainWindow* w_)
{
	iMainWindow = w_;
	iRunProc = new QProcess();
	connect(iRunProc,SIGNAL(error(QProcess::ProcessError)),this,SLOT(procError(QProcess::ProcessError)));
	connect(iRunProc,SIGNAL(finished(int,QProcess::ExitStatus)),this,SLOT(procFinished(int,QProcess::ExitStatus)));

	iMode = ModeNone;
	iReady = false;
	iError = tr("Undefined error");
}

AptInterface::~AptInterface()
{
	delete iRunProc;
}

void AptInterface::updateCatalogs()
{
	iMode = ModeAptGetUpdate;

	QString runBinary = "/usr/bin/apt-get";
	QStringList runParameters;
	runParameters << "update";

	iRunProc->start(runBinary,runParameters);
}

void AptInterface::updatePackageList()
{
	if( iMode != ModeNone ) return;

	toInstall.clear();
	toRemove.clear();
	pkgNamesList.clear();

	QDir logdir(KlogFileDir);
	logdir.mkpath(KlogFileDir);
	QFile logfile(KlogFileName);
	logfile.remove();

	QHashIterator<QString, Package*> i( iPackages );
	while (i.hasNext()) {
		i.next();
		if( i.value()!=0 ) delete i.value();
	}
	iPackages.clear();
	iSelectedCount = 0;

	iMode = ModeReadInfo;
	QString runBinary = "/usr/bin/apt-cache";
	QStringList runParameters;
	runParameters << "dumpavail";

	iRunProc->start(runBinary,runParameters);

}

bool AptInterface::isInterfaceReady()
{
	if( iMode==ModeNone && iReady) {
		return true;
	}
	return false;
}

void AptInterface::procError(QProcess::ProcessError error)
{
	QByteArray output = iRunProc->readAllStandardOutput();
	logToFile(output);

	iError.setNum(error);
	iError.prepend(tr("QProcess error "));
	iMainWindow->operationFinished(iMode, false, iError);
	iMode = ModeNone;
}

void AptInterface::procFinished(int exitCode, QProcess::ExitStatus exitStatus)
{
	aptMode lastMode = iMode;
	iMode = ModeNone;
	if( exitCode != 0 ) {
		QByteArray output = iRunProc->readAllStandardOutput();
		logToFile(output);

		iError.setNum(exitCode);
		iError.prepend(tr("Exit code "));
		QString s;
		s.setNum(exitStatus);
		iError.append(", status " + s);
		iMainWindow->operationFinished(lastMode, false, iError);
	} else {

		if( lastMode == ModeAptGetUpdate )
		{
			iRunProc->close();
			iMainWindow->operationFinished(lastMode, true, "");
		}

		if( lastMode == ModeReadInfo )
		{
			if( !parseAptCacheOutput() ) {
				iMainWindow->operationFinished(lastMode, false, iError);
				return;
			}
			iRunProc->close();
			if( !readDpkgDatabase() ) {
				iRunProc->close();
				iMainWindow->operationFinished(lastMode, false, iError);
				return;
			}
			iRunProc->close();
			iReady = true;
			iMainWindow->operationFinished(lastMode, true, "");
		}

		if( lastMode == ModeAptGetSimulateInstall || lastMode == ModeAptGetSimulateRemove )
		{
			if( !parseSimulated() ) {
				iRunProc->close();
				iMainWindow->operationFinished(lastMode, false, iError);
				return;
			}
			iRunProc->close();
			iMainWindow->operationFinished(lastMode, true, "");
		}

		if( lastMode == ModeAptGetInstall || lastMode == ModeAptGetRemove )
		{
			QByteArray output = iRunProc->readAllStandardOutput();
			logToFile(output);

			iRunProc->close();
			iMainWindow->operationFinished(lastMode, true, "");
		}

		if( lastMode == ModeAptGetClean )
		{
			iRunProc->close();
			iMainWindow->operationFinished(lastMode, true, "");
		}
	}
}

bool AptInterface::parseAptCacheOutput()
{
	QList<QByteArray> lines = iRunProc->readAllStandardOutput().split('\n');

	Package* pkg = 0;
	for( int i=0; i<lines.count(); i++)
	{
		if( lines.at(i).startsWith("Package: ") )
		{
			QString pkgname = lines.at(i).mid(9).trimmed();
			if( pkgname==0 || pkgname == "" ) {
				iError = tr("Empty package name");
				std::cout << "Encountered empty package name!" << std::endl;
				return false;
			}
			pkg = new Package(pkgname);
			iPackages.insert(pkgname,pkg);
		}
		if( lines.at(i).startsWith("Section: ") && pkg != 0 )
		{
			QString sect = lines.at(i).mid(9).trimmed();
			pkg->setSection(sect);
		}
		if( lines.at(i).startsWith("Version: ") && pkg != 0 )
		{
			QString ver = lines.at(i).mid(9).trimmed();
			pkg->setVersionAvailable(ver);
		}
		if( lines.at(i).startsWith("Maemo-Display-Name: ") && pkg != 0 )
		{
			QString mdn = lines.at(i).mid(20).trimmed();
			pkg->setMaemoDisplayName(mdn);
		}
		if( lines.at(i).startsWith("Description: ") && pkg != 0 )
		{
			QString ds = lines.at(i).mid(13).trimmed();
			pkg->setDescShort(ds);
		}
	}
	std::cout << "apt-cache: read " << iPackages.count() << " available packages" << std::endl;
	return true;
}

bool AptInterface::readDpkgDatabase()
{
	// available at /var/lib/dpkg/status
	QFile db("/var/lib/dpkg/status");
	if (!db.open(QIODevice::ReadOnly | QIODevice::Text)) {
		iError = tr("Unable to read dpkg database");
		return false;
	}

	Package* pkg = 0;
	int newpkg=0;
	int totpkg=0;

	while (!db.atEnd()) {
		QByteArray line = db.readLine();

		if( line.startsWith("Package: ") )
		{
			QString pkgname = line.mid(9).trimmed();
			if( pkgname==0 || pkgname == "" ) {
				iError = tr("Empty package name");
				std::cout << "Encountered empty package name!" << std::endl;
				return false;
			}
			pkg = iPackages.value(pkgname,0);
			if( pkg==0 ) {
				pkg = new Package(pkgname);
				iPackages.insert(pkgname,pkg);
				pkg->setLocal(true);
				newpkg++;
			}
			totpkg++;
		}
		if( line.startsWith("Status: ") && pkg != 0 )
		{
			QString stat = line.mid(8).trimmed();
			if( stat == "install ok installed" )
				pkg->setInstalled(true);
		}
		if( line.startsWith("Section: ") && pkg != 0 && pkg->section()=="" )
		{
			QString sect = line.mid(9).trimmed();
			pkg->setSection(sect);
		}
		if( line.startsWith("Version: ") && pkg != 0 )
		{
			QString ver = line.mid(9).trimmed();
			pkg->setVersionInstalled(ver);
		}
		if( line.startsWith("Description: ") && pkg != 0 && pkg->descShort()=="" )
		{
			QString ds = line.mid(13).trimmed();
			pkg->setDescShort(ds);
		}

	}

	db.close();

	std::cout << "Read dpkg database info for " << totpkg << " packages (" << newpkg << " new)" << std::endl;

	return true;
}

bool AptInterface::parseSimulated()
{
	QByteArray output = iRunProc->readAllStandardOutput();
	logToFile(output);
	QList<QByteArray> lines = output.split('\n');

	toInstall.clear();
	toRemove.clear();

	for( int i=0; i<lines.count(); i++)
	{
		QString s = lines.at(i);
		if( s.startsWith("Inst ") )
		{
			toInstall <<  s.section(' ',1,1);
		}
		if( s.startsWith("Remv ") )
		{
			toRemove << s.section(' ',1,1);
		}
	}

	std::cout << "Simulated results:" << std::endl;
	std::cout << "  Install: ";
	for(int x=0; x<toInstall.count(); x++)
		std::cout << toInstall.at(x).toStdString() << " ";
	std::cout << std::endl << "  Remove: ";
	for(int x=0; x<toRemove.count(); x++)
		std::cout << toRemove.at(x).toStdString() << " ";
	std::cout << std::endl;

	if( toInstall.count()==0 && toRemove.count()==0 )
		return false;

	return true;
}

void AptInterface::simulateInstall(QStringList names_)
{
	pkgNamesList = names_;
	iMainWindow->busyDialog(true, "Operation in progress", "Reading dependencies");
	std::cout << "SIMULATE apt-get install: ";
	for( int i=0; i<names_.count(); i++)
	{
		std::cout << names_.at(i).toStdString() + " ";
	}
	std::cout << std::endl;

	iMode = ModeAptGetSimulateInstall;

	QString runBinary = "/usr/bin/apt-get";
	QStringList runParameters;
	runParameters << "-qsy" << "install";
	runParameters << names_;

	iRunProc->start(runBinary,runParameters);
}

void AptInterface::simulateRemove(QStringList names_)
{
	pkgNamesList = names_;
	iMainWindow->busyDialog(true, "Operation in progress", "Reading dependencies");
	std::cout << "SIMULATE apt-get remove: ";
	for( int i=0; i<names_.count(); i++)
	{
		std::cout << names_.at(i).toStdString() + " ";
	}
	std::cout << std::endl;

	iMode = ModeAptGetSimulateRemove;

	QString runBinary = "/usr/bin/apt-get";
	QStringList runParameters;
	runParameters << "-qsy" << "remove";
	runParameters << names_;

	iRunProc->start(runBinary,runParameters);
}

void AptInterface::installPackages(QStringList names_)
{
	iMainWindow->busyDialog(true, "Operation in progress", "Downloading and installing packages");
	std::cout << "apt-get install ";
	for( int i=0; i<names_.count(); i++)
	{
		std::cout << names_.at(i).toStdString() + " ";
	}
	std::cout << std::endl;

	iMode = ModeAptGetInstall;

	QString runBinary = "/usr/bin/apt-get";
	QStringList runParameters;
	runParameters << "-qy" << "install";
	runParameters << names_;

	iRunProc->start(runBinary,runParameters);
}

void AptInterface::removePackages(QStringList names_)
{
	iMainWindow->busyDialog(true, "Operation in progress", "Removing packages");
	std::cout << "apt-get remove ";
	for( int i=0; i<names_.count(); i++)
	{
		std::cout << names_.at(i).toStdString() + " ";
	}
	std::cout << std::endl;

	iMode = ModeAptGetRemove;

	QString runBinary = "/usr/bin/apt-get";
	QStringList runParameters;
	runParameters << "-qy" << "remove";
	runParameters << names_;

	iRunProc->start(runBinary,runParameters);
}

void AptInterface::aptGetClean()
{
	iMainWindow->busyDialog(true, "Operation in progress", "Cleaning up");
	std::cout << "apt-get clean" << std::endl;

	iMode = ModeAptGetClean;

	QString runBinary = "/usr/bin/apt-get";
	QStringList runParameters;
	runParameters << "clean";

	iRunProc->start(runBinary,runParameters);
}

void AptInterface::logToFile( QByteArray data )
{
	QFile f(KlogFileName);

	if( f.open( QIODevice::Append | QIODevice::WriteOnly | QIODevice::Text ) )
	{
		QTextStream out(&f);
		out << data;
		f.close();
	}
}
