/*
	This file is part of Faster Application Manager.

	Faster Application Manager is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Faster Application Manager is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Faster Application Manager.  If not, see <http://www.gnu.org/licenses/>.

	(C) Heikki Holstila 2010
*/

#include <iostream>
#include "packageview.h"
#include "ui_packageview.h"
#include "package.h"
#include "filterselect.h"
#include "confirmdialog.h"
#include "dimmer.h"

PackageListWidgetItem::PackageListWidgetItem(Package* p_, QString name_) : QListWidgetItem(name_)
{
	iPackage = p_;
}


void ListItemDelegate::paint(QPainter *painter, const QStyleOptionViewItem &option, const QModelIndex &index) const
{
	QString name = index.data(UserRoleName).toString();
	QString verInst = index.data(UserRoleVerInst).toString();
	QString verAvail = index.data(UserRoleVerAvail).toString();
	QString desc = index.data(UserRoleDescShort).toString();
	bool marked = index.data(UserRoleMarked).toBool();
	int oper = index.data(UserRoleOperation).toInt();

	QString ver = verAvail;
	if( oper==MainWindow::OpUninstall ) {
		ver = verInst;
	}

	painter->save();
	QRect r = option.rect;

	if( marked )
		painter->fillRect(r, option.palette.color(QPalette::Highlight));
	else
		painter->fillRect(r, option.palette.color(QPalette::Background));

	painter->drawLine(QPoint(r.left(),r.top()), QPoint(r.right(),r.top()));
	painter->drawLine(QPoint(r.left(),r.bottom()), QPoint(r.right(),r.bottom()));

	painter->drawText(r, Qt::AlignTop|Qt::AlignLeft, name);
	painter->drawText(r, Qt::AlignTop|Qt::AlignRight, ver);

	QFont f = painter->font();
	f.setPointSize( f.pointSize()-4 );
	painter->setFont( f );
	r = option.rect.adjusted(0,25,0,0);
	painter->drawText(r.left(), r.top(), r.width(), r.height(), Qt::AlignBottom|Qt::AlignLeft, desc, &r);

	painter->restore();
}

QSize ListItemDelegate::sizeHint(const QStyleOptionViewItem&, const QModelIndex&) const
{
	return QSize(500, 60);
}


PackageView::PackageView(QWidget *parent) : QMainWindow(parent), ui(new Ui::PackageView)
{
	iMainWindow = static_cast<MainWindow*>(parent);
	ui->setupUi(this);
#ifdef Q_WS_MAEMO_5
	this->setAttribute(Qt::WA_Maemo5StackedWindow);
	this->setWindowFlags(Qt::Window);
	this->setAttribute(Qt::WA_Maemo5AutoOrientation);
#endif

	iFilterLabels
			<< tr("All marked packages")	// a special case
			<< tr("All user categories")	// special case: don't show "hidden" category
			<< tr("\tDesktop")
			<< tr("\tEducation")
			<< tr("\tGames")
			<< tr("\tGraphics")
			<< tr("\tInternet & Networking")
			<< tr("\tLocation & Navigation")
			<< tr("\tMultimedia")
			<< tr("\tOffice")
			<< tr("\tOther")
			<< tr("\tProgramming")
			<< tr("\tScience")
			<< tr("\tSystem")
			<< tr("\tUtilities")
			<< tr("All packages (ADVANCED)");

	iFilterStrings
			<< ""
			<< "user/"
			<< "user/desktop"
			<< "user/education"
			<< "user/games"
			<< "user/graphics"
			<< "user/network"
			<< "user/navigation"
			<< "user/multimedia"
			<< "user/office"
			<< "user/other"
			<< "user/development"
			<< "user/science"
			<< "user/system"
			<< "user/utilities"
			<< "";

	iDefaultFilter = FilterAllUser;
	iSelectedFilter = iDefaultFilter;

	iDimmer = new dimmer(this);
}

PackageView::~PackageView()
{
    delete ui;
}

void PackageView::openWin(MainWindow::operation op_)
{
	iOperation = op_;
	ui->listWidget->setEnabled(true);
	ui->listWidget->clear();
	ui->listWidget->setSortingEnabled(true);

	delete ui->listWidget->itemDelegate();
	ui->listWidget->setItemDelegate(new ListItemDelegate(ui->listWidget));

	iDontSelect = false;

	if( iOperation == MainWindow::OpInstall )
		this->setWindowTitle(tr("Install applications"));
	if( iOperation == MainWindow::OpUninstall )
		this->setWindowTitle(tr("Uninstall applications"));
	if( iOperation == MainWindow::OpUpgrade )
		this->setWindowTitle(tr("Upgrade applications"));

	QHashIterator<QString, Package*> i( *iAptInterface->packages() );
	while (i.hasNext())
	{
		i.next();

		QString name = i.value()->name();
		if( i.value()->maemoDisplayName()!="" )
			name = i.value()->maemoDisplayName();
		QString n( name.at(0) );
		n = n.toUpper();
		name.replace(0,1,n);

		if( iOperation == MainWindow::OpInstall )
		{
			if( !i.value()->isInstalled() && iSelectedFilter!=FilterAllMarked && i.value()->section().startsWith(iFilterStrings.at(iSelectedFilter))
				&& !i.value()->isLocal() )
			{
				if( iSelectedFilter==FilterAllUser && !i.value()->section().startsWith("user/hidden") )
					addListItem(i.value(), name);
				else if( iSelectedFilter!=FilterAllUser )
					addListItem(i.value(), name);
			} else if(iSelectedFilter==FilterAllMarked && i.value()->isMarkedForOperation() ) {
				addListItem(i.value(), name);
			}
		} else if( iOperation == MainWindow::OpUninstall )
		{
			if( i.value()->isInstalled() && iSelectedFilter!=FilterAllMarked && i.value()->section().startsWith(iFilterStrings.at(iSelectedFilter)) )
			{
				addListItem(i.value(), name);
			} else if(iSelectedFilter==FilterAllMarked && i.value()->isMarkedForOperation() ) {
				addListItem(i.value(), name);
			}
		} else if( iOperation == MainWindow::OpUpgrade )
		{
			if( i.value()->isInstalled() && iSelectedFilter!=FilterAllMarked && i.value()->versionAvailable()!=i.value()->versionInstalled()
				&& i.value()->versionAvailable()!="" && i.value()->section().startsWith(iFilterStrings.at(iSelectedFilter))
				&& !i.value()->isLocal() )
			{
				if( iSelectedFilter==FilterAllUser && !i.value()->section().startsWith("user/hidden") )
					addListItem(i.value(), name);
				else if( iSelectedFilter!=FilterAllUser )
					addListItem(i.value(), name);
			} else if(iSelectedFilter==FilterAllMarked && i.value()->isMarkedForOperation() ) {
				addListItem(i.value(), name);
			}
		}
	}

	if( ui->listWidget->count() == 0 )
	{
		addListItem(0, tr("No packages"));
		ui->listWidget->setEnabled(false);
		iDontSelect = true;
	}

	updateLabel();

	show();
}

void PackageView::enableMenu()
{
	ui->menuMenu->setEnabled(true);
}

void PackageView::disableMenu()
{
	ui->menuMenu->setEnabled(false);
}

void PackageView::addListItem(Package* pkg_, QString listname_)
{
	PackageListWidgetItem* p = new PackageListWidgetItem( pkg_, listname_ );

	if( pkg_ != 0 ) {
		QString name = pkg_->name();
		if( pkg_->maemoDisplayName()!="" )
			name = pkg_->maemoDisplayName();
		p->setData(UserRoleName, name);
	} else {
		p->setData(UserRoleName, listname_);
	}

	p->setData(UserRoleOperation, iOperation);

	if( pkg_ != 0 ) {
		p->setData(UserRoleDescShort, pkg_->descShort());
		p->setData(UserRoleVerAvail, pkg_->versionAvailable());
		p->setData(UserRoleVerInst, pkg_->versionInstalled());
		p->setData(UserRoleMarked, pkg_->isMarkedForOperation());
	}
	ui->listWidget->addItem( p );
}

void PackageView::closeEvent(QCloseEvent *event)
{
	if( iDimmer->busy() ) {
		event->ignore();
		return;
	}

	if( iAptInterface->selectedCount() == 0 )
	{		
		iSelectedFilter = iDefaultFilter;
		iMainWindow->setOperation(MainWindow::OpNone);
		event->accept();
	} else {
		QString c;
		c.setNum( iAptInterface->selectedCount() );
		ConfirmDialog d(true, this);
		d.setText(tr("Returning to main menu"), tr("Clear ") + c + tr(" package selection(s) and lose all the pending changes?"));
		if( d.exec() ) {
			iSelectedFilter = iDefaultFilter;
			iMainWindow->setOperation(MainWindow::OpNone);
			event->accept();
		} else {
			event->ignore();
		}
	}
}

void PackageView::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void PackageView::on_btn_Commit_clicked()
{
	QStringList pkgnames;
	QHashIterator<QString, Package*> i( *iAptInterface->packages() );

	std::cout << "COMMIT: " << std::endl;

	while (i.hasNext())
	{
		i.next();
		if( i.value()->isMarkedForOperation() ) {
			pkgnames << i.value()->name();

			std::cout << i.value()->name().toStdString() << " (" << i.value()->maemoDisplayName().toStdString() << ") "
					<< i.value()->versionInstalled().toStdString() << " "
					<< i.value()->versionAvailable().toStdString() << " "
					<< i.value()->section().toStdString() << std::endl;
		}
	}

	if( iOperation == MainWindow::OpInstall )
	{
		iAptInterface->simulateInstall(pkgnames);
	}
	if( iOperation == MainWindow::OpUninstall )
	{
		iAptInterface->simulateRemove(pkgnames);
	}
	if( iOperation == MainWindow::OpUpgrade )
	{
		iAptInterface->simulateInstall(pkgnames);
	}
}

void PackageView::on_actionClear_selections_triggered()
{
	QString c;
	c.setNum( iAptInterface->selectedCount() );
	ConfirmDialog d(true, this);
	d.setText(tr("Confirmation"), tr("Clear ") + c + tr(" package selection(s) and lose all the pending changes?"));
	if( d.exec() )
	{
		QHashIterator<QString, Package*> i( *iAptInterface->packages() );
		while (i.hasNext())
		{
			i.next();
			i.value()->setMarkedForOperation(false);
		}

		iAptInterface->resetSelectedCount();

		openWin(iOperation);
	}
}

void PackageView::on_actionRefresh_list_triggered()
{

}

void PackageView::on_actionFilter_triggered()
{
	on_btn_Filter_clicked();
}

void PackageView::on_actionSearch_triggered()
{

}

void PackageView::on_listWidget_itemClicked(QListWidgetItem* item)
{
	if( iDontSelect ) return;

	if( static_cast<PackageListWidgetItem*>(item)->package()->isMarkedForOperation() )
	{
		iAptInterface->decreaseSelected();
		static_cast<PackageListWidgetItem*>(item)->package()->setMarkedForOperation(false);
		static_cast<PackageListWidgetItem*>(item)->setData(UserRoleMarked, false);
	} else {
		iAptInterface->increaseSelected();
		static_cast<PackageListWidgetItem*>(item)->package()->setMarkedForOperation(true);
		static_cast<PackageListWidgetItem*>(item)->setData(UserRoleMarked, true);
	}

	updateLabel();
}

void PackageView::updateLabel()
{
	QString operation = tr("operation");
	if( iOperation == MainWindow::OpInstall )
	{
		operation = tr("installation");
	}
	if( iOperation == MainWindow::OpUninstall )
	{
		operation = tr("uninstallation");
	}
	if( iOperation == MainWindow::OpUpgrade )
	{
		operation = tr("upgrade");
	}

	QString s;
	s.setNum( iAptInterface->selectedCount() );
	QString s2;
	s2.setNum( ui->listWidget->count() );
	if( iDontSelect ) s2="0";
	ui->label->setText("<font size=\"-1\">" + s + tr(" package(s) selected for ") + operation + "</font><br><font size=\"-2\">"
					   + tr("Filter:") + " " + iFilterLabels.at(iSelectedFilter) + " - " + s2 + tr(" package(s)</font>"));

	if( iAptInterface->selectedCount()==0 ) {
		ui->btn_Commit->setEnabled(false);
		ui->actionClear_selections->setVisible(false);
	} else {
		ui->btn_Commit->setEnabled(true);
		ui->actionClear_selections->setVisible(true);
	}
}

void PackageView::on_btn_Filter_clicked()
{
	FilterSelect f(iMainWindow);
	f.setList( iFilterLabels, iSelectedFilter );
	if( f.exec() ) {
		iSelectedFilter = f.selection();
	}

	openWin(iOperation);
}

