/*******************************************************************************

 This file is a part of Fahrplan for maemo 2009-2010

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 ============================================================================
 Name        : fahrplan_gui.c
 Author      : smurfy
 Version     : 0.1
 Description : Handles all Gui functions
 ============================================================================
 */

#include "fahrplan_utils.h"
#include "fahrplan_gui.h"
#include "fahrplan_parser.h"
#include "fahrplan_parser_openstreetmap.h"

/**
 * OnClick Event for a HildonButton
 * shows a simple about dialog.
 */
void onAboutClicked(HildonButton * button, AppData *data) {

	char infoLabel[1000] = "";

	strcat(infoLabel, "<b>About Fahrplan</b>\n");
	strcat(infoLabel, "by smurfy (maemo@smurfy.de)\n");
	strcat(infoLabel, "Version ");
	strcat(infoLabel, APP_VER);
	strcat(infoLabel, "\n");
	strcat(infoLabel, "\n");
	strcat(infoLabel, "with contributions by\n");
	strcat(infoLabel, "<i>hcm</i> (various patches)\n");
	strcat(infoLabel, "<i>thp</i> (hildon themed buttons)\n");
	strcat(infoLabel, "<i>qwerty12</i> (two min crash)\n");
	strcat(infoLabel, "<i>qbast</i> (openstreetmap backend)");


	hildon_banner_show_information_with_markup(
			GTK_WIDGET(data->journeyWindow),
			NULL,
			infoLabel);
}

/**
 * OnClick Event for a HildonButton
 * Switchs to Depature Monitor
 */
void onDepartureMonitorClicked(HildonButton * button, AppData *data) {

	gtk_widget_hide_all(GTK_WIDGET ( data->journeyWindow ));
	gtk_widget_show_all(GTK_WIDGET ( data->departureWindow ));
}

/**
 * OnClick Event for a HildonButton
 * Switchs to Journey Planner
 */
void onJourneyPlannerClicked(HildonButton * button, AppData *data) {

	gtk_widget_hide_all(GTK_WIDGET ( data->departureWindow ));
	gtk_widget_show_all(GTK_WIDGET ( data->journeyWindow ));
}

/**
 * OnClick Event for a HildonButton
 * Handles display of station selector
 */
void onSelectStationClicked(HildonButton * button, AppData *data) {

	data->currentSelectorFor = GTK_WIDGET(button);
	initStationSelectorDialog(data);
}

/**
 * on value-changed Event for a HildonPickerButton
 * Handles setting HildonPickerButton's Value
 */
void onStationSelected(HildonPickerButton * button, AppData *data) {

	const gchar *currentSelection = hildon_button_get_value(HILDON_BUTTON (button));

	if (strcmp(currentSelection, "")) {

		hildon_button_set_value(HILDON_BUTTON(data->currentSelectorFor),
				currentSelection);
	}
}

/**
 * OnClick/OnActivate Event for a GtkWidget
 * Handles start search event for search for stations by name
 */
void onSearchStationClicked(GtkWidget *w, AppData *data) {

	const gchar *searchFor = NULL;

	searchFor = hildon_entry_get_text(HILDON_ENTRY(data->currentSearchTextBox));

	if (strlen(searchFor) == 0) {

		hildon_banner_show_information(
				GTK_WIDGET(data->journeyWindow),
				NULL,
				_("Please enter a Station name"));
		return;
	}

	GArray *result = getStationsByName((gchar *) searchFor, data);

	initStationPickerForArray(result, data);
}

/**
 * error-verbose handler for GPS
 * Displays an error if an GPS error Occures
 */
void onGPSError(LocationGPSDControl *control,gint error, AppData *data)
{
	hildon_banner_show_information(GTK_WIDGET(data->journeyWindow), NULL,
							_("Error with GPS"));

	hildon_gtk_window_set_progress_indicator(GTK_WINDOW(data->currentDialog), 0);
	location_gpsd_control_stop(data->currentGPSControl);
}

/**
 * OnClick Event for a HildonButton
 * Handles start of gps selection
 */
void onGetStationByGPSClicked(HildonButton * button, AppData *data) {

	if (!data->currentGPSDevice)
		return;

	if (data->currentGPSDevice->fix && data->currentGPSDevice->status == LOCATION_GPS_DEVICE_STATUS_FIX) {

		if (data->currentGPSDevice->fix->fields & LOCATION_GPS_DEVICE_LATLONG_SET) {

			struct lconv *lcp = localeconv();
			gchar *tmp = g_strdup_printf("%f", data->currentGPSDevice->fix->latitude);
			tmp = stringReplace(tmp, lcp->decimal_point, "");
			int poslat = atoi(tmp);
			tmp = g_strdup_printf("%f", data->currentGPSDevice->fix->longitude);
			tmp = stringReplace(tmp, lcp->decimal_point, "");
			int poslong =atoi(tmp);

			GArray *result = getStationsByGPS(poslong, poslat, data);

			/* Nothing found try openstreetmap backend */
			if (result==NULL || result->len == 0) {

				hildon_banner_show_information(
						GTK_WIDGET(data->journeyWindow),
						NULL,
						_("Trying openstreetmap to find your city by location"));

				gchar *cityname = getCityNameByGPS(data->currentGPSDevice->fix->longitude, data->currentGPSDevice->fix->latitude);

				if (cityname==NULL) {

					hildon_banner_show_information(
							GTK_WIDGET(data->journeyWindow),
							NULL,
							_("No Stations found near your location!"));
				} else {

					GArray *result = getStationsByName(cityname, data);

					initStationPickerForArray(result, data);
				}
			} else {

				initStationPickerForArray(result, data);
			}
		}
	} else {
		hildon_banner_show_information(GTK_WIDGET(data->journeyWindow), NULL,
				_("No GPS Fix yet!"));
	}
}
/**
 *
 * OnClick Event for a HildonButton
 * Handles start of the search
 */
void onStartSearchClicked(HildonButton * button, AppData *data) {

	const gchar *fromText = hildon_button_get_value(HILDON_BUTTON(data->fromButton));
	const gchar *toText = hildon_button_get_value(HILDON_BUTTON(data->toButton));

	if (strcmp(fromText, "") && strcmp(fromText, _("please select")) && strcmp(
			toText, "") && strcmp(toText, _("please select"))) {

		gint optionsSelected = hildon_touch_selector_get_active(
				HILDON_TOUCH_SELECTOR(data->optionsSelector), 0);
		gint year;
		gint month;
		gint day;
		gint hour;
		gint minutes;

		hildon_date_button_get_date(HILDON_DATE_BUTTON(data->timeDateButton),
				&year, &month, &day);

		hildon_time_button_get_time(HILDON_TIME_BUTTON(data->timeTimeButton),
				&hour, &minutes);

		ResultInfo *result = getJourneyData((gchar*)fromText, NULL, (gchar*)toText, day, month+1, year, hour, minutes, optionsSelected, data);

		initJournyResults(result, data);

	} else {

		hildon_banner_show_information(GTK_WIDGET(data->journeyWindow), NULL,
				_("Please select the from and the to station name."));
		return;
	}
}

/**
 *
 * OnClick Event for a HildonButton
 * Handles start of the departure search
 */
void onStartDepartureSearchClicked(HildonButton * button, AppData *data) {

	const gchar *fromText = hildon_button_get_value(HILDON_BUTTON(data->departureFromButton));

	if (strcmp(fromText, "") && strcmp(fromText, _("please select"))) {

		gint optionsSelected = hildon_touch_selector_get_active(
				HILDON_TOUCH_SELECTOR(data->optionsSelector), 0);
		gint year;
		gint month;
		gint day;
		gint hour;
		gint minutes;

		hildon_date_button_get_date(HILDON_DATE_BUTTON(data->departureTimeDateButton),
				&year, &month, &day);

		hildon_time_button_get_time(HILDON_TIME_BUTTON(data->departureTimeTimeButton),
				&hour, &minutes);

		/*

		REQTrain_name                                (??)
		REQTrain_name_filterSelf	1                (??)
		ao	yes                                      (??)
		boardType	Abfahrt                          (Or Ankunft/optionsSelected)
		date	20.03.10                             (Normal Date/departureTimeDateButton)
		input	m                                    (input string, could come from dropdown in parser / fromText)
		maxJourneys	10                               (we lock this to 10)
		productsFilter	1111111111000000             (train types bit mask hard to ALL?)
		start	Suchen                               (??)
		time	12:27                                (Normal Time/departureTimeTimeButton)

		ResultInfo *result = getJourneyData((gchar*)fromText, NULL, (gchar*)toText, day, month+1, year, hour, minutes, optionsSelected, data);

		initDepartureResults(result, data);

		*/

	} else {

		hildon_banner_show_information(GTK_WIDGET(data->departureWindow), NULL,
				_("Please select the station you want to get the Departure informations from."));
		return;
	}
}

/**
 *
 * OnClick Event for a HildonButton
 * Handles swapping of the from/to stations
 */
void onSwapClicked(HildonButton * button, AppData *data) {
	gchar *tmp = g_strdup_printf("%s", hildon_button_get_value(HILDON_BUTTON(data->toButton)));
	hildon_button_set_value(HILDON_BUTTON(data->toButton),
				hildon_button_get_value(HILDON_BUTTON(data->fromButton)));
	hildon_button_set_value(HILDON_BUTTON(data->fromButton), tmp);
}

/**
 * onClick Event hanlder for GtkButton
 * Function to handle the selection of an entry of the results page
 */
void onJournyResultClicked(GtkButton * button, AppData *data) {

	gchar *current_selection;
	current_selection = g_object_get_data(G_OBJECT(button), "URL");

	DetailInfo *details = getJourneyDetailDataByUrl(current_selection, data);
	initJournyDetails(details, data);
}

/**
 * onClick Event hanlder for GtkButton
 */
void onJournyLaterClicked(GtkButton * button, AppData *data) {

	gtk_widget_destroy(GTK_WIDGET(data->resultWindow));

	ResultInfo *result = getJourneyDataByUrl(data->currentResult->headerLaterUrl, data);
	initJournyResults(result, data);
}

/**
 * onClick Event hanlder for GtkButton
 */
void onJournyEarlierClicked(GtkButton * button, AppData *data) {

	gtk_widget_destroy(GTK_WIDGET(data->resultWindow));

	ResultInfo *result = getJourneyDataByUrl(data->currentResult->headerEarlierUrl, data);
	initJournyResults(result, data);
}

/**
 * Shows a Results list from ResultInfo Structure
 */
void initJournyDetails(DetailInfo *details, AppData *data) {

	if (!details) {

		hildon_banner_show_information(GTK_WIDGET(data->journeyWindow), NULL,
						_("An error has occured"));
	}

	if (details->errorMsg) {

		hildon_banner_show_information(GTK_WIDGET(data->journeyWindow), NULL,
				details->errorMsg);

		g_free(details);
		return;
	}

	/* Create the Subview for details */
	data->detailsWindow = HILDON_WINDOW(hildon_stackable_window_new());
	gtk_window_set_title(GTK_WINDOW (data->detailsWindow),
			_("Details"));

	GtkWidget *line;
	GtkWidget *lineHbox;
	GtkWidget *lineText;

	data->currentDetailsMainVbox = gtk_vbox_new(FALSE, 5);

	data->currentDetailsPannableArea = hildon_pannable_area_new();

    hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA (data->currentDetailsPannableArea), GTK_WIDGET (data->currentDetailsMainVbox));

    /* check, if request comes from gps data it occures that data is empty.*/
    if (!details->departureStr) {

		hildon_banner_show_information(GTK_WIDGET(data->journeyWindow), NULL,
						_("Error with received data, please try again"));
		return;
    }

    /* Journy Details Header */
	char infoLabel[2000] = "";

	strcat(infoLabel, "<b>");
	strcat(infoLabel, details->departureStr);
	strcat(infoLabel, "</b>");

	if (details->departureStr && details->arrivalStr && strcmp(details->departureStr, details->arrivalStr)) {

		strcat(infoLabel, _(" to "));
		strcat(infoLabel, "<b>");
		strcat(infoLabel, details->arrivalStr);
		strcat(infoLabel, "</b>");
	}
	strcat(infoLabel, "\n");

	strcat(infoLabel, "<b>");
	strcat(infoLabel, _("Duration: "));
	strcat(infoLabel, "</b>");
	strcat(infoLabel, details->durationStr);
	strcat(infoLabel, "\n");

	if (details->infoStr && strcmp(details->infoStr,"")) {

		strcat(infoLabel, "<b>");
		strcat(infoLabel, details->infoStr);
		strcat(infoLabel, "</b>\n");
	}

	if (details->pricingStr && strcmp(details->pricingStr,"")) {

		strcat(infoLabel, "<i>");
		strcat(infoLabel, details->pricingStr);
		strcat(infoLabel, "</i>\n");
	}

	infoLabel[strlen(infoLabel) -1] = '\0';

        line = hildon_button_new(HILDON_SIZE_THUMB_HEIGHT | HILDON_SIZE_AUTO_WIDTH, 0);
	gtk_button_set_focus_on_click(GTK_BUTTON(line), FALSE);
	gtk_button_set_relief(GTK_BUTTON(line), GTK_RELIEF_NONE);

	lineHbox = gtk_hbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(line), lineHbox);
	lineText = gtk_label_new(NULL);
	gtk_label_set_justify(GTK_LABEL(lineText), GTK_JUSTIFY_FILL);
	gtk_label_set_markup(GTK_LABEL(lineText), infoLabel);
	gtk_box_pack_start(GTK_BOX(lineHbox), lineText, FALSE, TRUE, 10);

	gtk_box_pack_start(GTK_BOX(data->currentDetailsMainVbox), line, FALSE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(data->currentDetailsMainVbox), gtk_hseparator_new(), FALSE, TRUE, 0);

	/* Journey Detail Stops */

	if (details->items) {

		int num = 0;
		DetailItem * value = NULL;
		while(value = (DetailItem*)g_array_index(details->items, DetailItem*, num)) {

			num++;

			char infoLabel[1000] = "";

			if (value->trainStr && strcmp(value->trainStr,"")) {

				strcat(infoLabel, "<b>");
				strcat(infoLabel, value->trainStr);
				strcat(infoLabel, "</b>\n");
			}

			if (value->fromStr && strcmp(value->fromStr,"")) {

				strcat(infoLabel, value->fromStr);
			}
			if (value->timeFromStr && strcmp(value->timeFromStr,"")) {

				strcat(infoLabel, " - ");
				strcat(infoLabel, value->timeFromStr);
			}
			if (value->fromStr && strcmp(value->fromStr,"") || value->timeFromStr && strcmp(value->timeFromStr,"")) {

				strcat(infoLabel, "\n");
			}

			if (value->toStr && strcmp(value->toStr,"")) {

				strcat(infoLabel, value->toStr);
			}
			if (value->timeToStr && strcmp(value->timeToStr,"")) {

				strcat(infoLabel, " - ");
				strcat(infoLabel, value->timeToStr);
			}
			if (value->toStr && strcmp(value->toStr,"") || value->timeToStr && strcmp(value->timeToStr,"")) {

				strcat(infoLabel, "\n");
			}

			if (value->alternateStr && strcmp(value->alternateStr,"")) {

				strcat(infoLabel, "<i>");
				strcat(infoLabel, value->alternateStr);
				strcat(infoLabel, "</i>\n");
			}

			if (value->infoStr && strcmp(value->infoStr,"")) {

				strcat(infoLabel, "<b>");
				strcat(infoLabel, value->infoStr);
				strcat(infoLabel, "</b>\n");
			}

			infoLabel[strlen(infoLabel) -1] = '\0';

            line = hildon_button_new(HILDON_SIZE_THUMB_HEIGHT | HILDON_SIZE_AUTO_WIDTH, 0);
			gtk_button_set_focus_on_click(GTK_BUTTON(line), FALSE);
			gtk_button_set_relief(GTK_BUTTON(line), GTK_RELIEF_NONE);

			lineHbox = gtk_hbox_new(FALSE, 0);
			gtk_container_add(GTK_CONTAINER(line), lineHbox);
			lineText = gtk_label_new(NULL);
			gtk_label_set_justify(GTK_LABEL(lineText), GTK_JUSTIFY_FILL);
			gtk_label_set_markup(GTK_LABEL(lineText), infoLabel);
			gtk_box_pack_start(GTK_BOX(lineHbox), lineText, FALSE, TRUE, 10);

			gtk_box_pack_start(GTK_BOX(data->currentDetailsMainVbox), line, FALSE, TRUE, 0);
			gtk_box_pack_start(GTK_BOX(data->currentDetailsMainVbox), gtk_hseparator_new(), FALSE, TRUE, 0);
		}
	}

	gtk_container_add(GTK_CONTAINER (data->detailsWindow), data->currentDetailsPannableArea);

	gtk_widget_show_all(GTK_WIDGET(data->detailsWindow));
}

/**
 * Shows a Results list from ResultInfo Structure
 */
void initJournyResults(ResultInfo *result, AppData *data) {

	if (!result) {

		hildon_banner_show_information(GTK_WIDGET(data->journeyWindow), NULL,
						_("An error has occured"));
	}

	if (result->errorMsg) {

		hildon_banner_show_information(GTK_WIDGET(data->journeyWindow), NULL,
				result->errorMsg);

		g_free(result);
		return;
	}

	/* Create the Subview for results */
	data->resultWindow = HILDON_WINDOW(hildon_stackable_window_new());
	data->currentResult = result;
	gtk_window_set_title(GTK_WINDOW (data->resultWindow),
			_("Results"));

	GtkWidget *vbox;
	vbox = gtk_vbox_new(FALSE, 0);


	/* Header */
	GtkWidget *infoLabel;
	gchar * labelText;
	labelText = g_strdup_printf("%s <b>%s</b> %s <b>%s</b>\n<b>%s</b>", _("From: "), result->headerFromStation, _("To: "), result->headerToStation, result->headerTimeInfo);
	infoLabel = gtk_label_new("");
	gtk_label_set_markup(GTK_LABEL(infoLabel), labelText);
	gtk_label_set_justify(GTK_LABEL(infoLabel), GTK_JUSTIFY_CENTER);

	g_free(labelText);
	gtk_box_pack_start(GTK_BOX (vbox), infoLabel, FALSE, TRUE, 5);

	/* Menu */
	GtkWidget * button;
	HildonAppMenu *menu = HILDON_APP_MENU (hildon_app_menu_new ());

	if (result->headerEarlierUrl) {

		/* earlier button */
		button = hildon_button_new(HILDON_SIZE_THUMB_HEIGHT | HILDON_SIZE_AUTO_WIDTH, 0);
			gtk_button_set_label(GTK_BUTTON (button), _("Earlier"));

		/* Attach callback to clicked signal */
		g_signal_connect_after (button, "clicked", G_CALLBACK (onJournyEarlierClicked), data);

		/* Add entry to the view menu */
		hildon_app_menu_append(menu, GTK_BUTTON (button));
	}

	if (result->headerLaterUrl) {

		/* later button */
		button = hildon_button_new(HILDON_SIZE_THUMB_HEIGHT | HILDON_SIZE_AUTO_WIDTH, 0);
		gtk_button_set_label(GTK_BUTTON (button), _("Later"));

		/* Attach callback to clicked signal */
		g_signal_connect_after (button, "clicked", G_CALLBACK (onJournyLaterClicked), data);

		/* Add entry to the view menu */
		hildon_app_menu_append(menu, GTK_BUTTON (button));
	}

	gtk_widget_show_all(GTK_WIDGET (menu));
	hildon_window_set_app_menu (HILDON_WINDOW (data->resultWindow), menu);

	/* Result Items */
	if (result->items) {

		data->currentResultPannableArea = hildon_pannable_area_new();

		data->currentResultMainVbox = gtk_vbox_new(FALSE, 5);

	    hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA (data->currentResultPannableArea), GTK_WIDGET (data->currentResultMainVbox));

		int num = 0;
		ResultItem * value = NULL;
		while(value = (ResultItem*)g_array_index(result->items, ResultItem*, num)) {

			num++;

			GtkWidget *line;
			GtkWidget *lineHbox;
			GtkWidget *lineText;

			gchar * labelText;

			char *transfersStr= _("transfers");

			if (atoi(value->changeStr)==1) {

				transfersStr= _("transfer");
			}

			if (!value->warningStr || !strcmp(value->warningStr, "")) {

				labelText = g_strdup_printf("<b>%s - %s</b> %s (%s %s)", value->fromTimeStr, value->toTimeStr, value->trainTypeStr, value->changeStr, transfersStr);
			} else {

				labelText = g_strdup_printf("<b>%s - %s</b> %s (%s %s)\n%s", value->fromTimeStr, value->toTimeStr, value->trainTypeStr, value->changeStr, transfersStr, value->warningStr);
			}

            line = hildon_button_new(HILDON_SIZE_THUMB_HEIGHT | HILDON_SIZE_AUTO_WIDTH, 0);
			gtk_button_set_focus_on_click(GTK_BUTTON(line), FALSE);
			gtk_button_set_relief(GTK_BUTTON(line), GTK_RELIEF_NONE);

			gtk_widget_set_size_request  (line, -1, 70);

			lineHbox = gtk_hbox_new(FALSE, 0);
			gtk_container_add(GTK_CONTAINER(line), lineHbox);
			lineText = gtk_label_new(NULL);
			gtk_label_set_justify(GTK_LABEL(lineText), GTK_JUSTIFY_FILL);
			gtk_label_set_markup(GTK_LABEL(lineText), labelText);
			gtk_box_pack_start(GTK_BOX(lineHbox), lineText, FALSE, TRUE, 10);

			gtk_box_pack_start(GTK_BOX(data->currentResultMainVbox), line, FALSE, TRUE, 0);
			gtk_box_pack_start(GTK_BOX(data->currentResultMainVbox), gtk_hseparator_new(), FALSE, TRUE, 0);

			g_signal_connect (G_OBJECT(line), "clicked",  G_CALLBACK (onJournyResultClicked), data);

			g_object_set_data(G_OBJECT(line), "URL", value->detailsUrl);
		}

		gtk_box_pack_start(GTK_BOX (vbox), data->currentResultPannableArea, TRUE, TRUE, 0);
	}

	gtk_container_add(GTK_CONTAINER (data->resultWindow), vbox);

	gtk_widget_show_all(GTK_WIDGET(data->resultWindow));
}

/**
 * Shows a picker dialog for Stations found array
 */
void initStationPickerForArray(GArray *result, AppData *data) {

	if (!result) {

		hildon_banner_show_information(
				GTK_WIDGET(data->journeyWindow),
				NULL,
				_("An error occured while requesting data over the internet."));

		gtk_widget_hide(data->currentDialog);
		gtk_widget_destroy(data->currentDialog);
		data->currentDialog = NULL;
	} else {

		gchar *value = NULL;

		/* nothing found */
		if (!g_array_index(result, gchar*, 0)) {

			hildon_banner_show_information(
					GTK_WIDGET(data->journeyWindow),
					NULL,
					_("Station not found!"));

		} /* only one result found */
		else if (!g_array_index(result, gchar*, 1)) {

			value = g_array_index(result, gchar*, 0);
			hildon_button_set_value(HILDON_BUTTON(data->currentSelectorFor), value);

			gtk_widget_hide(data->currentDialog);
			gtk_widget_destroy(data->currentDialog);
			data->currentDialog = NULL;

		} /* more than one result found */
		else {

			if (data->currentSelector) {

				gtk_widget_destroy(data->currentSelector);
			}
			data->currentSelector = hildon_touch_selector_new_text();

			int num = 0;
			value = NULL;
			while(value = g_array_index(result, gchar*, num)) {

				num++;
				hildon_touch_selector_append_text(
									HILDON_TOUCH_SELECTOR (data->currentSelector),
									value);

			}

			data->currentTmpPicker = hildon_picker_button_new(HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH,
					HILDON_BUTTON_ARRANGEMENT_VERTICAL);
			hildon_button_set_title(HILDON_BUTTON (data->currentTmpPicker),
					_("Stations found"));

			g_signal_connect (G_OBJECT (data->currentTmpPicker), "value-changed",
					G_CALLBACK (onStationSelected), data);

			hildon_picker_button_set_selector(
					HILDON_PICKER_BUTTON (data->currentTmpPicker),
					HILDON_TOUCH_SELECTOR (data->currentSelector));

			gtk_button_clicked(GTK_BUTTON (data->currentTmpPicker));

			gtk_widget_hide(data->currentDialog);
			gtk_widget_destroy(data->currentDialog);
			data->currentDialog = NULL;
		}

		g_array_free(result, TRUE);
	}
}

/**
 * Opens a GTK Dialog for station Selection
 */
void initStationSelectorDialog(AppData *data) {

	GtkWidget *submitButton;
	GtkWidget *gpsButton;

	data->currentDialog = gtk_dialog_new();
	gtk_window_set_transient_for(GTK_WINDOW (data->currentDialog),
			GTK_WINDOW (data->journeyWindow));

	gtk_window_set_title(GTK_WINDOW (data->currentDialog),
			_("City Stationname"));

	data->currentSearchTextBox = hildon_entry_new(HILDON_SIZE_AUTO);

	hildon_entry_set_text(HILDON_ENTRY(data->currentSearchTextBox), "");

	gtk_box_pack_start(GTK_BOX (GTK_DIALOG(data->currentDialog)->vbox),
			data->currentSearchTextBox, TRUE, TRUE, 0);

	submitButton = gtk_dialog_add_button(GTK_DIALOG(data->currentDialog), _("Search"), 0);

	g_signal_connect (G_OBJECT (submitButton), "clicked", G_CALLBACK (onSearchStationClicked), data);

	if (data->GPSEnabled)
	{
		gpsButton = gtk_dialog_add_button(GTK_DIALOG(data->currentDialog), _("GPS"), 0);
		g_signal_connect (G_OBJECT (gpsButton), "clicked", G_CALLBACK (onGetStationByGPSClicked), data);
	}

	g_signal_connect (G_OBJECT (data->currentSearchTextBox), "activate", G_CALLBACK (onSearchStationClicked), data);

	gtk_box_pack_start(GTK_BOX (GTK_DIALOG(data->currentDialog)->vbox),
			submitButton, TRUE, TRUE, 0);

	gtk_widget_show_all(GTK_WIDGET (data->currentDialog));
	gtk_dialog_run(GTK_DIALOG (data->currentDialog));
}
/**
 * Init Departure Monitor Gui Window
 */
void initDepartureMonitorGui(AppData *data) {

	/* Create HildonWindow and set it to HildonProgram */
	data->departureWindow = HILDON_WINDOW(hildon_stackable_window_new());
	hildon_program_add_window(data->program, data->departureWindow);

	/* Quit program when window is closed. */
	g_signal_connect (G_OBJECT (data->departureWindow), "delete_event",
			G_CALLBACK (gtk_main_quit), NULL);

	/* Quit program when window is otherwise destroyed. */
	g_signal_connect (G_OBJECT (data->departureWindow), "destroy",
			G_CALLBACK (gtk_main_quit), NULL);

	/* create vertical container to stack the other containers vertically and add to main view */
	data->departureVerticalContainer = gtk_vbox_new(FALSE, 4);
	gtk_container_add(GTK_CONTAINER(data->departureWindow), data->departureVerticalContainer);

	/* From Button */
	data->departureFromButton = hildon_button_new_with_text(HILDON_SIZE_AUTO,
			HILDON_BUTTON_ARRANGEMENT_HORIZONTAL, _("Station"),
			_("please select"));

	hildon_button_set_style(HILDON_BUTTON(data->departureFromButton),
			HILDON_BUTTON_STYLE_PICKER);

	g_signal_connect (G_OBJECT (data->departureFromButton), "clicked", G_CALLBACK (onSelectStationClicked), data);

	gtk_box_pack_start(GTK_BOX (data->departureVerticalContainer), data->departureFromButton, TRUE,
			TRUE, 0);


	/* create "time" elements */
	data->departureTimeContainer = gtk_hbox_new(FALSE, 4);

	data->departureTimeDateButton = hildon_date_button_new(HILDON_SIZE_AUTO,
			HILDON_BUTTON_ARRANGEMENT_HORIZONTAL);

	hildon_button_set_title(HILDON_BUTTON (data->departureTimeDateButton), _("Date"));
	gtk_box_pack_start(GTK_BOX (data->departureTimeContainer), data->departureTimeDateButton,
			TRUE, TRUE, 0);

	data->departureTimeTimeButton = hildon_time_button_new(HILDON_SIZE_AUTO,
			HILDON_BUTTON_ARRANGEMENT_HORIZONTAL);
	hildon_button_set_title(HILDON_BUTTON (data->departureTimeTimeButton), _("Time"));

	gtk_box_pack_start(GTK_BOX (data->departureTimeContainer), data->departureTimeTimeButton,
			TRUE, TRUE, 0);

	gtk_box_pack_start(GTK_BOX (data->departureVerticalContainer), data->departureTimeContainer,
			TRUE, TRUE, 0);

	/* options button (for the time */
	data->departureOptionsButton = hildon_picker_button_new(HILDON_SIZE_AUTO,
			HILDON_BUTTON_ARRANGEMENT_HORIZONTAL);
	hildon_button_set_title(HILDON_BUTTON (data->departureOptionsButton),
			_(""));

	data->departureOptionsSelector = hildon_touch_selector_new_text();

	hildon_touch_selector_append_text(
			HILDON_TOUCH_SELECTOR (data->departureOptionsSelector), _("Arrival"));
	hildon_touch_selector_append_text(
			HILDON_TOUCH_SELECTOR (data->departureOptionsSelector), _("Departure"));

	hildon_touch_selector_set_active(
			HILDON_TOUCH_SELECTOR (data->departureOptionsSelector), 0, 1);

	hildon_picker_button_set_selector(
			HILDON_PICKER_BUTTON (data->departureOptionsButton),
			HILDON_TOUCH_SELECTOR (data->departureOptionsSelector));

	gtk_box_pack_start(GTK_BOX (data->departureVerticalContainer), data->departureOptionsButton,
			TRUE, TRUE, 0);

	/* search button */
	data->departureSearchButton = gtk_button_new_with_label(_("Search"));

	g_signal_connect(G_OBJECT(data->departureSearchButton), "clicked",
			G_CALLBACK(onStartDepartureSearchClicked), data);

	gtk_box_pack_start(GTK_BOX (data->departureVerticalContainer), data->departureSearchButton,
			TRUE, TRUE, 0);

}
/**
 * Init Journey Planer Gui Window
 */
void initJourneyPlanerGui(AppData *data) {

	/* Create HildonWindow and set it to HildonProgram */
	data->journeyWindow = HILDON_WINDOW(hildon_stackable_window_new());
	hildon_program_add_window(data->program, data->journeyWindow);

	/* Quit program when window is closed. */
	g_signal_connect (G_OBJECT (data->journeyWindow), "delete_event",
			G_CALLBACK (gtk_main_quit), NULL);

	/* Quit program when window is otherwise destroyed. */
	g_signal_connect (G_OBJECT (data->journeyWindow), "destroy",
			G_CALLBACK (gtk_main_quit), NULL);

	/* create vertical container to stack the other containers vertically and add to main view */
	data->verticalContainer = gtk_vbox_new(FALSE, 4);
	gtk_container_add(GTK_CONTAINER(data->journeyWindow), data->verticalContainer);

	/* create stations container */
	data->stationsVContainer = gtk_hbox_new(FALSE, 4);
	data->stationsHContainer = gtk_vbox_new(FALSE, 4);

	/* create "from" elements */
	data->fromContainer = gtk_hbox_new(FALSE, 4);

	data->fromLabel = gtk_label_new(_("From:"));
	gtk_box_pack_start(GTK_BOX (data->fromContainer), data->fromLabel, FALSE,
			TRUE, 0);
	gtk_widget_set_usize(data->fromLabel, 100, 10);

	data->fromButton = hildon_button_new_with_text(HILDON_SIZE_THUMB_HEIGHT | HILDON_SIZE_AUTO_WIDTH,
			HILDON_BUTTON_ARRANGEMENT_HORIZONTAL, _("Station"),
			_("please select"));

	hildon_button_set_style(HILDON_BUTTON(data->fromButton),
			HILDON_BUTTON_STYLE_PICKER);

	g_signal_connect (G_OBJECT (data->fromButton), "clicked", G_CALLBACK (onSelectStationClicked), data);

	gtk_box_pack_start(GTK_BOX (data->fromContainer), data->fromButton, TRUE,
			TRUE, 0);

	gtk_box_pack_start(GTK_BOX (data->stationsHContainer), data->fromContainer,
			TRUE, TRUE, 0);

	/* create "to" elements */
	data->toContainer = gtk_hbox_new(FALSE, 4);

	data->toLabel = gtk_label_new(_("To:"));
	gtk_box_pack_start(GTK_BOX (data->toContainer), data->toLabel, FALSE, TRUE,
			0);
	gtk_widget_set_usize(data->toLabel, 100, 10);

	data->toButton = hildon_button_new_with_text(HILDON_SIZE_THUMB_HEIGHT | HILDON_SIZE_AUTO_WIDTH,
			HILDON_BUTTON_ARRANGEMENT_HORIZONTAL, _("Station"),
			_("please select"));

	hildon_button_set_style(HILDON_BUTTON(data->toButton),
			HILDON_BUTTON_STYLE_PICKER);

	g_signal_connect (G_OBJECT (data->toButton), "clicked", G_CALLBACK (onSelectStationClicked), data);

	gtk_box_pack_start(GTK_BOX (data->toContainer), data->toButton, TRUE, TRUE,
			0);

	gtk_box_pack_start(GTK_BOX (data->stationsHContainer), data->toContainer,
			TRUE, TRUE, 0);

	gtk_box_pack_start(GTK_BOX(data->stationsVContainer), data->stationsHContainer, TRUE, TRUE, 0);

	/* create swap button */
    data->swapButton = gtk_button_new_with_label("\xe2\x86\x95");
	g_signal_connect(G_OBJECT(data->swapButton), "clicked", G_CALLBACK(onSwapClicked), data);
	gtk_box_pack_start(GTK_BOX(data->stationsVContainer), data->swapButton, FALSE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(data->verticalContainer), data->stationsVContainer, TRUE, TRUE, 0);

	/* create "time" elements */
	data->timeContainer = gtk_hbox_new(FALSE, 4);

	data->timeDateButton = hildon_date_button_new(HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH,
			HILDON_BUTTON_ARRANGEMENT_HORIZONTAL);

	hildon_button_set_title(HILDON_BUTTON (data->timeDateButton), _("Date"));
	gtk_box_pack_start(GTK_BOX (data->timeContainer), data->timeDateButton,
			TRUE, TRUE, 0);

	data->timeTimeButton = hildon_time_button_new(HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH,
			HILDON_BUTTON_ARRANGEMENT_HORIZONTAL);
	hildon_button_set_title(HILDON_BUTTON (data->timeTimeButton), _("Time"));

	gtk_box_pack_start(GTK_BOX (data->timeContainer), data->timeTimeButton,
			TRUE, TRUE, 0);

	gtk_box_pack_start(GTK_BOX (data->verticalContainer), data->timeContainer,
			TRUE, TRUE, 0);

	/* options button (for the time */
	data->optionsButton = hildon_picker_button_new(HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH,
			HILDON_BUTTON_ARRANGEMENT_HORIZONTAL);
	hildon_button_set_title(HILDON_BUTTON (data->optionsButton),
			_("Use selected date/time as"));

	data->optionsSelector = hildon_touch_selector_new_text();

	hildon_touch_selector_append_text(
			HILDON_TOUCH_SELECTOR (data->optionsSelector), _("Arrival time"));
	hildon_touch_selector_append_text(
			HILDON_TOUCH_SELECTOR (data->optionsSelector), _("Departure time"));

	hildon_touch_selector_set_active(
			HILDON_TOUCH_SELECTOR (data->optionsSelector), 0, 1);

	hildon_picker_button_set_selector(
			HILDON_PICKER_BUTTON (data->optionsButton),
			HILDON_TOUCH_SELECTOR (data->optionsSelector));

	gtk_box_pack_start(GTK_BOX (data->verticalContainer), data->optionsButton,
			TRUE, TRUE, 0);

	/* search button */
	data->searchButton = hildon_button_new_with_text(HILDON_SIZE_THUMB_HEIGHT | HILDON_SIZE_AUTO_WIDTH, 0,
	                _("Search"), NULL);

	g_signal_connect(G_OBJECT(data->searchButton), "clicked",
			G_CALLBACK(onStartSearchClicked), data);

	gtk_box_pack_start(GTK_BOX (data->verticalContainer), data->searchButton,
			TRUE, TRUE, 0);

}

/** Creates the Main Menu and appends to the main windows */
void initMainWindowsMenu(AppData *data) {

	/* Create menu */
	GtkWidget * button;
	HildonAppMenu *menu = HILDON_APP_MENU (hildon_app_menu_new ());

	button = hildon_gtk_button_new(HILDON_SIZE_AUTO);
	gtk_button_set_label(GTK_BUTTON (button), _("About"));

	/* Attach callback to clicked signal */
	g_signal_connect_after (button, "clicked", G_CALLBACK (onAboutClicked), data);

	/* Add entry to the view menu */
	hildon_app_menu_append(menu, GTK_BUTTON (button));

	/* Create filters */

	button = hildon_gtk_radio_button_new(HILDON_SIZE_AUTO, NULL);
	gtk_button_set_label(GTK_BUTTON (button), _("Journey Planner"));
	hildon_app_menu_add_filter(menu, GTK_BUTTON (button));
	gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON (button), FALSE);

	/* Attach callback to clicked signal */
	g_signal_connect_after (button, "clicked", G_CALLBACK (onJourneyPlannerClicked), data);

	/*

	button = hildon_gtk_radio_button_new_from_widget(HILDON_SIZE_AUTO, GTK_RADIO_BUTTON(button));
	gtk_button_set_label(GTK_BUTTON (button), _("Departures"));
	hildon_app_menu_add_filter(menu, GTK_BUTTON (button));
	gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON (button), FALSE);

	*/

	/* Attach callback to clicked signal */
	/*
		g_signal_connect_after (button, "clicked", G_CALLBACK (onDepartureMonitorClicked), data);
	*/

	gtk_widget_show_all(GTK_WIDGET (menu));

	/* Attach menu to the windows */
	hildon_window_set_app_menu (HILDON_WINDOW (data->journeyWindow), menu);
	hildon_window_set_app_menu (HILDON_WINDOW (data->departureWindow), menu);
}

/**
 * Loads test Elements
 */
void initTestElements(AppData *data) {

	hildon_button_set_value(HILDON_BUTTON(data->fromButton),
					"Rosenheim");

	hildon_button_set_value(HILDON_BUTTON(data->toButton),
						"Augsburg Hbf");
}

/**
 * Inits the GPS System
 */
void initGPS(AppData *data) {

	data->currentGPSControl = location_gpsd_control_get_default();
	data->currentGPSDevice = g_object_new(LOCATION_TYPE_GPS_DEVICE, NULL);

	g_object_set(G_OBJECT(data->currentGPSControl), "preferred-method",
			LOCATION_METHOD_USER_SELECTED, "preferred-interval",
			LOCATION_INTERVAL_DEFAULT, NULL);

	g_signal_connect(data->currentGPSControl, "error-verbose", G_CALLBACK(onGPSError), data);

	location_gpsd_control_start(data->currentGPSControl);

	if ((data->currentGPSDevice->fix->latitude > 0) || (data->currentGPSDevice->fix->latitude < 0)) {

		data->GPSEnabled = 1;

	} else {

		data->GPSEnabled = 0;

		hildon_banner_show_information(GTK_WIDGET(data->journeyWindow), NULL,
				_("GPS not available!"));

		location_gpsd_control_stop(data->currentGPSControl);

		return;
	}

}

/**
 * Creates the main ui with all its buttons and stuff.
 */
void initApplication(AppData *data) {

	/* Create the hildon program and setup the title */
	data->program = HILDON_PROGRAM(hildon_program_get_instance());
	g_set_application_name(_("Fahrplan"));

	/* Init Journy Planer window */
	initJourneyPlanerGui(data);

	/* Init Departure Monitor Planer window */
	initDepartureMonitorGui(data);

	/* Init Test Elements */
	/* initTestElements(data); */

	/* Create Main Menu */
	initMainWindowsMenu(data);

	/* Show Default Window Window */
	gtk_widget_show_all(GTK_WIDGET ( data->journeyWindow ));

	/* Init GPS */
	initGPS(data);
}
