/* Evil Plumber is a small puzzle game.
   Copyright (C) 2010 Marja Hassinen

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef GAME__H
#define GAME__H

#include <QObject>
#include <QHash>
#include <QTimer>
#include <QStringList>

class QTableWidget;
class QTableWidgetItem;
class QLabel;
class QWidget;
class QPushButton;
class QListWidget;

enum PieceType
{
    PieceNone = 0,
    PieceStraight,
    PieceCorner,
    PieceCross,
    PieceCorners,
    PieceStart,
    PieceEnd,
    PieceBlock,
    PiecesEnd
};

enum Direction
{
    DirNone = 0,
    DirLeft,
    DirUp,
    DirRight,
    DirDown,
    DirDone,
    DirFailed,
    DirPassed
};

typedef struct Piece
{
    PieceType type;
    int rotation;
    bool userCanAdd;
    int uiRow;
    int uiColumn;
    Direction flows[4];
} Piece;

static const Piece ppieces[] = {
    {PieceNone, 0, false, 0, 0,              // 0 
     {DirNone, DirNone, DirNone, DirNone}},
    {PieceStraight, 0, true, 0, 0,           // 1
     {DirUp, DirDown, DirNone, DirNone}},
    {PieceStraight, 90, true, 0, 1,          // 2
     {DirLeft, DirRight, DirNone, DirNone}},
    {PieceCorner, 270, true, 1, 0,           // 3
     {DirDown, DirRight, DirNone, DirNone}},
    {PieceCorner, 0, true, 1, 1,             // 4
     {DirLeft, DirDown, DirNone, DirNone}},
    {PieceCorner, 180, true, 2, 0,           // 5
     {DirUp, DirRight, DirNone, DirNone}},
    {PieceCorner, 90, true, 2, 1,            // 6
     {DirLeft, DirUp, DirNone, DirNone}},
    {PieceCross, 0, true, 3, 0,              // 7
     {DirLeft, DirRight, DirUp, DirDown}},
    {PieceCorners, 0, true, 4, 0,            // 8
     {DirLeft, DirDown, DirRight, DirUp}},
    {PieceCorners, 90, true, 4, 1,          // 9
     {DirRight, DirDown, DirLeft, DirUp}},
    {PieceStart, 0, false, 0, 0,             // 10
     {DirLeft, DirNone, DirNone, DirNone}},
    {PieceStart, 90, false, 0, 0,            // 11
     {DirUp, DirNone, DirNone, DirNone}},
    {PieceStart, 180, false, 0, 0,           // 12
     {DirRight, DirNone, DirNone, DirNone}},
    {PieceStart, 270, false, 0, 0,           // 13
     {DirDown, DirNone, DirNone, DirNone}},
    {PieceEnd, 0, false, 0, 0,               // 14
     {DirLeft, DirDone, DirNone, DirNone}},
    {PieceEnd, 90, false, 0, 0,              // 15
     {DirUp, DirDone, DirNone, DirNone}},
    {PieceEnd, 180, false, 0, 0,             // 16
     {DirRight, DirDone, DirNone, DirNone}},
    {PieceEnd, 270, false, 0, 0,             // 17
     {DirDown, DirDone, DirNone, DirNone}},
    {PieceBlock, 0, false, 0, 0,             // 18
     {DirNone, DirNone, DirNone, DirNone}},
    {PiecesEnd, 0, false, 0, 0,              // 19
     {DirNone, DirNone, DirNone, DirNone}}};

const int noPieces = 20;

typedef struct _PlacedPiece
{
    const Piece* piece;
    bool fixed;
    bool flow[2]; // which directions the liquid has already flown
} PlacedPiece;

typedef struct _PrePlacedPiece
{
    const Piece* piece;
    int row;
    int col;
} PrePlacedPiece;

typedef struct _AvailablePiece
{
    const Piece* piece;
    int count;
} AvailablePiece;

class GameField : public QObject
{
    Q_OBJECT
public:
    GameField(QTableWidget* ui);
    void initGame(int rows_, int cols_, int count, PrePlacedPiece* prePlaced);

    bool setPiece(int row, int col, const Piece* piece, bool fixed = false);
    const Piece* pieceAt(int row, int col);
    bool isPrePlaced(int row, int col);
    void indicateFlow(int row, int col, Direction dir);

signals:
    void cellClicked(int, int);

private:
    int toIndex(int row, int col);

    QTableWidget* fieldUi;
    PlacedPiece* field;
    int rows;
    int cols;
};

class AvailablePieces : public QObject
{
    Q_OBJECT
public:
    AvailablePieces(QTableWidget* pieceTable);
    void initGame(int count, AvailablePiece* pieces);

signals:
    void validPieceSelected(const Piece* piece);
    void invalidPieceSelected();

public slots:
    void onPieceUsed(const Piece* piece);

private slots:
    void onItemClicked(QTableWidgetItem* item);

private:
    static int pieceToId(const Piece* piece);
    static const Piece* idToPiece(int id);

    QTableWidget* pieceUi;
    QHash<const Piece*, int> pieceCounts;
};

class GameController : public QObject
{
    Q_OBJECT
public:
    GameController(AvailablePieces* pieceUi, GameField* fieldUi, 
                   QLabel* timeLabel, QPushButton* doneButton);
    void startLevel(QString fileName);

signals:
    void pieceUsed(const Piece* piece);
    void levelPassed(int score);
    void levelFailed();

private slots:
    void onDoneClicked();
    void onCellClicked(int row, int column);
    void onValidPieceSelected(const Piece* piece);
    void onInvalidPieceSelected();
    void onTimeout();
    void computeFlow();

private:
    void levelEnds();

    AvailablePieces* pieceUi; // Not owned
    GameField* fieldUi; // Not owned
    QLabel* timeLabel; // Not owned
    QPushButton* doneButton; // Not owned

    // Data about the current situation in the game
    const Piece* currentPiece;
    int rows, cols;
    QTimer timer;
    QTimer flowTimer;
    int timeLeft;
    bool levelRunning;
    // how many times does the liquid need to flow through the pre-placed pieces
    int neededFlow;
    int startRow;
    int startCol;
    Direction startDir;
    int flowRow;
    int flowCol;
    Direction flowDir;
    int flowPreplaced;
    int flowScore;
    bool animate;
};

class LevelSwitcher : public QObject
{
    Q_OBJECT
public:
    LevelSwitcher(GameController* gameController,
                  QWidget* levelWidget, QListWidget* levelList, QPushButton* levelStartButton,
                  QWidget* startWidget, QLabel* startTitle, QLabel* startLabel, QPushButton* startButton,
                  QWidget* gameWidget, QLabel* levelLabel, QLabel* scoreLabel,
                  QStringList levels);

private slots:
    void onLevelCollectionChosen();
    void onStartClicked();
    void onLevelPassed(int score);
    void onLevelFailed();
    void chooseLevelCollection();

private:
    void initiateLevel();
    void readSavedGames();
    void writeSavedGames();
    void readLevelCollections(QStringList collections);

    GameController* gameController; // Not owned
    QWidget* levelWidget; // Not owned
    QListWidget* levelList; // Not owned
    QPushButton* levelStartButton; // Not owned
    QWidget* startWidget; // Not owned
    QLabel* startTitle; // Not owned
    QLabel* startLabel; // Not owned
    QPushButton* startButton; // Not owned
    QWidget* gameWidget; // Not owned
    QLabel* levelLabel; // Not owned
    QLabel* scoreLabel; // Not owned

    QString curColl;
    QStringList levels;
    int level;
    int totalScore;
    QHash<QString, int> savedGames; // level collection -> next level ix
    QHash<QString, QStringList> levelCollections; // level collection -> level names
};

#endif
