// Copyright (C) 2010 Jaakko Kyro <jkyro@korjaussarja.net>
// This file is licenced under GPL, see COPYING
// for full licence information
#include "widget.h"
#include <QPainter>
#include "eveaccount.h"
#include "evesettingsdialog.h"
#include "eveskilltraining.h"
#include "skilltree.h"
#include <QtDebug>
//#include <QNetworkConfigurationManager>
#include <QSettings>
#include <QTimer>
#include <QDateTime>

//QTM_USE_NAMESPACE

Widget::Widget(QWidget *parent)
    : QLabel(parent),
    m_settings(new EveSettingsDialog(this)),
    m_training(NULL),
    //m_net(new QNetworkConfigurationManager(this)),
    m_skills(new SkillTree(this)),
    m_model(new EveModel),
    m_timer(new QTimer(this)),
    m_updateTimer(new QTimer(this)),
    m_pressed(false)
{
    setAlignment(Qt::AlignCenter);
    setAttribute(Qt::WA_TranslucentBackground);
    setGeometry(0,0,150,180);
    //connect(m_net,SIGNAL(onlineStateChanged(bool)),this,SLOT(onlineStateChanged(bool)));
    connect(m_skills,SIGNAL(skillsLoaded()),this,SLOT(update()));
    m_skills->loadSkills();
    m_model->loadSettings();
    m_settings->setModel(m_model);
    EveCharacter *c = m_model->selectedCharacter();
    if (c != NULL) {
        loadTraining();
    }
    m_timer->setSingleShot(true);
    connect(m_timer,SIGNAL(timeout()),this,SLOT(loadTraining()));
    m_updateTimer->setSingleShot(true);
    connect(m_updateTimer,SIGNAL(timeout()),this,SLOT(update()));
}

Widget::~Widget()
{

}


QSize Widget::sizeHint() const
{
    return 2 * QLabel::sizeHint();
}

void Widget::paintEvent(QPaintEvent *event)
{
    QPainter p(this);
    QColor bgColor(0, 0, 0, 128);
    if (m_pressed)
        bgColor.setRgb(101,150,254,128);
    p.setBrush(bgColor);
    p.setPen(Qt::NoPen);
    p.drawRoundedRect(rect(), 25, 25);

    QFont dFont(p.font());
    dFont.setPixelSize(15);
    p.setFont(dFont);
    EveCharacter *character = m_model->selectedCharacter();
    if (character != NULL) {
        QPoint iconLoc((150-64)/2,(150-64)/2);
        if (character->characterIcon) {
            p.drawPixmap(iconLoc,*(character->characterIcon));
        }
        QRect nameLoc(0,10,150,20);
        p.setPen(Qt::white);
        p.drawText(nameLoc,Qt::AlignCenter,character->name);
        if (m_training) {
            if (m_training->isLoading()) {
                QRect skillTitle(0,100,150,50);
                p.drawText(skillTitle,Qt::AlignCenter|Qt::TextWordWrap,
                           "Loading skills...");
            } else if (m_training->training) {
                int timeLeft = m_training->currentTime.secsTo(m_training->lastTraining().endTime);
                qDebug() << "Time left:" << timeLeft;
                if (timeLeft < 60*60*24) {
                    // Queue not full -> yellow text

                    QColor penColor(254,241,53);
                    p.setPen(penColor);
                }

                QRect skillTitle(0,100,150,50);
                p.drawText(skillTitle,Qt::AlignCenter|Qt::TextWordWrap,
                           QString("%1 %2").arg(m_skills->skillName(m_training->firstTraining().typeId))
                           .arg(m_training->firstTraining().level));
                QRect skillLoc(0,145,150,20);
                p.drawText(skillLoc,Qt::AlignCenter,m_training->firstTraining().endTime.toString(Qt::SystemLocaleShortDate));

            } else {

                QRect skillTitle(0,110,150,50);
                p.setPen(Qt::red);
                p.drawText(skillTitle,Qt::AlignCenter|Qt::TextWordWrap,
                           "No skill in training");
            }
        }
    }
    p.end();

    QLabel::paintEvent(event);
}

void Widget::showSettingsDialog()
{

    int result = m_settings->exec();
    if (result == QDialog::Accepted) {

        m_model->setSelected(m_settings->selectedCharacter());
        EveCharacter *character = m_model->selectedCharacter();
        if (character != NULL) {
            qDebug() << "Got character " << character->name;
            connect(character,SIGNAL(imageLoaded()),this,SLOT(skillReady()));
            character->fetchImage();

            loadTraining();
        }
    }
}

void Widget::loadTraining()
{
    EveCharacter *character = m_model->selectedCharacter();

    if (character != NULL) {

        qDebug() << "Fetch skills";
        if (m_training == NULL) {
            m_training = new EveSkillTraining(this);
            connect(m_training,SIGNAL(finished()),this,SLOT(trainingLoaded()));
        }
        m_training->setAccount(m_settings->model().data());
        m_training->setCharacter(character);

        m_training->fetchInfo();
    }
    update();
}

void Widget::skillReady()
{
    m_training->fetchInfo();
    m_model->saveSettings();
    update();
}

void Widget::onlineStateChanged(bool online)
{
    qDebug() << "Online status changed, reloading info";
    if (online) {
        m_training->fetchInfo();
        update();
    }
}

void Widget::trainingLoaded()
{
    qDebug() << "Training loaded";
    if (m_training != NULL && m_training->training) {
        QDateTime limit = (m_training->cachedUntil > m_training->lastTraining().endTime) ?
                          m_training->cachedUntil : m_training->lastTraining().endTime;

        qDebug() << "Next refresh:" << limit.toString(Qt::SystemLocaleShortDate);
        int interval = 1000 * (m_training->currentTime.secsTo(limit) + 15);
        //qDebug() << "Timer interval" << interval;
        if (m_timer->isActive()) {
            m_timer->stop();
        }

        m_timer->setInterval( interval );
        m_timer->start();
        if (m_updateTimer->isActive())
            m_updateTimer->stop();

        QDateTime nextUpdate = m_training->lastTraining().endTime;

        nextUpdate.addSecs(-1*60*60*24);
        if (nextUpdate > m_training->currentTime) {
            m_updateTimer->setInterval(1000 * (m_training->currentTime.secsTo(nextUpdate) + 15));
            m_updateTimer->start();
        }
    }
    update();
}

void Widget::mousePressEvent(QMouseEvent *event)
{
    Q_UNUSED(event);
    qDebug() << "Press event";
    m_pressed = true;
    update();
}

void Widget::mouseReleaseEvent(QMouseEvent *event)
{
    Q_UNUSED(event);
    qDebug() << "Release event";
    m_pressed = false;
    m_training->fetchInfo();
    update();
}
