/* ethos-c-plugin-loader.c
 *
 * Copyright (C) 2009 Christian Hergert <chris@dronelabs.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston MA 
 * 02110-1301 USA
 */

#include <gmodule.h>

#include <ethos/ethos-error.h>
#include <ethos/ethos-plugin-loader.h>
#include <ethos/ethos-plugin-info-private.h>

#include "ethos-c-plugin-loader.h"

struct _EthosCPluginLoaderPrivate
{
	gpointer dummy;
};

static G_CONST_RETURN gchar*
get_name (EthosPluginLoader *plugin_loader)
{
	/* NULL is for default loader */
	return NULL;
}

static void
gc (EthosPluginLoader *plugin_loader)
{
}

static void
initialize (EthosPluginLoader *plugin_loader,
            EthosManager      *manager)
{
}

static void
unload (EthosPluginLoader *plugin_loader)
{
}

static EthosPlugin*
ethos_c_plugin_loader_load (EthosPluginLoader  *plugin_loader,
                            EthosPluginInfo    *plugin_info,
                            GError            **error)
{
	const gchar *name;
	gchar       *plugin_dir;
	gchar       *plugin_path;
	GModule     *module;
	EthosPlugin *plugin = NULL;
	EthosPlugin* (*register_func) (void) = NULL;

	g_return_val_if_fail (ETHOS_IS_PLUGIN_LOADER (plugin_loader), NULL);
	g_return_val_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info), NULL);
	g_return_val_if_fail (g_module_supported (), NULL);

	plugin_dir = g_path_get_dirname (ethos_plugin_info_get_filename (plugin_info));
	name = ethos_plugin_info_get_module (plugin_info);
	plugin_path = g_module_build_path (plugin_dir, name);

	if (!(module = g_module_open (plugin_path, G_MODULE_BIND_LAZY | G_MODULE_BIND_LOCAL)))
		goto cleanup;

	if (!g_module_symbol (module, "ethos_plugin_register",
	                      (gpointer*)&register_func))
	        goto cleanup;

	if (register_func == NULL)
		goto cleanup;

	plugin = register_func ();

cleanup:
	if (!plugin) {
		g_set_error (error, ETHOS_ERROR, ETHOS_ERROR_PLUGIN,
		             "%s: %s", plugin_path, g_module_error ());
		if (module && !g_module_close (module))
			g_warning ("%s: %s", plugin_path, g_module_error ());
	}

	g_free (plugin_path);

	return plugin;
}

static void
ethos_c_plugin_loader_base_init (EthosPluginLoaderIface *iface)
{
	iface->get_name = get_name;
	iface->gc = gc;
	iface->initialize = initialize;
	iface->unload = unload;
	iface->load = ethos_c_plugin_loader_load;
}

G_DEFINE_TYPE_EXTENDED (EthosCPluginLoader,
                        ethos_c_plugin_loader,
                        G_TYPE_OBJECT,
                        0,
                        G_IMPLEMENT_INTERFACE (ETHOS_TYPE_PLUGIN_LOADER,
                                               ethos_c_plugin_loader_base_init));

static void
ethos_c_plugin_loader_finalize (GObject *object)
{
	G_OBJECT_CLASS (ethos_c_plugin_loader_parent_class)->finalize (object);
}

static void
ethos_c_plugin_loader_class_init (EthosCPluginLoaderClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	object_class->finalize = ethos_c_plugin_loader_finalize;

	g_type_class_add_private (object_class, sizeof(EthosCPluginLoaderPrivate));
}

static void
ethos_c_plugin_loader_init (EthosCPluginLoader *plugin_loader)
{
	plugin_loader->priv = G_TYPE_INSTANCE_GET_PRIVATE (plugin_loader,
	                                                   ETHOS_TYPE_C_PLUGIN_LOADER,
	                                                   EthosCPluginLoaderPrivate);
}

EthosPluginLoader*
ethos_c_plugin_loader_new ()
{
	return g_object_new (ETHOS_TYPE_C_PLUGIN_LOADER, NULL);
}

G_MODULE_EXPORT EthosPluginLoader*
ethos_plugin_loader_register (void)
{
	return ethos_c_plugin_loader_new ();
}
