/* ethos-plugin-loader.c
 *
 * Copyright (C) 2009 Christian Hergert <chris@dronelabs.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston MA 
 * 02110-1301 USA
 */

#include "ethos-plugin-loader.h"

/**
 * SECTION:ethos-plugin-loader
 * @title: EthosPluginLoader
 * @short_description: loading of plugins during runtime
 *
 * The #EthosPluginLoader interface is responsible for loading plugins
 * during runtime.  Plugins can specify a plugin-loader that should load
 * them in their plugin desciption file.  This is done using the Loader
 * keyword such as follows.
 *
 * |[
 * [Test Plugin]
 * Name=Sample
 * Loader=python
 * Module=sample
 * IAge=1
 * ]|
 *
 * In this example, the #EthosPythonLoader will be used to load the plugin.
 * The python loader, like all loaders, requires that the Module keyword
 * is set.  In this case, the module is "sample", so the python loader will
 * look for "sample.py" to load the plugin.
 */

static EthosPluginLoader *active_loader = NULL;

GType
ethos_plugin_loader_get_type (void)
{
	static GType g_type = 0;
	if (G_UNLIKELY (!g_type)) {
		const GTypeInfo g_type_info = {
			sizeof (EthosPluginLoaderIface),
			NULL,
			NULL,
			NULL,
			NULL,
			NULL,
			0,
			0,
			NULL
		};
		g_type = g_type_register_static (G_TYPE_INTERFACE,
		                                 "EthosPluginLoader",
		                                 &g_type_info,
		                                 0);
		g_type_interface_add_prerequisite (g_type, G_TYPE_OBJECT);
	}
	return g_type;
}

/**
 * ethos_plugin_loader_get_name:
 * @plugin_loader: An #EthosPluginLoader
 *
 * Retrieves the name of the plugin loader.  The default plugin loader's name
 * is %NULL.
 *
 * Return value: The name of the plugin loader or %NULL
 */
G_CONST_RETURN gchar*
ethos_plugin_loader_get_name (EthosPluginLoader *plugin_loader)
{
	return ETHOS_PLUGIN_LOADER_GET_INTERFACE (plugin_loader)->
		get_name (plugin_loader);
}

/**
 * ethos_plugin_loader_initialize:
 * @plugin_loader: An #EthosPluginLoader
 * @manager: An #EthosManager
 *
 * Initializes the plugin loader.
 */
void
ethos_plugin_loader_initialize (EthosPluginLoader *plugin_loader,
                                EthosManager      *manager)
{
	ETHOS_PLUGIN_LOADER_GET_INTERFACE (plugin_loader)->initialize (plugin_loader, manager);
}

/**
 * ethos_plugin_loader_unload:
 * @plugin_loader: An #EthosPluginLoader
 *
 * Unloads the plugin loader and cleans up related resources.
 */
void
ethos_plugin_loader_unload (EthosPluginLoader *plugin_loader)
{
	ETHOS_PLUGIN_LOADER_GET_INTERFACE (plugin_loader)->unload (plugin_loader);
}

/**
 * ethos_plugin_loader_load:
 * @plugin_loader: An #EthosPluginLoader
 * @plugin_info: An #EthosPluginInfo
 * @error: A location for a #GError or %NULL
 *
 * Creates an instance of the plugin from the information in the
 * @plugin_info object.
 *
 * In case of failure, @error is set and %NULL is returned.
 *
 * Return value: The newly created #EthosPlugin instance or %NULL.
 */
EthosPlugin*
ethos_plugin_loader_load (EthosPluginLoader  *plugin_loader,
                          EthosPluginInfo    *plugin_info,
                          GError            **error)
{
	EthosPlugin *plugin;

	active_loader = plugin_loader;
	plugin = ETHOS_PLUGIN_LOADER_GET_INTERFACE (plugin_loader)->
		load (plugin_loader, plugin_info, error);
	active_loader = NULL;

	return plugin;
}

/**
 * ethos_plugin_loader_gc:
 * @plugin_loader: An #EthosPluginLoader
 *
 * Runs the garbage collector for an associated vm for the plugin loader.
 */
void
ethos_plugin_loader_gc (EthosPluginLoader *plugin_loader)
{
	ETHOS_PLUGIN_LOADER_GET_INTERFACE (plugin_loader)->gc (plugin_loader);
}

/**
 * ethos_plugin_loader_register_plugin:
 * @plugin_loader: An #EthosPluginLoader
 * @plugin: An #EthosPlugin
 *
 * For plugin-loaders that do not support creation of #GObjectClass<!-- -->'s during runtime,
 * this method can be used to instantiate a plugin instance and the plugin-loader will be
 * notified.  For example, a JavaScript plugin may create an instance of #EthosPlugin and connect
 * signals rather than creating its own child class.
 *
 * See ethos_register_plugin().
 */
void
ethos_plugin_loader_register_plugin (EthosPluginLoader *plugin_loader,
                                     EthosPlugin       *plugin)
{
	if (ETHOS_PLUGIN_LOADER_GET_INTERFACE (active_loader)->register_plugin)
		ETHOS_PLUGIN_LOADER_GET_INTERFACE (active_loader)->register_plugin (active_loader,
		                                                                    plugin);
}

/**
 * ethos_register_plugin:
 * @plugin: An #EthosPlugin
 *
 * This is a helper method that can be used by plugins whom were loaded
 * by a loader that does not have access to reading into the plugin's
 * vm.  This will look to see what plugin loader is active and deliver
 * it to the loader.
 *
 * See ethos_plugin_loader_register_plugin().
 */
void
ethos_register_plugin (EthosPlugin *plugin)
{
	g_return_if_fail (ETHOS_IS_PLUGIN (plugin));
	g_return_if_fail (ETHOS_IS_PLUGIN_LOADER (active_loader));
	ethos_plugin_loader_register_plugin (active_loader, plugin);
}
