/*-*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4; tab-width: 4 -*- */
/*
 *    Copyright (C) 2012 Luca Vaudano vaudano@gmail.com
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License along
 *    with this program; if not, write to the Free Software Foundation, Inc.,
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.Warning
 */
 /**
  * It should be able to handle command shell result and gui result
  * 
  * @file estardict-engine-parsing.c
  * @author Luca Vaudano
  */
#include "estardict-engine.h"
#include "estardict-engine-strip.h"
#include "estardict-engine-parsing.h"

// Add image to the elementary entry
const gchar* ADD_IMAGE_TO_ELEMENTARY_ENTRY = "<br><item absize=%dx%d vsize=full href=file://%s/%s/%s></item><br>";

// Image size (N900 800x480 - GTA02 640x480)
const guint IMAGE_SIZE = 450;

// Standard folder name for resources
const gchar* RESOURCE_FOLDER = "res";

// Memory block for realloc
const guint MEMORY_BLOCK = 1024;

// Size of the output string
guint outputSize;


/**
 * Check if it is needed to realloc memory
 * 
 * @param text String text to check
 * @param actualSize Current size of the output string
 * @param additional size of the additional memory needed
 * @return text with realloc memory if needed
 */
gchar* checkForRealloc( gchar* text, guint actualSize, guint additional ) {
    //g_debug("-> %s %s()", __FILE__, __FUNCTION__);
   
    if ( outputSize <= actualSize + additional ) {
        g_message("Realloc during parsing");         
        outputSize += MEMORY_BLOCK;
        text = g_realloc(text, outputSize );
        if ( text == NULL) {
            g_critical("Error re-allocating memory for the %s method", __FUNCTION__);
            estardictEngineError = ESTARDICT_ENGINE_ERROR_MEMORY_ERROR_DEFINITION;
            return NULL;
        }
    }
    return text;
}

/**
 * @brief Really simple tag checker method.
 * No check for invalid tag format, no replace for HTML special
 * characters.
 * Only special handler for the img tag
 *
 * @param starDictInfo Dictionary
 * @param html HTML string
 * @return Text string
 */
gchar* checkTag(StarDictInfo* starDictInfo, gchar* html) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    gint i;
    gint j;
    gint z;
    gint textLenght;
    gboolean insideTag = FALSE;
    gboolean insideTagImage = FALSE;
    gchar c;
    gchar* text = NULL;

    textLenght = strlen(html);
    g_debug("HTML %s", html);

    // We set the inizial size of the output string equals to the html lenght + memory block
    outputSize = textLenght + MEMORY_BLOCK;
    
    text = g_try_malloc( outputSize );
    
    if ( text == NULL) {
        g_critical("Error allocating memory for the %s method", __FUNCTION__);
        estardictEngineError = ESTARDICT_ENGINE_ERROR_MEMORY_ERROR_DEFINITION;
        return NULL;
    }
    
    // Loop over the input string
    for (i = 0, j = 0; i < strlen(html); i++ ) {
        c = html[i];

        if ( c == '<') {
            insideTag = TRUE;
        }

        if ( insideTag ) {
          
            // Consider image tag
            if ( 
                html[i] == '<' 
                && html[i+1] == 'i'
                && html[i+2] == 'm'
                && html[i+3] == 'g'
            ) {
                insideTagImage = TRUE;
            }
            
            // src attribute inside image tag
            if (
                insideTagImage
                && html[i] == ' ' 
                && html[i+1] == 's'
                && html[i+2] == 'r'
                && html[i+3] == 'c'
                && html[i+4] == '='
                && html[i+5] == '"'
            )
            {
                gchar* imageFilename;
                gchar *temp;
                gint tempLenght;
                gchar *endpos;
                gchar buf[500];
                
                // Copy pointer to the html buffer
                temp = html;
                
                // Lenght of the string from the current position to the end
                tempLenght = strlen(temp) - (i+6);
                
                // Find the quote ("). It defines the end of the image filename
                endpos = g_strstr_len(temp+i+6, tempLenght, "\"");
                
                // Lenght of the image filename
                tempLenght = endpos - (temp+i+6);
                
                // File name image
                imageFilename = g_strndup(temp+i+6, tempLenght);
                
                snprintf(
                    buf, 
                    sizeof(buf), 
                    ADD_IMAGE_TO_ELEMENTARY_ENTRY,
                    IMAGE_SIZE,
                    IMAGE_SIZE,
                    starDictInfo->path,
                    RESOURCE_FOLDER,
                    imageFilename
                );
                
                // Realloc
                text = checkForRealloc( text, j, strlen(buf) );
                for (z = 0; z < strlen(buf); z++ ) {
                    text[j] = buf[z];
                    j++;
                }
                
                g_free(imageFilename);
            }

        }


        if ( !insideTagImage ) {
            // Take care also of '\0' char
            text = checkForRealloc( text, j, 2 );
            text[j] = c;
            j++;
        }


        if ( c == '>') {
            insideTag = FALSE;
            insideTagImage = FALSE;
        }
    }
    text[j] = '\0';

    g_debug("TEXT %s", text);

    return text;
}


/**
 * @brief Parse the text from the dict file and returns the formatted
 * definition.
 *
 * @param starDictInfo Dictionary
 * @param dictText Definition text from dict file
 * @return Formatted definition
 */
gchar* parseDefinition(StarDictInfo* starDictInfo, gchar* dictText) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    gchar* result = NULL;
    
    gchar* word = g_strconcat(dictText, NULL);
    gchar* sametypesequence = starDictInfo->sametypesequence;

    // Single sametypesequence pure text
    if (
        sametypesequence != NULL &&
        strlen(sametypesequence) == 1 &&
        (g_ascii_strcasecmp(sametypesequence, "m") == 0 ||
        g_ascii_strcasecmp(sametypesequence, "l") == 0 ||
        g_ascii_strcasecmp(sametypesequence, "t") == 0 ||
        g_ascii_strcasecmp(sametypesequence, "y") == 0)
    ) {
        g_message("---> Sametypesequence pure text");
        result = g_strdup(word);

    // Single sametypesequence HTML and pango
    } else if (
        sametypesequence != NULL &&
        strlen(sametypesequence) == 1 &&
        (
        g_ascii_strcasecmp(sametypesequence, "h") == 0 ||
        g_ascii_strcasecmp(sametypesequence, "g") == 0
        )
    ) {
        g_message("---> Sametypesequence markup text");
        result = checkTag(starDictInfo, word);
    } else {
        estardictEngineError = ESTARDICT_ENGINE_ERROR_PARSING_DEFINITION;
    }

    g_free(word);
    return result;
}

