#include "../engine/estardict-engine.h"
#include <stdlib.h>

#undef G_LOG_DOMAIN
#define G_LOG_DOMAIN "estardict-engine-tests"

#define WORD_DOG "dog"
#define LONGMAN "stardict-longman-2.4.2"
#define BABYLON_EN_IT "stardict-babylon-Babylon_English_Italian-2.4.2"
#define ENGLISH_CZECH "stardict-english-czech"

#define LONGMAN_DEFINITION_DOG_LENGHT 3184
#define BABYLON_EN_IT_DEFINITION_DOG_LENGHT 95
#define ENGLISH_CZECH_DEFINITION_DOG_LENGHT 188

#define ARR_SIZE(x) (sizeof(x)/sizeof((x)[0]))
#define MAX_DICTIONARIES 10

/** Enumeration with use cases */
typedef enum {
    UC_M,
    UC_HTML,
    UC_PANGO,
    UC_THREE_DICTS,
    UC_UNDEFINED,
} EnumUseCase;

/** Menu item structure */
typedef struct DictionaryStruct {
    const char* dictPath;
    const char* word;
    int definitionDogLenght;
} Dictionary;

/** Signature of the test method */
typedef gboolean (*ItemMethodPtr)(void*);

/** Menu item structure */
typedef struct MenuItemStruct {
    EnumUseCase key;
    const char* name;
    ItemMethodPtr method;
    Dictionary dictionary[MAX_DICTIONARIES];
} MenuItem;

/** Test methods */
static gboolean testDict();

/** Use cases */
static const MenuItem tests[] = {
    {
        UC_M, 
        "Longman (samesequence m, 43.052 words)", 
        testDict, 
        {
            {LONGMAN, WORD_DOG, LONGMAN_DEFINITION_DOG_LENGHT},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0}
        }
    },
    {
        UC_HTML, 
        "Babylon English-Italian (samesequence h, 142.077 words, synonym)", 
        testDict, 
        {
            {BABYLON_EN_IT, WORD_DOG, BABYLON_EN_IT_DEFINITION_DOG_LENGHT},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0}
        }
    },
    {
        UC_PANGO, 
        "English-Czech (samesequence g, 86.072 words)", 
        testDict, 
        {
            {ENGLISH_CZECH, WORD_DOG, ENGLISH_CZECH_DEFINITION_DOG_LENGHT},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0}
        }
    },
    {
        UC_THREE_DICTS, 
        "Three dictionaries (longman, Babylon English-Italian, English-Czech)", 
        testDict, 
        {
            {LONGMAN, WORD_DOG, LONGMAN_DEFINITION_DOG_LENGHT},
            {BABYLON_EN_IT, WORD_DOG, BABYLON_EN_IT_DEFINITION_DOG_LENGHT},
            {ENGLISH_CZECH, WORD_DOG, ENGLISH_CZECH_DEFINITION_DOG_LENGHT},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0}
        }
    },
    {
        UC_UNDEFINED, 
        "Invalid test case", 
        NULL, 
        {
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0}
        }
    },
};

/** Dictionary path */
const gchar * path;

/** Selected use case */
MenuItem menuItem;

/**
 * Test the given dictionary
 * 
 * @return TRUE success, FALSE failed
 */
static gboolean testDict() {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);

    gboolean result = FALSE;
    gchar* definition = NULL;
    gchar* absDictPath = NULL;
    StarDictInfo* starDictInfo[MAX_DICTIONARIES];
    int i = 0;
    int numDict = 0;
    
    for ( i = 0; i < MAX_DICTIONARIES; i++) {
        starDictInfo[MAX_DICTIONARIES] = NULL;
    }
    
    if ( path != NULL ) {
        
        // Load dictionaries
        for ( i = 0; i < MAX_DICTIONARIES; i++) {
            if ( menuItem.dictionary[i].dictPath != NULL ) {
                absDictPath = g_strconcat(path, "/", menuItem.dictionary[i].dictPath, NULL);
                starDictInfo[i] = loadDictionary(absDictPath);
            } else {
                break;
            }
        }
        
        numDict = i;
        g_message("Loaded %d dictionary/ies", numDict);
        
        // Get definition
        for ( i = 0; i < numDict; i++) {
            if (starDictInfo[i] != NULL) {
                definition = getDefinition(starDictInfo[i], menuItem.dictionary[i].word);
                g_message("\t Dictionary %s", menuItem.dictionary[i].dictPath);
                g_message("\t type %s", starDictInfo[i]->sametypesequence);
                g_debug("definition for %s:\n%s", menuItem.dictionary[i].word, definition);
                g_debug("lenght definition for %s:\n%d", menuItem.dictionary[i].word, strlen(definition));

                if (
                    strlen(definition) == menuItem.dictionary[i].definitionDogLenght
                ) {
                    g_message("\t OK");
                    result = TRUE;
                    
                // ERROR    
                } else {
                    g_message("\t FAILED - Definition");
                }

            } else {
                g_message("\t FAILED - Load stardict");
            }
        }
        
        // Free
        for ( i = 0; i < numDict; i++) {
            freeData(starDictInfo[i]);
        }
        
    } else {
        g_message("\t FAILED - Path not defined");
    }
    return result;
}

/**
 * Test all the use cases
 */
static char * all_tests() {
    //g_message("TEST1");
    //mu_run_test(testLongman);
    //g_message("TEST2");
    //mu_run_test(testBabylonEnIt);
    //g_message("TEST3");
    //mu_run_test(testLongmanBabylonDicts);
    //g_message("TEST4");
    //mu_run_test(testEnglishCzech);
    //g_message("TEST5");
    //mu_run_test(testDiacritics);
    //g_message("TEST");
    //mu_run_test(failed);
    return 0;
}

void logMessage(const gchar *log_domain, GLogLevelFlags log_level, const gchar *message, gpointer user_data) {

    // Enable message
    if ( log_domain != NULL && strcmp(log_domain, G_LOG_DOMAIN) == 0 && log_level == G_LOG_LEVEL_MESSAGE ) {
        printf("%s\n", message);
    }

    // Enable debug message
    if ( log_domain != NULL && strcmp(log_domain, G_LOG_DOMAIN) == 0 && log_level == G_LOG_LEVEL_DEBUG ) {
        printf("%s\n", message);
    }
}

void disableMessage(const gchar *log_domain, GLogLevelFlags log_level, const gchar *message, gpointer user_data) {
}


/**
 * Print welcome screen with all the use cases
 */
void printUsage(void) {
    g_message("estardict-engine-tests <dictionary_path> <use_case>");
    g_message("\t where");
    g_message("<path with dictionaries> contains test dictionaries");
    g_message("<use_case> could be:");
    int i;
    
    for ( i = 0; i < ARR_SIZE(tests); i++ ) {
        g_message("%d : %s", (int)tests[i].key, tests[i].name );
    }
}

/**
 * Lookup use case from key value
 * 
 * @param key Use case key
 * @return Use case menu item
 */
MenuItem lookup(EnumUseCase key) {
    int i;
    MenuItem result;
    
    for ( i = 0; i < ARR_SIZE(tests); i++ ) {
        if ( tests[i].key == key ) {
            result = tests[i];
            break;
        }
    }
    
    return result;
}


/**
 * Main function
 */
int main(int argc, char **argv) {
    int i = 0;
    gboolean result = FALSE;
    
    g_log_set_handler(G_LOG_DOMAIN, G_LOG_LEVEL_MESSAGE | G_LOG_LEVEL_DEBUG, logMessage, NULL);
    g_log_set_handler(NULL, G_LOG_LEVEL_MESSAGE | G_LOG_LEVEL_DEBUG, disableMessage, NULL);
    
    /* argc 0 command name
     * argc 1 dictionary path
     * argc 2 test case
     */
    if ( argc <= 2 ) {
        printUsage();
        return 0;
    }
    path = argv[1];

    for ( i = 2; i < argc; i++ ) {
        int index = atoi( argv[i] );
        
        if ( index < ARR_SIZE( tests) ) {
            menuItem = lookup(index);
            g_message("Testing: %s", menuItem.name);
            
            ItemMethodPtr itemMethod = menuItem.method;
            if ( itemMethod ) {
                result = itemMethod(NULL);
                if ( result ) {
                    g_message("success");
                } else {
                    g_message("failed");
                }
            } else {
                g_message("Method not implemented!");
            }
        } else {
            g_message("Use case not defined!");
            printUsage();
        }
    }
    
    return 0;
}
