/*-*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4; tab-width: 4 -*- */
/*
 *    Copyright (C) 2009 Luca Vaudano vaudano@gmail.com
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License along
 *    with this program; if not, write to the Free Software Foundation, Inc.,
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.Warning
 */
 /**
  * @file estardict-gui-dictionariesPage.c
  * @author Luca Vaudano
  */
#include "estardict-gui-dictionariesPage.h"
#include "estardict-gui-logic.h"
#include "estardict-gui.h"
#include "estardict-config.h"
#include "estardict-gui-utils.h"
#include "i18n.h"

Evas_Object *genList;
static Elm_Genlist_Item_Class itc;
Eina_List *dictionariesList;

/**
 * @brief Remove dictionary from genList
 *
 * @param data genList
 * @param obj
 * @param event_info
 */
static void removeDictionary(void *data, Evas_Object *obj, void *event_info) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);

    Evas_Object *genList = data;
    Elm_Object_Item *gli = elm_genlist_selected_item_get(genList);
    if (!gli) {
        g_message("Item not selected.");
        return;
    }

    gchar* dictionaryToRemove = (gchar *)elm_object_item_data_get(gli);

    // Remove from dictionary list
    const Eina_List *l;
    gchar *dictionaryPath;
    EINA_LIST_FOREACH(dictionariesList, l, dictionaryPath) {
        if ( strcmp(dictionaryToRemove, dictionaryPath) == 0 ) {
            dictionariesList = eina_list_remove(dictionariesList, dictionaryPath );
            g_debug("Item %s removed", dictionaryPath);
        }
    }

    // Remove from genList
    elm_object_item_del(gli);
}


/**
 * @brief Get dictionary list
 *
 * @return data Dictionary list
 */
Eina_List* getDictionaryList() {
    return eina_list_clone(dictionariesList);
}


/**
 * @brief Label for single item in the genList
 *
 * @param data Dictionary path
 * @param obj
 * @param part
 * @return Dictionary name
 */
char *getTextList(void *data, Evas_Object *obj, const char *part) {
    char buf[256];
    gchar* dictName = getNameFromPath( (gchar *)data );
    snprintf(buf, sizeof(buf), "%s", truncateWord(dictName));
    return strdup(buf);
}


/**
 * @brief Icon for single item in the genList
 *
 * @param data genList
 * @param obj
 * @param part
 * @return No icon
 */
Evas_Object *gl_content_get(void *data, Evas_Object *obj, const char *part) {
    return NULL;
}


/**
 * @brief State for single item in the genList
 *
 * @param data genList
 * @param obj
 * @param part
 * @return No state
 */
Eina_Bool gl_state_get(void *data, Evas_Object *obj, const char *part) {
   return EINA_FALSE;
}


/**
 * @brief Deletion for single item in the genList
 *
 * @param data genList
 * @param obj
 * @param part
 */
void gl_del(void *data, Evas_Object *obj) {}


/**
 * @brief Create dictionary frame
 */
void createDictionaryFrame() {
    Evas_Object *ic, *bt, *bx2;
    char buf[PATH_MAX];

    genList = elm_genlist_add(win);
    evas_object_size_hint_align_set(genList, EVAS_HINT_FILL, EVAS_HINT_FILL);
    evas_object_size_hint_weight_set(genList, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
    evas_object_show(genList);

    itc.item_style = "default";
    itc.func.text_get = getTextList;
    itc.func.content_get  = gl_content_get;
    itc.func.state_get = gl_state_get;
    itc.func.del = gl_del;

    elm_box_pack_end(dictionariesBox, genList);

    bx2 = elm_box_add(win);
    elm_box_horizontal_set(bx2, EINA_TRUE);
    evas_object_size_hint_weight_set(bx2, EVAS_HINT_EXPAND, 0.0);
    evas_object_size_hint_align_set(bx2, EVAS_HINT_FILL, EVAS_HINT_FILL);

    ic = elm_icon_add(win);
    snprintf(buf, sizeof(buf), ADD_ICON, ROOT_DIR);
    elm_icon_file_set(ic, buf, NULL);
    evas_object_size_hint_aspect_set(ic, EVAS_ASPECT_CONTROL_VERTICAL, 1, 1);
    bt = elm_button_add(win);
    elm_object_text_set(bt, ADD_BUTTON_LABEL);
    elm_object_part_content_set(bt, "icon", ic);
    evas_object_smart_callback_add(bt, "clicked", addDictionary, NULL);
    evas_object_size_hint_align_set(bt, EVAS_HINT_FILL, EVAS_HINT_FILL);
    evas_object_size_hint_weight_set(bt, EVAS_HINT_EXPAND, 0.0);
    elm_box_pack_end(bx2, bt);
    evas_object_show(bt);

    ic = elm_icon_add(win);
    snprintf(buf, sizeof(buf), REMOVE_ICON, ROOT_DIR);
    elm_icon_file_set(ic, buf, NULL);
    evas_object_size_hint_aspect_set(ic, EVAS_ASPECT_CONTROL_VERTICAL, 1, 1);
    bt = elm_button_add(win);
    elm_object_text_set(bt, REMOVE_BUTTON_LABEL);
    elm_object_part_content_set(bt, "icon", ic);
    evas_object_smart_callback_add(bt, "clicked", removeDictionary, genList);
    evas_object_size_hint_align_set(bt, EVAS_HINT_FILL, EVAS_HINT_FILL);
    evas_object_size_hint_weight_set(bt, EVAS_HINT_EXPAND, 0.0);
    elm_box_pack_end(bx2, bt);
    evas_object_show(bt);

    elm_box_pack_end(dictionariesBox, bx2);
    evas_object_show(bx2);
}


/**
 * @brief Create dictionaries page, if needed
 */
void createDictionariesPage() {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    Evas_Object *ic, *bt, *frameLabel;
    char buf[PATH_MAX];

    // Box Window
    dictionariesBox = elm_box_add(win);
    elm_box_horizontal_set(dictionariesBox, EINA_FALSE);
    evas_object_size_hint_weight_set(dictionariesBox, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
    evas_object_size_hint_align_set(dictionariesBox, EVAS_HINT_FILL, EVAS_HINT_FILL);
    elm_win_resize_object_add(win, dictionariesBox);

    // Label
    frameLabel = elm_label_add(win);
    elm_label_line_wrap_set(frameLabel, EINA_FALSE);
    elm_object_text_set(frameLabel, DICTIONARIES_FRAME_LABEL);
    elm_object_style_set(frameLabel, "title");
    evas_object_size_hint_align_set(frameLabel, EVAS_HINT_FILL, EVAS_HINT_FILL);
    evas_object_show(frameLabel);
    elm_box_pack_end(dictionariesBox, frameLabel);

    // Create dictionary frame
    createDictionaryFrame();

    // Save button
    bt = elm_button_add(win);
    elm_object_text_set(bt, SAVE_BUTTON_LABEL);
    evas_object_size_hint_align_set(bt, EVAS_HINT_FILL, 0);
    elm_box_pack_end(dictionariesBox, bt);
    evas_object_show(bt);
    evas_object_smart_callback_add(bt, "clicked", saveDictionaries, NULL );

    // Quit button
    ic = elm_icon_add(win);
    snprintf(buf, sizeof(buf), QUIT_ICON, ROOT_DIR);
    elm_icon_file_set(ic, buf, NULL);
    evas_object_size_hint_aspect_set(ic, EVAS_ASPECT_CONTROL_VERTICAL, 1, 1);
    bt = elm_button_add(win);
    elm_object_part_content_set(bt, "icon", ic);
    elm_object_text_set(bt, QUIT_BUTTON_LABEL);
    evas_object_size_hint_align_set(bt, EVAS_HINT_FILL, 0);
    elm_box_pack_end(dictionariesBox, bt);
    evas_object_show(bt);
    evas_object_show(ic);
    evas_object_smart_callback_add(bt, "clicked", quitDictionaries, NULL);
}

/**
 * @brief Clear dictionary list
 */
void clearDictionaryList() {
  g_debug("-> %s %s()", __FILE__, __FUNCTION__);
  elm_genlist_clear(genList);
  dictionariesList = eina_list_free(dictionariesList);
}


/**
 * @brief Add dictionaries, defined in the configuration file, in the genList
 * if the dictionary list is not initialized
 *
 * @param dictionaries List of dictionary
 */
void addDictionaries(Eina_List* dictionaries) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    const Eina_List* l;
    DictionaryProperties *dictionary;

    // If genList is empty
    if ( elm_genlist_first_item_get(genList) == NULL ) {
        EINA_LIST_FOREACH(dictionaries, l, dictionary) {
            if ( dictionary != NULL && 
                dictionary->dictionaryPath != NULL && 
                strlen(dictionary->dictionaryPath) != 0
            ) {
                addDictionaryToGenlist(dictionary->dictionaryPath);
            }
        }
    }
}


/**
 * @brief Add the selected dictionary in the genList
 *
 * @param selected Selected dictionary file
 */
void addDictionaryToGenlist(const gchar *selected) {
    Elm_Object_Item *gli;

    if ( selected != NULL )  {
        itc.item_style = "default";
        itc.func.text_get = getTextList;
        itc.func.content_get  = gl_content_get;
        itc.func.state_get = gl_state_get;
        itc.func.del = gl_del;

        gli = elm_genlist_item_append(
                genList,
                &itc,
                g_strdup(selected), // item data
                NULL, // parent
                ELM_GENLIST_ITEM_NONE,
                NULL, // select function
                NULL // function data
            );
        dictionariesList = eina_list_append( dictionariesList, g_strdup(selected) );
    }
}
