/*
  EightyxOne - A simple Sudoku solving game
  Copyright (C) 2006  Tim Teulings

  The source code of this module is based on java code by
  Stan Chesnutt.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "Dancing.h"

Node::Node()
{
 // no code
}

Node::Node(int rowNumber, int label, Column *column)
: rowNumber(rowNumber),
  label(label)
{
  setLeft(this);
  setRight(this);
  setUp(this);
  setDown(this);
  setColumn(column);
}

Node::~Node()
{
  // no code
}

Column::Column()
{
  // no code
}

Column::Column(int label, bool optional)
: Node(0,label,NULL),
  size(0),
  optional(optional)
{
  setColumn(this);
}

/**
 * Add a node (representing a row that contains a 1 in this
 * column) to the end (or bottom) of this column.  Columns grow
 * downward as rows are added.
 */
void Column::addAtEnd(Node *node)
{
  Node *end=getUp();

  node->setUp(end);
  node->setDown(this);
  end->setDown(node);
  this->setUp(node);
  ++size;
}

DancingLinksArena::DancingLinksArena()
{
  // no code
}

DancingLinksArena::~DancingLinksArena()
{
  for (std::list<Column*>::iterator iter=columnUsed.begin(); iter!=columnUsed.end(); ++iter) {
    delete *iter;
  }

  for (std::list<Node*>::iterator iter=nodeUsed.begin(); iter!=nodeUsed.end(); ++iter) {
    delete *iter;
  }
}

void DancingLinksArena::assignLabels(int *labels, bool *optional, size_t length)
{
  std::vector<Column*> columns(length,NULL);

  for (size_t i=0; i<length; ++i) {
    columns[i]=newColumn(labels[i],optional[i]);
    columns[i]->setRight(NULL);

    if (i>0) {
      columns[i]->setLeft(columns[i-1]);
      columns[i-1]->setRight(columns[i]);
    }
  }

  firstColumn=newColumn(0,false);
  columns[0]->setLeft(firstColumn);
  firstColumn->setRight(columns[0]);
  columns[length-1]->setRight(firstColumn);
  firstColumn->setLeft(columns[length-1]);
  solution.resize(length,NULL),
  solutionIndex=0;

  rowCount=0;
  startingCount=0;
  traveller=NULL;
}

void DancingLinksArena::assignLabels(int *labels, size_t length)
{
  std::vector<Column*> columns(length,NULL);

  for (size_t i=0; i<length; ++i) {
    columns[i]=newColumn(labels[i],false);
    columns[i]->setRight(NULL);

    if (i>0) {
      columns[i]->setLeft(columns[i-1]);
      columns[i-1]->setRight(columns[i]);
    }
  }

  firstColumn=newColumn(0,false);
  columns[0]->setLeft(firstColumn);
  firstColumn->setRight(columns[0]);
  columns[length-1]->setRight(firstColumn);
  firstColumn->setLeft(columns[length-1]);
  solution.resize(length,NULL),
  solutionIndex=0;

  rowCount=0;
  startingCount=0;
  traveller=NULL;
}

Node* DancingLinksArena::addInitialRow(int *labels, size_t length)
{
  Node *result=NULL;

  if (length!=0) {
    Node *prev=NULL;
    Node *first=NULL;

    rowCount++;

    for (size_t i=0; i<length; i++) {
      Node       *node;
      Column *searcher;
      Column *theColumn=NULL;

      /*
       * slow search for column
       */
      searcher=firstColumn;

      do {
        if (searcher->getLabel()==labels[i]) {
          theColumn=searcher;
        }

        searcher=dynamic_cast<Column*>(searcher->getRight());
      } while (searcher!=firstColumn && theColumn==NULL);

      assert(theColumn!=NULL);
      node=newNode(rowCount,labels[i],theColumn);
      theColumn->addAtEnd(node);

      node->setLeft(prev);
      node->setRight(NULL);

      if (prev!=NULL) {
        prev->setRight(node);
      }
      else {
        first=node;
      }

      prev=node;
    }

    /*
     * "prev" now points to the last node.  "first" points
     * to the first.  Complete the circular list
     */
    first->setLeft(prev);
    prev->setRight(first);
    result=first;
  }

  return result;
}

void DancingLinksArena::snapshot()
{
  columnSnapshot.clear();
  for (std::list<Column*>::iterator iter=columnUsed.begin(); iter!=columnUsed.end(); ++iter) {
    ColumnSnapshot snap;

    snap.pointer=*iter;
    snap.value=*(*iter);

    columnSnapshot.push_front(snap);
  }

  nodeSnapshot.clear();
  for (std::list<Node*>::iterator iter=nodeUsed.begin(); iter!=nodeUsed.end(); ++iter) {
    NodeSnapshot snap;

    snap.pointer=*iter;
    snap.value=*(*iter);

    nodeSnapshot.push_front(snap);
  }

  firstColumnSnapshot=firstColumn;
}

void DancingLinksArena::restore()
{
  for (std::list<ColumnSnapshot>::iterator iter=columnSnapshot.begin(); iter!=columnSnapshot.end(); ++iter) {
    (*iter->pointer)=iter->value;
  }

  for (std::list<NodeSnapshot>::iterator iter=nodeSnapshot.begin(); iter!=nodeSnapshot.end(); ++iter) {
    (*iter->pointer)=iter->value;
  }

  firstColumn=firstColumnSnapshot;

  for (size_t i=0; i<solution.size(); i++) {
    solution[i]=NULL;
  }
  solutionIndex=0;

  rowCount=0;
  startingCount=0;
  traveller=NULL;
}

Column* DancingLinksArena::newColumn(int label, bool optional)
{
  Column *res=new Column(label,optional);

  columnUsed.push_front(res);

  return res;
}

Node* DancingLinksArena::newNode(int rowNumber, int label, Column *column)
{
  Node *res=new Node(rowNumber,label,column);

  nodeUsed.push_front(res);

  return res;
}

void DancingLinksArena::removeColumn(Column *columnHead)
{
  Node *scanner=columnHead->getDown();

  /*
   * Unsnap the elements of each row in the column
   */
  while (scanner!=columnHead) {
    Node *rowTraveller=scanner->getRight();

    /*
     * remove this row
     */
    while (rowTraveller!=scanner) {
      rowTraveller->getUp()->setDown(rowTraveller->getDown());
      rowTraveller->getDown()->setUp(rowTraveller->getUp());
      rowTraveller->getColumn()->decrementSize();
      rowTraveller=rowTraveller->getRight();
    }

    scanner=scanner->getDown();
  }

  /*
   * Now remove the column
   */
  columnHead->getLeft()->setRight(columnHead->getRight());
  columnHead->getRight()->setLeft(columnHead->getLeft());
}

void DancingLinksArena::reinsertColumn(Column *columnNode)
{
  Node *scanner=columnNode->getUp();

  /*
   * Iterate through the rows
   */
  while (scanner!=columnNode) {
    Node *rowTraveller=scanner->getLeft();

    while (rowTraveller!=scanner) {
      rowTraveller->getUp()->setDown(rowTraveller);
      rowTraveller->getDown()->setUp(rowTraveller);
      rowTraveller->getColumn()->incrementSize();
      rowTraveller=rowTraveller->getLeft();
    }

    scanner=scanner->getUp();
  }

  /*
   * put the column back in the column-header list
   */
  columnNode->getLeft()->setRight(columnNode);
  columnNode->getRight()->setLeft(columnNode);
}

/**
 * Remove the row.  For each element in the row, traverse the
 * corresponding column and remove the rows in that column.  A
 * side-effect of removing the first column is that this row will
 * be removed.
 */
void DancingLinksArena::removeRow(Node *rowHead)
{
  Node *scanner=rowHead;

  do {
    Node *next=scanner->getRight();

    removeColumn(scanner->getColumn());
    scanner=next;
  } while (scanner!=rowHead);
}

void DancingLinksArena::reinsertRow(Node *rowHead)
{
  Node *scanner=rowHead;

  do {
    scanner=scanner->getLeft();
    reinsertColumn(scanner->getColumn());
  } while (scanner!=rowHead);
}

/**
 * Return the next non-optional column, when we need to move
 * deeper (to the right) into the matrix.  Optional columns are
 * used in certain situations (i.e. the Eight Queens) when certain
 * constraints need to be fulfilled once or not at all.
 */
Column* DancingLinksArena::getNextNonOptionalColumn(Node *node)
{
  /*
   * Get the column that the specified node resides within
   */
  Column *nextColumn=node->getColumn();

  /*
   * Move to the right, until the first non-optional column is
   * found
   */
  do {
    nextColumn=nextColumn->getRight()->getColumn();
  } while (nextColumn->getCount()==0 && nextColumn->isOptional());

  return nextColumn;
}

/**
 * Traverse the solution space.  The instance variable "traveller"
 * moves (depth-first) from column to column.  After the traveller
 * has moved through each columns to the right of the current
 * column, it moves down one row in the current column and then
 * moves to the right.
 */
void DancingLinksArena::solveNonRecurse(std::vector<int>& result)
{
  /**
   * Move the traveller until a solution is found, or until no
   * more traversals can be made.  Once a solution is found,
   * return it as an integer array.  If a solution can't be
   * found, return null.
   */
  result.clear();
  while (result.size()==0) {
    Column *thisColumn=traveller->getColumn();

    /*
     * First, check to see if we've reached the last column or if
     * our traveller has moved past the last row in the current
     * column.
     *
     * Columns are linked in a ring, so the column that
     * follows the last column is the first column.
     */
    if (thisColumn==firstColumn || thisColumn==traveller) {
      /*
       * We've moved beyond the last column (while going
       * "rightwards"), or we've moved beyond the last row (via
       * "downwards" traversal of the rows).
       *
       * If we've moved beyond the right column, then the
       * entire column space has been traversed, and a
       * solution has been found.
       */
      if (thisColumn==firstColumn) {
        /*
         * create the new solution value
         */
        result.resize(solutionIndex);

        for (size_t i=0; i<solutionIndex; i++) {
          result[i]=solution[i]->getRowNumber()-1;
        }
      }

      /*
       * Move to the left, and start traversing down the
       * rows of the column immediately to the left.  This
       * is a "pop" operation.  If there are no more
       * columns, then we've reached the starting point, and
       * it is time to return.
       */
      if (solutionIndex==startingCount) {
        return;
      }
      else {
        /*
         * here is the "pop"
         */
        traveller=solution[--solutionIndex];
        reinsertRow(traveller);
        traveller=traveller->getDown();
      }
    }
    else {
      /*
       * We haven't reached the rightmost column, so go
       * deeper (to the right).  This is a "push" operation.
       */
      removeRow(traveller);
      solution[solutionIndex++]=traveller;
      traveller=getNextNonOptionalColumn(firstColumn)->getDown();
    }
  }
}

/**
 * Perform "Algorithm X" on this sparse boolean matrix
 */
void DancingLinksArena::solve(std::vector<int>& solution)
{
  if (traveller==NULL) {
    traveller=getNextNonOptionalColumn(firstColumn)->getDown();
    startingCount=solutionIndex;
  }

  solveNonRecurse(solution);
}

/**
 * Used to remove portions of the possible solution space which
 * are already "known" to be part of the solution.  The provided
 * array lists all such rows.
 */

void DancingLinksArena::removeInitialSolutionSet(const std::list<Node*>& solutions)
{
  for (std::list<Node*>::const_iterator iter=solutions.begin(); iter!=solutions.end(); ++iter) {
    removeRow(*iter);
    solution[solutionIndex++]=*iter;
  }
}

DancingLinksSudoku::DancingLinksSudoku()
{
  /*
   * The data row for the Sudoku sparse matrix looks like:
   *  <1-81>: the cell position in question: (row * 9 + column)
   *        <1-81>: the row constraint (row# * 9 + digit)
   *              <1-81>: the column constraint (column# * 9 + digit)
   *                    <1-81>: the 3x3 box constraint (box# * 9 + digit)
   *
   * yielding a grand total of 324 columns
   */

  for (size_t i=0; i<324; i++) {
    labels[i]=i+1;
  }

  dla.assignLabels(labels,324);

  for (size_t row=0; row<9; row++) {
    for (size_t column=0; column<9; column++) {
      for (size_t digit=0; digit<9; digit++) {
        int  boxrow, boxcol;

        /*
         * Compute the four constraint column numbers
         */
        rowData[row][column][digit][0]=1+(row*9+column);
        rowData[row][column][digit][1]=1+81+(row*9+digit);
        rowData[row][column][digit][2]=1+81+81+(column*9+digit);
        boxrow=row/3;
        boxcol=column/3;
        rowData[row][column][digit][3]=1+81+81+81+((boxrow*3+boxcol)*9+digit);
      }
    }
  }

  for (size_t row=0; row<9; row++) {
    for (size_t column=0; column<9; column++) {
      for (size_t digit=0; digit<9; digit++) {
        /*
         * Add this row to the sparse matrix
         */
        rows[row][column][digit]=dla.addInitialRow(rowData[row][column][digit],4);
      }
    }
  }

  dla.snapshot();
}

bool DancingLinksSudoku::handleSolution(const std::vector<int>& rowIndex, std::vector<int>& solution)
{
  if (rowIndex.size()!=0) {
    solution.resize(9*9);

    for (size_t i=0; i<81; i++) {
      int value=rowIndex[i];
      int digit,row,column;

      digit=value%9;
      value=value/9;
      column=value%9;
      value=value/9;
      row=value%9;

      solution[row*9+column]=digit+1;
    }

    return true;
  }
  else {
    return false;
  }
}

void DancingLinksSudoku::assignRiddle(size_t* puzzle)
{
  std::list<Node*> givenList;

  dla.restore();

  for (size_t row=0; row<9; row++) {
    for (size_t column=0; column<9; column++) {
      for (size_t digit=0; digit<9; digit++) {
        /*
         * See if the square is already filled with the
         * digit of interest.  If so, the row that
         * describes this value will be removed later
         * (also removing any other interfering rows).
         * And, if it is one of the "givens", add it to our
         * collection of constraints.
         */
        if (puzzle[row*9+column]==(digit+1)) {
          givenList.push_back(rows[row][column][digit]);
        }
      }
    }
  }

  /*
   * Remove all of the "givens".  They are removed after all of
   * the entries are added, so that the interfering rows can be
   * removed as well.
   */
  dla.removeInitialSolutionSet(givenList);
}

void DancingLinksSudoku::assignRiddle(const std::vector<size_t>& puzzle)
{
  std::list<Node*> givenList;

  dla.restore();

  for (size_t row=0; row<9; row++) {
    for (size_t column=0; column<9; column++) {
      for (size_t digit=0; digit<9; digit++) {
        /*
         * See if the square is already filled with the
         * digit of interest.  If so, the row that
         * describes this value will be removed later
         * (also removing any other interfering rows).
         * And, if it is one of the "givens", add it to our
         * collection of constraints.
         */
        if (puzzle[row*9+column]==(digit+1)) {
          givenList.push_back(rows[row][column][digit]);
        }
      }
    }
  }

  /*
   * Remove all of the "givens".  They are removed after all of
   * the entries are added, so that the interfering rows can be
   * removed as well.
   */
  dla.removeInitialSolutionSet(givenList);
}

bool DancingLinksSudoku::solveit(std::vector<int>& solution)
{
  std::vector<int> tmp;

  dla.solve(tmp);
  return handleSolution(tmp,solution);
}

