/*
 * This file is part of maptile-loader
 * Copyright (C) 2007  Pekka Rönkkö (pronkko@gmail.com)
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */


#include <config.h>

#include <string.h>
#include <unistd.h>

#include <gmodule.h>

#include "maptile-types.h"

#include "libmaptileloader/maptile-module.h"

#include "maptile-module-manager.h"


enum
{
  PROP_0,
  PROP_MODULE_PATH
};


static GObject * maptile_module_manager_constructor   (GType             type,
                                                   guint             n_params,
                                                   GObjectConstructParam *params);
static void      maptile_module_manager_finalize      (GObject          *object);
static void      maptile_module_manager_get_property  (GObject          *object,
                                                   guint             param_id,
                                                   GValue           *value,
                                                   GParamSpec       *pspec);
static void      maptile_module_manager_set_property  (GObject          *object,
                                                   guint             param_id,
                                                   const GValue     *value,
                                                   GParamSpec       *pspec);
static gboolean  maptile_module_manager_query_modules (MaptileModuleManager *manager);


G_DEFINE_TYPE (MaptileModuleManager, maptile_module_manager, G_TYPE_OBJECT);


static void
maptile_module_manager_class_init (MaptileModuleManagerClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  object_class->constructor  = maptile_module_manager_constructor;
  object_class->finalize     = maptile_module_manager_finalize;
  object_class->get_property = maptile_module_manager_get_property;
  object_class->set_property = maptile_module_manager_set_property;

  g_object_class_install_property (object_class, PROP_MODULE_PATH,
                                   g_param_spec_string ("module-path",
                                                        "Module Path",
                                                        "The path where to look for modules",
                                                        NULL,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_CONSTRUCT_ONLY));
}

static void
maptile_module_manager_init (MaptileModuleManager *manager)
{
  manager->module_path = NULL;
  manager->modules     = NULL;
}

static GObject *
maptile_module_manager_constructor (GType                  type,
                                guint                  n_params,
                                GObjectConstructParam *params)
{
  GObject          *object;
  MaptileModuleManager *manager;

  object =
    G_OBJECT_CLASS (maptile_module_manager_parent_class)->constructor (type,
                                                                   n_params,
                                                                   params);

  manager = MAPTILE_MODULE_MANAGER (object);

  if (manager->module_path)
    maptile_module_manager_query_modules (manager);

  return object;
}

static void
maptile_module_manager_finalize (GObject *object)
{
  MaptileModuleManager *manager = MAPTILE_MODULE_MANAGER (object);

  g_free (manager->module_path);

  /* GTypeModules must not be finalized, don't unref them */
  g_list_free (manager->modules);

  G_OBJECT_CLASS (maptile_module_manager_parent_class)->finalize (object);
}

static void
maptile_module_manager_get_property (GObject    *object,
                                 guint       param_id,
                                 GValue     *value,
                                 GParamSpec *pspec)
{
  MaptileModuleManager *manager = MAPTILE_MODULE_MANAGER (object);

  switch (param_id)
    {
    case PROP_MODULE_PATH:
      g_value_set_string (value, manager->module_path);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
      break;
    }
}

static void
maptile_module_manager_set_property (GObject      *object,
                                 guint         param_id,
                                 const GValue *value,
                                 GParamSpec   *pspec)
{
  MaptileModuleManager *manager = MAPTILE_MODULE_MANAGER (object);

  switch (param_id)
    {
    case PROP_MODULE_PATH:
      manager->module_path = g_value_dup_string (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
      break;
    }
}

static gboolean
maptile_module_manager_is_valid_module_name (const gchar *name)
{
  return g_str_has_suffix (name, ".so");
}

static gboolean
maptile_module_manager_query_modules (MaptileModuleManager *manager)
{
  const gchar *name;
  GDir        *dir;
  GError      *error = NULL;

  dir = g_dir_open (manager->module_path, 0, &error);

  if (!dir)
    {
      g_printerr ("Error while opening module dir: %s\n", error->message);
      g_clear_error (&error);
      return FALSE;
    }

  while ((name = g_dir_read_name (dir)))
    {
      if (maptile_module_manager_is_valid_module_name (name))
        {
          MaptileModule *module;
          gchar     *path;

          path = g_build_filename (manager->module_path, name, NULL);
          module = maptile_module_new (path);

          if (! g_type_module_use (G_TYPE_MODULE (module)))
            {
              g_printerr ("Failed to load module: %s\n", path);
              g_object_unref (module);
              g_free (path);
              continue;
            }

          g_free (path);

          g_type_module_unuse (G_TYPE_MODULE (module));

          manager->modules = g_list_prepend (manager->modules, module);
        }
    }

  g_dir_close (dir);

  return TRUE;
}

