/*
 * This file is part of maptile-loader
 * Copyright (C) 2007  Pekka Rönkkö (pronkko@gmail.com)
 * Copyright (C) 2008  Jukka Alasalmi (jualasal@mail.student.oulu.fi)
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#define _GNU_SOURCE

#include <config.h>
#include <time.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <locale.h>
#include <math.h>

#include <glib.h>
#include <glib/gstdio.h>
#include <libgnomevfs/gnome-vfs.h>

#include <curl/curl.h>

#include "libmaptileloader/maptile-repo.h"
#include "libmaptileloader/maptile-module.h"

#define MAPTILE_TYPE_MAPPER_REPO     (maptile_mapper_repo_type)
#define MAPTILE_MAPPER_REPO(obj)     (G_TYPE_CHECK_INSTANCE_CAST ((obj), \
                                       MAPTILE_TYPE_MAPPER_REPO, \
                                       MaptileMapperRepo))
#define MAPTILE_MAPPER_REPO_CLASS(k) (G_TYPE_CHECK_CLASS_CAST((k), \
                                       MAPTILE_TYPE_MAPPER_REPO, \
                                       MaptileMapperRepoClass))
#define MAPTILE_IS_MAPPER_REPO(obj)  (G_TYPE_CHECK_INSTANCE_TYPE ((obj), \
                                       MAPTILE_TYPE_MAPPER_REPO))

/* Definitions below from Maemo Mapper, needed for URL conversions */
#define BUFFER_SIZE (2048)
#define MAX_ZOOM 16

#define PI   (3.14159265358979323846f)

#define MERCATOR_SPAN (-6.28318377773622f)
#define MERCATOR_TOP (3.14159188886811f)

#define WORLD_SIZE_UNITS (2 << (MAX_ZOOM + TILE_SIZE_P2))
#define TILE_SIZE_PIXELS (256)
#define TILE_SIZE_P2 (8)

#define tile2zunit(tile, zoom) ((tile) << (8 + zoom))
#define pixel2zunit(pixel, zoom) ((pixel) << (zoom))

#define unit2latlon(unitx, unity, lat, lon) { \
    (lon) = ((unitx) * (360.f / WORLD_SIZE_UNITS)) - 180.f; \
    (lat) = (360.f * (atanf(expf(((unity) \
                                  * (MERCATOR_SPAN / WORLD_SIZE_UNITS)) \
                     + MERCATOR_TOP)))) * (1.f / PI) - 90.f; \
}



typedef struct _MaptileMapperRepo       MaptileMapperRepo;
typedef struct _MaptileMapperRepoClass  MaptileMapperRepoClass;

struct _MaptileMapperRepo
{
  MaptileRepo parent_instance;

  CURL *easy_handle;
};

struct _MaptileMapperRepoClass
{
  MaptileRepoClass parent_class;
};


static GType maptile_mapper_repo_get_type   (GTypeModule             *module);
static void  maptile_mapper_repo_class_init (MaptileMapperRepoClass *class);
static void  maptile_mapper_repo_init       (MaptileMapperRepo      *filter);

static void  maptile_mapper_repo_get_tile     (MaptileRepo   *repo,
                                               guint         tilex,
                                               guint         tiley,
                                               guint         zoom,
                                               gchar         **tile,
					       ConICCallback check_conic_cb);

static void maptile_mapper_convert_coords_to_quadtree_string(gint x,
                                                             gint y,
                                                             gint zoomlevel,
                                                             gchar *buffer,
                                                             const gchar initial,
                                                             const gchar *const quadrant);

static gchar* maptile_mapper_convert_wms_to_wms(gint tilex,
                                                gint tiley,
                                                gint zoomlevel,
                                                gchar* uri);

static gchar* maptile_mapper_construct_url( gchar *uri,
                                            guint tilex,
                                            guint tiley,
                                            guint zoom);


static GType           maptile_mapper_repo_type         = 0;
static MaptileRepoClass *maptile_mapper_repo_parent_class = NULL;


G_MODULE_EXPORT void
maptile_module_load (MaptileModule *module)
{
  maptile_mapper_repo_get_type (G_TYPE_MODULE (module));
}

G_MODULE_EXPORT void
maptile_module_unload (MaptileModule *module)
{
}

static GType
maptile_mapper_repo_get_type (GTypeModule *module)
{
  if (!maptile_mapper_repo_type)
    {
      static const GTypeInfo repo_info =
      {
        sizeof (MaptileMapperRepoClass),
        (GBaseInitFunc) NULL,
        (GBaseFinalizeFunc) NULL,
        (GClassInitFunc) maptile_mapper_repo_class_init,
        NULL,           /* class_finalize */
        NULL,           /* class_data     */
        sizeof (MaptileMapperRepo),
        0,              /* n_preallocs    */
        (GInstanceInitFunc) maptile_mapper_repo_init
      };

      maptile_mapper_repo_type =
        g_type_module_register_type (module, MAPTILE_TYPE_REPO,
                                     "MaptileMapperRepo", &repo_info, 0);
    }

  return maptile_mapper_repo_type;
}

static void
maptile_mapper_repo_class_init (MaptileMapperRepoClass *klass)
{
  MaptileRepoClass *repo_class = MAPTILE_REPO_CLASS (klass);

  maptile_mapper_repo_parent_class = g_type_class_peek_parent (klass);

  repo_class->name   = "MapperMaps";

  repo_class->get_tile = maptile_mapper_repo_get_tile;
}

static void
maptile_mapper_repo_init (MaptileMapperRepo *instance)
{
    instance->easy_handle = curl_easy_init();
    curl_easy_setopt(instance->easy_handle, CURLOPT_NOPROGRESS, 1);
    curl_easy_setopt(instance->easy_handle, CURLOPT_FOLLOWLOCATION, 1);
    curl_easy_setopt(instance->easy_handle, CURLOPT_FAILONERROR, 1);
    curl_easy_setopt(instance->easy_handle, CURLOPT_USERAGENT,
            "Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.8.0.4) Gecko/20060701 Firefox/1.5.0.4");
    curl_easy_setopt(instance->easy_handle, CURLOPT_TIMEOUT, 30);
    curl_easy_setopt(instance->easy_handle, CURLOPT_CONNECTTIMEOUT, 10);
}

static void
maptile_mapper_repo_get_tile (MaptileRepo     *repo,
                              guint           tilex,
                              guint           tiley,
                              guint           zoom,
                              gchar           **tile,
			      ConICCallback   check_conic_cb)
{
    GValue value1 = {0,};
    const gchar *data_uri;
    gchar *tile_uri = NULL;
    const gchar *cache_uri;
    FILE *f;

    MaptileMapperRepo *mapper_repo = MAPTILE_MAPPER_REPO(repo);

    /* Check first for local cache file */
    g_value_init(&value1, G_TYPE_STRING);
    g_object_get_property(G_OBJECT(repo), "map-cache-uri", &value1);
    cache_uri = g_value_get_string(&value1);

    *tile = g_strdup_printf("%s/%u/%u/%u.jpg", cache_uri, zoom, tilex, tiley);
    if ( !g_file_test(*tile, G_FILE_TEST_EXISTS) )
    {
	g_debug("Local file %s does not exist", *tile);
        /* If local cache file does not exist then we try download it from the net */
        /* First ensure that the directory exists, if not then create it */
        if(!(f = g_fopen(*tile, "w")) && errno == ENOENT)
        {
            if ( !gnome_vfs_initialized() )
                gnome_vfs_init();
            gchar buffer[BUFFER_SIZE];
            snprintf(buffer, sizeof(buffer), "%s/%u",
                    cache_uri, zoom);
            gnome_vfs_make_directory(buffer, 0775);
            snprintf(buffer, sizeof(buffer), "%s/%u/%u",
                    cache_uri, zoom, tilex);
            gnome_vfs_make_directory(buffer, 0775);
            f = g_fopen(*tile, "w");
        }
        if ( f )
        {
	    /* Check connection status */
	    check_conic_cb();
            GValue value2 = {0,};
            g_value_init(&value2, G_TYPE_STRING);
            g_object_get_property(G_OBJECT(repo), "map-data-uri", &value2);
            data_uri = g_value_get_string(&value2);

            tile_uri = maptile_mapper_construct_url(data_uri, tilex, tiley, zoom);

	    gchar *error_buffer = g_malloc(CURL_ERROR_SIZE);
            curl_easy_setopt(mapper_repo->easy_handle, CURLOPT_URL, tile_uri);
            curl_easy_setopt(mapper_repo->easy_handle, CURLOPT_WRITEDATA, f);
	    curl_easy_setopt(mapper_repo->easy_handle, CURLOPT_ERRORBUFFER, error_buffer);
            if ( curl_easy_perform(mapper_repo->easy_handle) )
	    {
                g_warning("Downloading tile %s failed: %s!", tile_uri,
				error_buffer);
	    }
	    g_free(error_buffer);

            fclose(f);
        }
        else
            g_warning("Downloading failed, unable to open file %s for writing!", *tile);
    }
    if ( tile_uri )
        g_free(tile_uri);
}

/* From maemo-mapper, construct Mapper Satellite coordinate string*/
static void
maptile_mapper_convert_coords_to_quadtree_string(gint x, gint y, gint zoomlevel,
                                      gchar *buffer, const gchar initial,
                                      const gchar *const quadrant)
{
    gchar *ptr = buffer;
    gint n;

    if (initial)
        *ptr++ = initial;

    for(n = 16 - zoomlevel; n >= 0; n--)
    {
        gint xbit = (x >> n) & 1;
        gint ybit = (y >> n) & 1;
        *ptr++ = quadrant[xbit + 2 * ybit];
    }
    *ptr++ = '\0';
}

/* From maemo-mapper, construct the download url */
static gchar*
maptile_mapper_construct_url(gchar *uri, guint tilex, guint tiley, guint zoom)
{
    if(strstr(uri, "%s"))
    {
        /* This is a qrts-based quadtree URI. */
        gchar location[MAX_ZOOM + 2];
        maptile_mapper_convert_coords_to_quadtree_string(tilex, tiley, zoom, location,
                                              't', "qrts");
        return g_strdup_printf(uri, location);
    }
    else if(strstr(uri, "%0s"))
    {
        /* This is a zero-based quadtree URI. */
	gchar location[MAX_ZOOM + 2];
        maptile_mapper_convert_coords_to_quadtree_string(tilex, tiley, zoom, location,
                                              '\0', "0123");
        return g_strdup_printf(uri, location); 

    }    
    else if(strstr(uri, "%0d"))
    {
        /* This is a inverted x/y/z URI. */
	return g_strdup_printf(uri, 17 - zoom, tilex, tiley);
    }

    else if(strstr(uri, "SERVICE=WMS"))
    {
        /* This is a wms-map URI. */
        return maptile_mapper_convert_wms_to_wms(tilex, tiley, zoom, uri);
    }
    else
    {
        /* This is a standard x/y/z URI. */
        return g_strdup_printf(uri, tilex, tiley, zoom);
    }
}

 /* Given a wms uri pattern, compute the coordinate transformation and
 * trimming.
 * 'proj' is used for the conversion
 */
static gchar*
maptile_mapper_convert_wms_to_wms(gint tilex, gint tiley, gint zoomlevel, gchar* uri)
{
    gchar cmd[BUFFER_SIZE], srs[BUFFER_SIZE];
    gchar *ret = NULL;
    FILE* in;
    gfloat lon1, lat1, lon2, lat2;

    gchar *widthstr   = strcasestr(uri,"WIDTH=");
    gchar *heightstr  = strcasestr(uri,"HEIGHT=");
    gchar *srsstr     = strcasestr(uri,"SRS=EPSG");
    gchar *srsstre    = strchr(srsstr,'&');

    /* missing: test if found */
    strcpy(srs,"epsg");
    strncpy(srs+4,srsstr+8,256);
    /* missing: test srsstre-srsstr < 526 */
    srs[srsstre-srsstr-4] = 0;
    /* convert to lower, as WMC is EPSG and cs2cs is epsg */

    gint dwidth  = widthstr ? atoi(widthstr+6) - TILE_SIZE_PIXELS : 0;
    gint dheight = heightstr ? atoi(heightstr+7) - TILE_SIZE_PIXELS : 0;

    unit2latlon(tile2zunit(tilex,zoomlevel)
            - pixel2zunit(dwidth/2,zoomlevel),
            tile2zunit(tiley+1,zoomlevel)
            + pixel2zunit((dheight+1)/2,zoomlevel),
            lat1, lon1);

    unit2latlon(tile2zunit(tilex+1,zoomlevel)
            + pixel2zunit((dwidth+1)/2,zoomlevel),
            tile2zunit(tiley,zoomlevel)
            - pixel2zunit(dheight/2,zoomlevel),
            lat2, lon2);

    setlocale(LC_NUMERIC, "C");

    snprintf(cmd, sizeof(cmd),
            "(echo \"%.6f %.6f\"; echo \"%.6f %.6f\") | "
            "/usr/bin/cs2cs +proj=longlat +datum=WGS84 +to +init=%s -f %%.6f "
            " > /tmp/tmpcs2cs ",
             lon1, lat1, lon2, lat2, srs);
    system(cmd);

    if(!(in = g_fopen("/tmp/tmpcs2cs","r")))
        g_printerr("Cannot open results of conversion\n");
    else if(5 != fscanf(in,"%f %f %s %f %f", &lon1, &lat1, cmd, &lon2, &lat2))
    {
        g_printerr("Wrong conversion\n");
        fclose(in);
    }
    else
    {
        fclose(in);
        ret = g_strdup_printf(uri, lon1, lat1, lon2, lat2);
    }

    setlocale(LC_NUMERIC, "");

    return ret;
}

/* EOF */
