/*
 * This file is part of maptile-loader
 * Copyright (C) 2007  Pekka Rönkkö (pronkko@gmail.com)
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include <config.h>

#include <gmodule.h>

#include "maptile-module.h"


enum
{
  PROP_0,
  PROP_FILENAME
};


static void      maptile_module_finalize      (GObject      *object);
static void      maptile_module_get_property  (GObject      *object,
                                           guint         param_id,
                                           GValue       *value,
                                           GParamSpec   *pspec);
static void      maptile_module_set_property  (GObject      *object,
                                           guint         param_id,
                                           const GValue *value,
                                           GParamSpec   *pspec);
static gboolean  maptile_module_load_module   (GTypeModule  *gmodule);
static void      maptile_module_unload_module (GTypeModule  *gmodule);


G_DEFINE_TYPE (MaptileModule, maptile_module, G_TYPE_TYPE_MODULE);


static void
maptile_module_class_init (MaptileModuleClass *class)
{
  GObjectClass     *object_class      = G_OBJECT_CLASS (class);
  GTypeModuleClass *type_module_class = G_TYPE_MODULE_CLASS (class);

  object_class->finalize     = maptile_module_finalize;
  object_class->get_property = maptile_module_get_property;
  object_class->set_property = maptile_module_set_property;

  type_module_class->load    = maptile_module_load_module;
  type_module_class->unload  = maptile_module_unload_module;

  g_object_class_install_property (object_class, PROP_FILENAME,
                                   g_param_spec_string ("filename",
                                                        "Filename",
                                                        "The filaname of the module",
                                                        NULL,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_CONSTRUCT_ONLY));
}

static void
maptile_module_init (MaptileModule *module)
{
  module->filename = NULL;
  module->library  = NULL;
  module->load     = NULL;
  module->unload   = NULL;
}

static void
maptile_module_finalize (GObject *object)
{
  MaptileModule *module = MAPTILE_MODULE (object);

  g_free (module->filename);

  G_OBJECT_CLASS (maptile_module_parent_class)->finalize (object);
}

static void
maptile_module_get_property (GObject    *object,
                         guint       param_id,
                         GValue     *value,
                         GParamSpec *pspec)
{
  MaptileModule *module = MAPTILE_MODULE (object);

  switch (param_id)
    {
    case PROP_FILENAME:
      g_value_set_string (value, module->filename);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
      break;
    }
}

static void
maptile_module_set_property (GObject      *object,
                         guint         param_id,
                         const GValue *value,
                         GParamSpec   *pspec)
{
  MaptileModule *module = MAPTILE_MODULE (object);

  switch (param_id)
    {
    case PROP_FILENAME:
      g_free (module->filename);
      module->filename = g_value_dup_string (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
      break;
    }
}

static gboolean
maptile_module_load_module (GTypeModule *gmodule)
{
  MaptileModule *module = MAPTILE_MODULE (gmodule);

  if (!module->filename)
    {
      g_warning ("Module path not set");
      return FALSE;
    }

  module->library = g_module_open (module->filename, 0);

  if (!module->library)
    {
      g_printerr ("%s\n", g_module_error ());
      return FALSE;
    }

  /* Make sure that the loaded library contains the required methods */
  if (! g_module_symbol (module->library,
                         "maptile_module_load",
                         (gpointer *) &module->load) ||
      ! g_module_symbol (module->library,
                         "maptile_module_unload",
                         (gpointer *) &module->unload))
    {
      g_printerr ("%s\n", g_module_error ());
      g_module_close (module->library);

      return FALSE;
    }

  /* Initialize the loaded module */
  module->load (module);

  return TRUE;
}

static void
maptile_module_unload_module (GTypeModule *gmodule)
{
  MaptileModule *module = MAPTILE_MODULE (gmodule);

  module->unload (module);

  g_module_close (module->library);
  module->library = NULL;

  module->load   = NULL;
  module->unload = NULL;
}

MaptileModule *
maptile_module_new (const gchar *filename)
{
  MaptileModule *module;

  g_return_val_if_fail (filename != NULL, NULL);

  module = g_object_new (MAPTILE_TYPE_MODULE,
                         "filename", filename,
                         NULL);

  return module;
}
